package fr.ifremer.adagio.core.dao.data.vessel.feature.use;

/*
 * #%L
 * Tutti :: Persistence API
 * $Id: GearUseFeaturesDaoImpl.java 607 2013-04-21 12:34:44Z tc1fbb1 $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/trunk/adagio/core-allegro/src/main/java/fr/ifremer/adagio/core/dao/data/vessel/feature/physical/GearUseFeaturesDaoImpl.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Collection;
import java.util.Date;

import org.apache.commons.collections4.CollectionUtils;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.dao.DataRetrievalFailureException;
import org.springframework.stereotype.Repository;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.core.dao.administration.user.Department;
import fr.ifremer.adagio.core.dao.administration.user.DepartmentImpl;
import fr.ifremer.adagio.core.dao.data.fishingArea.FishingArea;
import fr.ifremer.adagio.core.dao.data.measure.GearUseMeasurement;
import fr.ifremer.adagio.core.dao.data.operation.Operation;
import fr.ifremer.adagio.core.dao.data.survey.activity.DailyActivityCalendar;
import fr.ifremer.adagio.core.dao.referential.QualityFlagCode;
import fr.ifremer.adagio.core.dao.referential.QualityFlagImpl;
import fr.ifremer.adagio.core.dao.referential.location.LocationImpl;
import fr.ifremer.adagio.core.dao.referential.pmfm.Pmfm;
import fr.ifremer.adagio.core.dao.referential.pmfm.PmfmImpl;
import fr.ifremer.adagio.core.dao.referential.pmfm.QualitativeValue;
import fr.ifremer.adagio.core.dao.referential.pmfm.QualitativeValueImpl;

@Repository("gearUseFeaturesDao")
@Lazy
public class GearUseFeaturesDaoImpl
		extends GearUseFeaturesDaoBase
		implements GearUseFeaturesExtendDao {

	/**
	 * Constructor used by Spring
	 */
	@Autowired
	public GearUseFeaturesDaoImpl(SessionFactory sessionFactory) {
		super();
		setSessionFactory(sessionFactory);
	}

	@Override
	public GearUseFeatures getGearUseFeatures(Operation operation,
			boolean createIfNotExists) {

		if (CollectionUtils.isNotEmpty(operation.getGearUseFeatures())) {
			GearUseFeatures guf = CollectionUtils.extractSingleton(operation.getGearUseFeatures());
			return guf;
		}

		if (!createIfNotExists) {
			return null;
		}

		// Create a new entity
		GearUseFeatures guf = GearUseFeatures.Factory.newInstance();
		operation.setGearUseFeatures(Sets.newHashSet(guf));
		guf.setOperation(operation);

		// Set default attributes (copy from operation)
		guf.setStartDate(operation.getStartDateTime());
		guf.setEndDate(operation.getEndDateTime());
		guf.setCreationDate(new Date());
		guf.setQualityFlag(load(QualityFlagImpl.class, QualityFlagCode.NOTQUALIFIED.getValue()));
		guf.setRankOrder((short) 1);

		return guf;
	}

	@Override
	public GearUseMeasurement getGearUseMeasurement(GearUseFeatures gearUseFeatures, Integer pmfmId) {
		return getGearUseMeasurement(gearUseFeatures, pmfmId, false);
	}

	protected GearUseMeasurement getGearUseMeasurement(GearUseFeatures gearUseFeatures,
			Integer pmfmId, boolean createIfNotExists) {
		GearUseMeasurement gearUseMeasurement = null;
		if (gearUseFeatures.getGearUseMeasurements() != null) {
			for (GearUseMeasurement vum : gearUseFeatures.getGearUseMeasurements()) {
				if (pmfmId.equals(vum.getPmfm().getId())) {
					gearUseMeasurement = vum;
					break;
				}
			}
		}
		if (gearUseMeasurement != null || !createIfNotExists) {
			return gearUseMeasurement;
		}

		gearUseMeasurement = GearUseMeasurement.Factory.newInstance();
		gearUseMeasurement.setGearUseFeatures(gearUseFeatures);
		if (gearUseFeatures.getGearUseMeasurements() == null) {
			gearUseFeatures.setGearUseMeasurements(Sets.newHashSet(gearUseMeasurement));
		} else {
			gearUseFeatures.getGearUseMeasurements().add(gearUseMeasurement);
		}
		gearUseMeasurement.setQualityFlag(load(QualityFlagImpl.class, QualityFlagCode.NOTQUALIFIED.getValue()));
		Pmfm pmfm = load(PmfmImpl.class, pmfmId);
		gearUseMeasurement.setPmfm(pmfm);

		return gearUseMeasurement;
	}

	@Override
	public void removeGearUseMeasurement(GearUseFeatures gearUseFeatures,
			Integer pmfmId) {
		GearUseMeasurement gearUseMeasurement = getGearUseMeasurement(gearUseFeatures, pmfmId, false);
		if (gearUseMeasurement == null) {
			return;
		}
		gearUseFeatures.getGearUseMeasurements().remove(gearUseMeasurement);
	}

	@Override
	public GearUseMeasurement setGearUseMeasurement(GearUseFeatures gearUseFeatures,
			Integer pmfmId,
			Float numericalValue,
			String alphanumericalValue,
			Integer qualitativevalueId,
			Department recorderDepartment) {
		GearUseMeasurement gearUseMeasurement = getGearUseMeasurement(gearUseFeatures, pmfmId, true);

		if (alphanumericalValue != null) {
			gearUseMeasurement.setAlphanumericalValue(alphanumericalValue);
		} else if (numericalValue != null) {
			gearUseMeasurement.setNumericalValue(numericalValue);
		} else if (qualitativevalueId != null) {
			QualitativeValue qv = load(QualitativeValueImpl.class, qualitativevalueId);
			gearUseMeasurement.setQualitativeValue(qv);
		}

		if (recorderDepartment != null) {
			gearUseMeasurement.setDepartment(recorderDepartment);
		}

		return gearUseMeasurement;
	}

	@Override
	public GearUseMeasurement setGearUseMeasurement(GearUseFeatures gearUseFeatures,
			Integer pmfmId,
			Float numericalValue,
			String alphanumericalValue,
			Integer qualitativevalueId,
			Integer recorderDepartmentId) {
		GearUseMeasurement gearUseMeasurement = getGearUseMeasurement(gearUseFeatures, pmfmId, true);

		if (alphanumericalValue != null) {
			gearUseMeasurement.setAlphanumericalValue(alphanumericalValue);
		} else if (numericalValue != null) {
			gearUseMeasurement.setNumericalValue(numericalValue);
		} else if (qualitativevalueId != null) {
			QualitativeValue qv = load(QualitativeValueImpl.class, qualitativevalueId);
			gearUseMeasurement.setQualitativeValue(qv);
		}

		if (recorderDepartmentId != null) {
			gearUseMeasurement.setDepartment(load(DepartmentImpl.class, recorderDepartmentId));
		}

		return gearUseMeasurement;
	}

	@Override
	public FishingArea getFishingAreaAsUnique(GearUseFeatures guf, boolean createIfNotExists) {
		return getFishingArea(guf, null, createIfNotExists);
	}

	@Override
	public FishingArea getFishingArea(GearUseFeatures guf, int locationId, boolean createIfNotExists) {
		return getFishingArea(guf, locationId, createIfNotExists);
	}

	/* -- Internal methods-- */

	private FishingArea getFishingArea(GearUseFeatures guf, Integer locationId, boolean createIfNotExists) {
		if (CollectionUtils.isNotEmpty(guf.getFishingAreas())) {

			// If no locationId specified, must have only one fishing area
			if (locationId == null) {
				if (guf.getFishingAreas().size() > 1) {
					throw new DataRetrievalFailureException("More than one fishing area found a gearUseFeatures");
				}
				return guf.getFishingAreas().iterator().next();
			}

			// If locationId has been specified, search into fishing area
			for (FishingArea fishingArea : guf.getFishingAreas()) {
				if (fishingArea.getLocation() != null
						&& locationId.equals(fishingArea.getLocation().getId())) {
					return fishingArea;
				}
			}
		}

		// Not found: exit if creation is not asked
		if (!createIfNotExists) {
			return null;
		}

		// Create a new entity
		FishingArea fishingArea = FishingArea.Factory.newInstance();
		if (locationId != null) {
			fishingArea.setLocation(load(LocationImpl.class, new Integer(locationId)));
		}

		// Link to the parent gearUseFeatures
		fishingArea.setGearUseFeatures(guf);
		if (guf.getFishingAreas() != null) {
			guf.getFishingAreas().add(fishingArea);
		} else {
			guf.setFishingAreas(Sets.newHashSet(fishingArea));
		}

		return fishingArea;
	}

	@Override
	public void remove(Collection<GearUseFeatures> entities) {
		if (entities == null) {
			throw new IllegalArgumentException(
					"GearUseFeatures.remove - 'entities' can not be null");
		}
		for (GearUseFeatures entity : entities) {
			remove(entity);
		}
	}

	@Override
	public void remove(GearUseFeatures entity) {
		if (CollectionUtils.isNotEmpty(entity.getGearUseMeasurements())) {
			entity.getGearUseMeasurements().clear();
		}
		super.remove(entity);
	}
}
