package fr.ifremer.adagio.core.dao.data.batch;

/*
 * #%L
 * SIH-Adagio Core for Allegro
 * $Id: CatchBatchExtendDao.java 12102 2014-05-28 21:04:57Z bl05b3e $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/tags/adagio-3.8.2/core-allegro/src/main/java/fr/ifremer/adagio/core/dao/data/batch/CatchBatchExtendDao.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Collection;
import java.util.List;

import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;

import fr.ifremer.adagio.core.dao.data.batch.validator.CatchBatchValidationError;
import fr.ifremer.adagio.core.dao.data.batch.validator.CatchBatchValidationException;
import fr.ifremer.adagio.core.dao.data.batch.validator.CatchBatchValidator;
import fr.ifremer.adagio.core.dao.data.measure.QuantificationMeasurement;
import fr.ifremer.adagio.core.dao.data.measure.SortingMeasurement;
import org.springframework.dao.DataRetrievalFailureException;

public interface CatchBatchExtendDao extends CatchBatchDao {

	/**
	 * Property name used to walk in a sorting batch identified by a pmfm.
	 * 
	 * @see #getSortingBatch(Collection, Object...)
	 * @since 3.3.6
	 */
	String PMFM_ID = "pmfmId";

	/**
	 * Property name used to walk in a sorting batch identified by a reference
	 * taxon.
	 * 
	 * @see #getSortingBatch(Collection, Object...)
	 * @since 3.3.6
	 */
	String REFERENCE_TAXON_ID = "referenceTaxonId";

	void registerCatchBatchValidator(CatchBatchValidator validator);

	void unregisterCatchBatchValidator(CatchBatchValidator validator);

	/**
	 * Load a catch batch and all its childrens. (No validation)
	 * 
	 * @param catchBatchId
	 *            id of the catch batch to load
	 * @return the catch batch with all children, or null if not found
	 */
	CatchBatch loadFullTree(Integer catchBatchId, Integer pmfmId);

	/**
	 * Load a catch batch and all its childrens.
	 * 
	 * @param catchBatchId
	 *            id of the catch batch to load
	 * @param validate
	 *            if true, method validate() is typically called
	 * @param quickFix
	 *            if true, will try to quick fix validation error
	 * @return the catch batch with all children
	 * @throws CatchBatchValidationException
	 *             only if validate = true, and error found (with gravity ERROR or FATAL)
	 */
	CatchBatch loadFullTree(Integer catchBatchId, Integer pmfmId, boolean validate, boolean quickFix) throws CatchBatchValidationException;

	/**
	 * Method typically using by other loadFullTree() methods.
	 * Please do not call directly, but throw other methods.
	 * 
	 * @param catchBatchId
	 *            id of the catch batch to load
	 * @param useCache
	 *            must the be enable ?
	 * @param forceReload
	 *            is cache is enable, force a eviction to refresh the cache
	 * @return A full batch tree
	 */
	@Cacheable(value = "fr.ifremer.adagio.core.dao.data.batch.CatchBatchCache", key = "#catchBatchId", condition = "#useCache == true")
	@CacheEvict(value = "fr.ifremer.adagio.core.dao.data.batch.CatchBatchCache", beforeInvocation = true, key = "#catchBatchId", condition = "#useCache==true and #forceReload==true")
	CatchBatch loadFullTreeWithCache(Integer catchBatchId, Integer pmfmId, boolean useCache, boolean forceReload);

	/**
	 * Validate a catch batch, using all registered validators that are enable for this catchBatch
	 * 
	 * @param catchBatch
	 *            batch to validate
	 * @return null if validation has no errors, or a list of errors found.
	 */
	List<CatchBatchValidationError> validate(CatchBatch catchBatch);

	/**
	 * retrieve a sorting batch, from path give in params. i.e:<br/>
	 * getSortingBatch(sortingBatchCollection,<br/>
	 * PMFM_ID, PmfmId.SEX, QualitativeValueId.FEMALE)
	 * 
	 * @param batchs
	 *            incoming batchs to filter
	 * @param params
	 * @return
	 */
	SortingBatch getSortingBatch(Collection<Batch> batchs, Object... params);

	/**
	 * Retrieve a child (in all tree depth) of the given catch batch, using an sorting batch id.
	 * 
	 * @param catchBatch
	 * @param sortingBatchId
	 *            id of the sorting batch to retrieve
	 * @return the sorting batch found
	 * @throws DataRetrievalFailureException
	 *             if the sorting batch could not be found
	 */
	SortingBatch getSortingBatchById(CatchBatch catchBatch, Integer sortingBatchId);

	/**
	 * @param fishingOperationId
	 *            id of the fishing operation
	 * @return {@code true} if there is a catchBatch for the given fishing
	 *         operation, {@code false} otherwise.
	 * @since 3.3.6
	 */
	boolean isCatchBatchExistsForFishingOperation(Integer fishingOperationId);

	/**
	 * Retrieve a catch batch Id, from a fishing operation id
	 * 
	 * @param fishingOperationId
	 *            id of the fishing operation
	 * @return the fid of the catch batch associated with the fishing operation
	 * @throws DataRetrievalFailureException
	 *             if catch batch id not found
	 */
	Integer getIdByFishingOperationId(Integer fishingOperationId) throws DataRetrievalFailureException;

	/**
	 * Retrieve a catch batch Id, from a sorting batch id
	 * 
	 * @param sortingBatchId
	 * @return
	 * @throws DataRetrievalFailureException
	 *             if catch batch id not found
	 */
	Integer getIdBySortingBatchId(Integer sortingBatchId) throws DataRetrievalFailureException;

	/**
	 * Remove a batch, with all children (could be CatchBatch or a SortingBatch)
	 * -> Method optimize for cache using in loadFullTree()
	 * 
	 * @param batchId
	 *            id of the batch to remove
	 */
	@CacheEvict(value = "fr.ifremer.adagio.core.dao.data.batch.CatchBatchCache", key = "#parentCatchBatch.id")
	void removeWithChildren(Integer batchId, CatchBatch parentCatchBatch);

	/**
	 * Remove a batch, with all children (could be CatchBatch or a SortingBatch)
	 * -> Method NOT optimize for cache using in loadFullTree : will evict all the cache
	 * 
	 * @param batchId
	 *            id of the batch to remove
	 */
	@CacheEvict(value = "fr.ifremer.adagio.core.dao.data.batch.CatchBatchCache", allEntries = true)
	void removeWithChildren(Integer batchId);

	/**
	 * Remove a batch and evict all the cache
	 * 
	 * @param catchBatchId
	 *            id of the catch batch to remove
	 */
	@CacheEvict(value = "fr.ifremer.adagio.core.dao.data.batch.CatchBatchCache", allEntries = true)
	void remove(Integer catchBatchId);

	/**
	 * Walk throw the batch given by the id and grab ids of all his shell.
	 * 
	 * <strong>Note:</strong> Will use a postorder traversal (child before
	 * parent) to keep ids.
	 * 
	 * @param batchId
	 *            id of the batch to scan
	 * @return list of all children ids in postorder traversal order
	 * @since 3.3.6
	 */
	List<Integer> getAllChildrenIds(Integer batchId);

	SortingMeasurement getSortingMeasurement(SortingBatch sortingBatch, Integer pmfmId, Integer recorderDepartmentId, boolean createIfNotExists);

	QuantificationMeasurement setQuantificationMeasurement(Batch batch, Integer pmfmId, Integer recorderDepartmentId, Float weightValue,
			boolean isReferenceQuantitification);

	QuantificationMeasurement getQuantificationMeasurement(Batch batch, Integer pmfmId);

	SortingMeasurement getInheritedSortingMeasurement(SortingBatch sortingBatch, Integer pmfmId);

	/**
	 * Update the {@link SortingBatch#getSamplingRatio()} and {@link SortingBatch#getSamplingRatioText()}.
	 * 
	 * @param sortingBatch
	 *            the batch to update
	 * @param weight
	 *            weight of the observed batch
	 * @param weightBeforeSampling
	 *            Weight before sampling
	 */
	void setSortingSamplingRatio(SortingBatch sortingBatch,
			Float weight,
			Float weightBeforeSampling);

	/**
	 * Change weights of a batch. This will update :
	 * <ul>
	 * <li>QuantificationMeasurement (define with isReferenceQuantification=true)
	 * <li>Batch.samplingRatio and Batch.samplingRatioText
	 * <li>Some technical fields: Batch.weight, Batch.weightBeforeSampling
	 * </ul>
	 * 
	 * @param sortingBatch
	 *            the batch to update
	 * @param weight
	 *            weight of the observed batch
	 * @param weightBeforeSampling
	 *            Weight before sampling
	 * @param weightPmfmId
	 *            PMFM Id to use in QuantificationMeasurement
	 * @param recorderDepartmentId
	 *            Id of department to use in QuantificationMeasurement
	 */
	public void setSortingBatchWeights(SortingBatch sortingBatch,
			Float weight,
			Float weightBeforeSampling,
			Integer weightPmfmId,
			Integer recorderDepartmentId);

	/**
	 * Change the species of a batch.
	 * -> Method NOT optimize for cache using in loadFullTree : will evict all the cache
	 * 
	 * @param batchId
	 * @param referenceTaxonId
	 */
	@CacheEvict(value = "fr.ifremer.adagio.core.dao.data.batch.CatchBatchCache", allEntries = true)
	void setSortingBatchReferenceTaxon(String batchId, Integer referenceTaxonId);

	@CacheEvict(value = "fr.ifremer.adagio.core.dao.data.batch.CatchBatchCache", key = "#catchBatch.id")
	void update(CatchBatch catchBatch);

	/**
	 * Create a new batch
	 * 
	 * @param sortingBatch
	 * @param parentCatchBatch
	 */
	@CacheEvict(value = "fr.ifremer.adagio.core.dao.data.batch.CatchBatchCache", key = "#parentCatchBatch.id")
	SortingBatch createSortingBatch(SortingBatch sortingBatch, CatchBatch parentCatchBatch);

	/**
	 * Load a batch.
	 * If the given catchBatch has been load using loadFullTree, this
	 * method will use cache stored inside the catchBatch
	 * 
	 * @param sortingBatchId
	 * @param parentCatchBatch
	 * @return
	 */
	SortingBatch loadSortingBatch(Integer sortingBatchId, CatchBatch parentCatchBatch);

	@CacheEvict(value = "fr.ifremer.adagio.core.dao.data.batch.CatchBatchCache", key = "#parentCatchBatch.id")
	void updateSortingBatch(SortingBatch sortingBatch, CatchBatch parentCatchBatch);

	@CacheEvict(value = "fr.ifremer.adagio.core.dao.data.batch.CatchBatchCache", key = "#parentCatchBatch.id")
	void updateSortingBatch(List<SortingBatch> sortingBatchs, CatchBatch parentCatchBatch);

}
