package fr.ifremer.adagio.core.action;

/*
 * #%L
 * SIH-Adagio :: Shared
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.sql.SQLException;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.springframework.dao.DataRetrievalFailureException;

import fr.ifremer.adagio.core.dao.administration.user.PersonDaoImpl;
import fr.ifremer.adagio.core.dao.administration.user.PersonExtendDao;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;
import fr.ifremer.adagio.core.service.data.synchro.DataSynchroDirection;
import fr.ifremer.adagio.core.service.data.synchro.DataSynchroService;
import fr.ifremer.adagio.core.service.data.synchro.DataSynchroServiceImpl;
import fr.ifremer.adagio.synchro.SynchroTechnicalException;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.service.SynchroContext;
import fr.ifremer.adagio.synchro.service.SynchroResult;

public class AllegroCoreImportDataAction {
	/* Logger */
	private static final Log log = LogFactory.getLog(AllegroCoreImportDataAction.class);

	public void run() {
		SynchroConfiguration config = SynchroConfiguration.getInstance();

		// Get connections properties :
		Properties sourceConnectionProperties = config.getImportConnectionProperties();
		Properties targetConnectionProperties = config.getConnectionProperties();

		// Log target connection
		if (log.isInfoEnabled()) {
			log.info("Starting database referential importation...");
		}

		// Check connections
		if (!checkConnection(config, sourceConnectionProperties, targetConnectionProperties)) {
			return;
		}

		// Create the result
		SynchroResult result = new SynchroResult();

		// Create the service
		DataSynchroService service = new DataSynchroServiceImpl();

		// Create the context
		Integer userId = getUserId(sourceConnectionProperties);
		SynchroContext context = service.createSynchroContext(sourceConnectionProperties,
				DataSynchroDirection.IMPORT_SERVER2TEMP,
				userId
				);

		// Prepare synchro
		service.prepare(context);

		if (!result.isSuccess()) {
			throw new SynchroTechnicalException(I18n.t("adagio.synchro.prepare.error"),
					result.getError());
		}

		// Run the synchro
		service.synchronize(context);

		if (!result.isSuccess()) {
			throw new SynchroTechnicalException(I18n.t("adagio.synchro.synchro.error"),
					result.getError());
		}

		// Shutdown database at end
		try {
			DaoUtils.shutdownDatabase(targetConnectionProperties);
		} catch (SQLException e) {
			throw new SynchroTechnicalException(I18n.t("adagio.synchro.shutdown.error"),
					result.getError());
		}
	}

	protected Integer getUserId(Properties connectionProperties) {
		Integer userId = null;

		while (userId == null) {
			// Get the username
			String username = CommandLines.readNotBlankInput(I18n.t("adagio.synchro.getUser.username"));

			// Find the user, by username
			PersonExtendDao personDao = new PersonDaoImpl();
			try {
				userId = personDao.getUserIdByUsername(connectionProperties, username);
			} catch (DataRetrievalFailureException e) {
				throw new SynchroTechnicalException(I18n.t("adagio.synchro.getUser.error"),
						e);
			}

			// Display a warning message before new loop iteration
			if (userId == null) {
				log.error(I18n.t("adagio.synchro.getUser.notFound", username));
			}
		}
		return userId;
	}

	protected boolean checkConnection(
			SynchroConfiguration config,
			Properties sourceConnectionProperties,
			Properties targetConnectionProperties) {

		// Log target connection
		if (log.isInfoEnabled()) {
			log.info("Connecting to target database...");
			log.info(String.format(" Database directory: %s", config.getDbDirectory()));
			log.info(String.format(" JDBC Driver: %s", config.getJdbcDriver()));
			log.info(String.format(" JDBC URL: %s", config.getJdbcURL()));
			log.info(String.format(" JDBC Username: %s", config.getJdbcUsername()));
		}

		// Check target connection
		boolean isValidConnection = DaoUtils.isValidConnectionProperties(targetConnectionProperties);
		if (!isValidConnection) {
			log.error("Connection error: could not connect to target database.");
			return false;
		}

		// Log source connection
		if (log.isInfoEnabled()) {
			log.info("Connecting to source database...");
			log.info(String.format(" JDBC Driver: %s", config.getImportJdbcDriver()));
			log.info(String.format(" JDBC URL: %s", config.getImportJdbcURL()));
			log.info(String.format(" JDBC Username: %s", config.getImportJdbcUsername()));
			String jdbcCatalog = config.getImportJdbcCatalog();
			if (StringUtils.isNotBlank(jdbcCatalog)) {
				log.info(String.format(" JDBC Catalog: %s", jdbcCatalog));
			}
			String jdbcSchema = config.getImportJdbcSchema();
			if (StringUtils.isNotBlank(jdbcSchema)) {
				log.info(String.format(" JDBC Schema: %s", jdbcSchema));
			}
		}

		// Check source connection
		String sourceJdbcUrl = DaoUtils.getUrl(sourceConnectionProperties);
		if (!sourceJdbcUrl.startsWith("jdbc:oracle:")) {
			// Source database = an oracle connection (see file synchro-test-write.properties)
			log.warn("Source database is not a Oracle database. Make sure your configuration file is correct.");
		}
		isValidConnection = DaoUtils.isValidConnectionProperties(sourceConnectionProperties);
		if (!isValidConnection) {
			log.warn("Connection error: could not connect to source database.");
			return false;
		}

		return true;
	}
}
