package fr.ifremer.adagio.core;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Locale;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.config.ApplicationConfig;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.i18n.init.UserI18nInitializer;

import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.service.ServiceLocator;
import fr.ifremer.adagio.synchro.config.SynchroConfiguration;
import fr.ifremer.adagio.synchro.config.SynchroConfigurationOption;

public class RunAllegroCore {

	/* Logger */
	private static final Log log = LogFactory.getLog(RunAllegroCore.class);

	public static final int DELETE_DB_EXIT_CODE = 89;

	public static final int UPATE_EXIT_CODE = 88;

	public static final int NORMAL_EXIT_CODE = 0;

	public static void main(String[] args) {
		if (log.isInfoEnabled()) {
			log.info("Starting SIH-Adagio :: Core for Allegro with arguments: " + Arrays.toString(args));
		}

		// By default, display help
		if (args == null || args.length == 0) {
			args = new String[] { "-h" };
		}

		// Could override config file name (useful for dev)
		String configFile = "allegro-core.config";
		if (System.getProperty(configFile) != null) {
			configFile = System.getProperty(configFile);
			configFile = configFile.replaceAll("\\\\", "/");
		}

		// Create configuration
		AdagioConfiguration config = new AdagioConfiguration(configFile, args) {
			protected void addAlias(ApplicationConfig applicationConfig) {
				super.addAlias(applicationConfig);
				// Add custom alias
				applicationConfig.addAlias("-iu", "--option", SynchroConfigurationOption.IMPORT_JDBC_USERNAME.getKey());
				applicationConfig.addAlias("--import-user", "--option", SynchroConfigurationOption.IMPORT_JDBC_USERNAME.getKey());
				applicationConfig.addAlias("-ip", "--option", SynchroConfigurationOption.IMPORT_JDBC_PASSWORD.getKey());
				applicationConfig.addAlias("--import-password", "--option", SynchroConfigurationOption.IMPORT_JDBC_PASSWORD.getKey());
				applicationConfig.addAlias("-idb", "--option", SynchroConfigurationOption.IMPORT_JDBC_URL.getKey());
				applicationConfig.addAlias("--import-database", "--option", SynchroConfigurationOption.IMPORT_JDBC_URL.getKey());
				applicationConfig.addAlias("-is", "--option", SynchroConfigurationOption.IMPORT_JDBC_SCHEMA.getKey());
				applicationConfig.addAlias("--import-schema", "--option", SynchroConfigurationOption.IMPORT_JDBC_SCHEMA.getKey());
			};
		};
		AdagioConfiguration.setInstance(config);

		// Cascade the application config to the synchro module
		SynchroConfiguration synchroConfig = new SynchroConfiguration(config.getApplicationConfig());
		SynchroConfiguration.setInstance(synchroConfig);

		// Init i18n
		try {
			initI18n(config);
		} catch (IOException e) {
			throw new AdagioTechnicalException("i18n initialization failed", e);
		}

		try {
			config.getApplicationConfig().doAllAction();
		} catch (Exception e) {
			e.printStackTrace();
		}

		IOUtils.closeQuietly(ServiceLocator.instance());
	}

	protected static void initI18n(AdagioConfiguration config) throws IOException {

		// --------------------------------------------------------------------//
		// init i18n
		// --------------------------------------------------------------------//
		File i18nDirectory = new File(config.getDataDirectory(), "i18n");
		if (i18nDirectory.exists()) {
			// clean i18n cache
			FileUtils.cleanDirectory(i18nDirectory);
		}

		FileUtils.forceMkdir(i18nDirectory);

		if (log.isDebugEnabled()) {
			log.debug("I18N directory: " + i18nDirectory);
		}

		Locale i18nLocale = config.getI18nLocale();

		if (log.isInfoEnabled()) {
			log.info(String.format("Starts i18n with locale [%s] at [%s]",
					i18nLocale, i18nDirectory));
		}
		I18n.init(new UserI18nInitializer(
				i18nDirectory, new DefaultI18nInitializer(getI18nBundleName())),
				i18nLocale);
	}

	protected static String getI18nBundleName() {
		return "adagio-core-allegro-i18n";
	}
}
