package fr.ifremer.adagio.core.service.technical.synchro.data;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */


import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.Properties;

import org.apache.commons.collections.CollectionUtils;
import org.hibernate.JDBCException;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.cfg.Environment;
import org.hibernate.dialect.Dialect;
import org.hibernate.exception.GenericJDBCException;
import org.hibernate.exception.spi.SQLExceptionConversionDelegate;
import org.hibernate.exception.spi.SQLExceptionConverter;

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;

import fr.ifremer.adagio.core.dao.technical.DaoUtils;

public class DataSynchroUtils {

	public static final List<String> ORACLE_EXCLUDE_TABLE_PATTERNS = Lists.newArrayList(
			"BIN%", // Trash Oracle
			"MDR%" // Table metadata spatial
	);

	protected DataSynchroUtils() {
		// helper class : non instanciation
	}

	/**
	 * Create a new hibernate configuration, with all hbm.xml files
	 * for the schema need for app
	 * 
	 * @return the hibernate Configuration
	 */
	public static Properties getConnectionProperties(String jdbcUrl, String username, String password, String schema, String dialect) {

		// Building a new configuration
		Properties p = new Properties();

		// Set hibernate dialect
		p.setProperty(Environment.DIALECT, dialect);

		// To be able to retrieve connection
		p.setProperty(Environment.URL, jdbcUrl);
		p.setProperty(Environment.USER, username);
		p.setProperty(Environment.PASS, password);

		p.setProperty(Environment.DEFAULT_SCHEMA, schema);

		// Try with synonyms enable
		p.setProperty(AvailableSettings.ENABLE_SYNONYMS, "true");

		// Pour tester avec le metadata generic (normalement plus long pour Oracle)
		// cfg.setProperty("hibernatetool.metadatadialect", "org.hibernate.cfg.rveng.dialect.JDBCMetaDataDialect");
		return p;
	}

	public static Predicate<String> newAllTablesOraclePredicate() {
		return newTablesOraclePredicate(null, null);
	}

	public static Predicate<String> newTablesOraclePredicate(List<String> excludes, final List<String> includes) {
		List<String> excludesPatterns = Lists.newArrayList(ORACLE_EXCLUDE_TABLE_PATTERNS);

		if (CollectionUtils.isNotEmpty(excludes)) {
			excludesPatterns.addAll(excludes);
		}

		return newTablePredicate(excludesPatterns, includes);
	}

	public static Predicate<String> newTablePredicate(final List<String> excludes, final List<String> includes) {

		return new Predicate<String>() {
			public boolean apply(String tableName) {

				if (CollectionUtils.isNotEmpty(excludes)) {
					for (String excludePattern : excludes) {
						if (tableName.matches(excludePattern.replaceAll("%", ".*"))) {
							return false;
						}
					}
				}
				if (CollectionUtils.isEmpty(includes)) {
					return true;
				}
				for (String includePattern : includes) {
					if (tableName.matches(includePattern.replaceAll("%", ".*"))) {
						return true;
					}
				}
				return false;
			}
		};
	}

	/**
	 * Override default Hibernate 'org.hibernate.tool.hbm2ddl.DatabaseMetadata', because of th use of deprecated method
	 * buildSqlExceptionConverter()
	 * (see https://hibernate.atlassian.net/browse/HHH-9131)
	 * 
	 * @return
	 */
	public static SQLExceptionConverter newSQLExceptionConverter(final Dialect dialect) {
		// Build a valid sql converter
		return new SQLExceptionConverter() {
			private static final long serialVersionUID = 5458961195167573495L;

			SQLExceptionConversionDelegate delegate = dialect.buildSQLExceptionConversionDelegate();;

			@Override
			public JDBCException convert(SQLException sqlException, String message, String sql) {
				JDBCException exception = delegate.convert(sqlException, message, sql);
				if (exception != null) {
					return exception;
				}
				return new GenericJDBCException(message, sqlException, sql);
			}
		};
	}

	public static void shutdownDatabase(Properties connectionProperties) throws SQLException {
		Connection conn = DaoUtils.createConnection(connectionProperties);
		try {
			DaoUtils.shutdownDatabase(conn);
		} finally {
			DaoUtils.closeSilently(conn);
		}
	}
}
