package fr.ifremer.adagio.core.service.technical.synchro.data;

/*
 * #%L
 * Tutti :: Persistence
 * $Id: ReferentialSynchroTableMetadata.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroTableMetadata.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.lang.reflect.Field;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.mapping.ForeignKey;
import org.hibernate.tool.hbm2ddl.ColumnMetadata;
import org.hibernate.tool.hbm2ddl.ForeignKeyMetadata;
import org.hibernate.tool.hbm2ddl.IndexMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;

/**
 * Overrides of the {@link TableMetadata} with some improvements:
 * <ul>
 * <li>Obtains number of columns via {@link #getColumnsCount()}</li>
 * <li>Obtains all columns names available via {@link #getColumnNames()}</li>
 * <li>Obtains primary key column names via {@link #getPkNames()}</li>
 * </ul>
 * <p/>
 * And others methods used to synchronize referentials:
 * <ul>
 * <li>Obtains query to update a row of the table (column names order is the one introduced by method
 * {@link #getColumnNames()}: {@link #getUpdateQuery()}</li>
 * <li>Obtains query to insert a row in the table (column names order is the one introduced by method
 * {@link #getColumnNames()}: {@link #getInsertQuery()}</li>
 * </ul>
 * Created on 1/14/14.
 * 
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 3.0
 */
public class DataSynchroTableMetadata {
	private static final Log log = LogFactory.getLog(DataSynchroTableMetadata.class);

	public static final String PK_SEPARATOR = "~~";

	protected final String selectPrimaryKeysQuery;

	protected final String selectMaxUpdateDateQuery;

	protected final String countQuery;

	protected final TableMetadata delegate;

	protected final Map<String, ColumnMetadata> columns;

	protected final List<String> columnNames;

	protected final Set<String> pkNames;

	protected final int[] pkIndexs;

	protected final String insertQuery;

	protected final String updateQuery;

	protected final boolean withUpdateDateColumn;

	protected final boolean withSynchronizationStatusColumn;

	protected final String countDataToUpdateQuery;

	protected final String selectDataToUpdateQuery;

	protected final String selectAllQuery;

	protected final String selectDataQueryFromPk;

	@SuppressWarnings({ "unchecked", "rawtypes" })
	public DataSynchroTableMetadata(TableMetadata delegate,
			DatabaseMetaData meta) {

		Preconditions.checkNotNull(delegate);
		this.delegate = delegate;

		try {
			Field field = TableMetadata.class.getDeclaredField("columns");
			field.setAccessible(true);
			this.columns = Maps.<String, ColumnMetadata> newLinkedHashMap((Map) field.get(delegate));
			this.withUpdateDateColumn = columns.containsKey("update_date");
			this.withSynchronizationStatusColumn = columns.containsKey("synchronization_status");
			this.columnNames = initColumnNames(columns);
			this.pkNames = initPrimaryKeys(meta);
			Preconditions.checkNotNull(pkNames);
		} catch (Exception e) {
			throw new AdagioTechnicalException(t("adagio.persistence.tableMetadata.instanciation.error", this), e);
		}

		this.pkIndexs = createPkIndex();
		this.insertQuery = createInsertQuery();
		this.updateQuery = createUpdateQuery();
		this.selectMaxUpdateDateQuery = createSelectMaxUpdateDateQuery();
		this.selectPrimaryKeysQuery = createSelectPrimaryKeysQuery();
		this.selectAllQuery = createSelectAllQuery();
		this.selectDataQueryFromPk = createSelectDataFromPkQuery();
		this.selectDataToUpdateQuery = createSelectDataToUpdateQuery();
		this.countQuery = createCountQuery();
		this.countDataToUpdateQuery = createCountDataToUpdateQuery();
	}

	// for tests purposes
	DataSynchroTableMetadata() {

		delegate = null;
		columns = null;
		columnNames = null;
		pkNames = null;
		pkIndexs = null;
		withUpdateDateColumn = false;
		withSynchronizationStatusColumn = false;
		insertQuery = null;
		updateQuery = null;
		countQuery = null;
		countDataToUpdateQuery = null;
		selectPrimaryKeysQuery = null;
		selectMaxUpdateDateQuery = null;
		selectDataToUpdateQuery = null;
		selectAllQuery = null;
		selectDataQueryFromPk = null;
	}

	public Set<String> getPkNames() {
		return pkNames;
	}

	public boolean isWithUpdateDateColumn() {
		return withUpdateDateColumn;
	}

	public boolean isWithSynchronizationStatusColumn() {
		return withSynchronizationStatusColumn;
	}

	public int getColumnsCount() {
		return columns.size();
	}

	public Set<String> getColumnNames() {
		return ImmutableSet.copyOf(columnNames);
	}

	public int getColumnIndex(String name) {
		int result = columnNames.indexOf(name);
		return result;
	}

	public String getName() {
		return delegate.getName();
	}

	public ForeignKeyMetadata getForeignKeyMetadata(ForeignKey fk) {
		return delegate.getForeignKeyMetadata(fk);
	}

	public ColumnMetadata getColumnMetadata(String columnName) {
		return delegate.getColumnMetadata(columnName);
	}

	public String getSchema() {
		return delegate.getSchema();
	}

	public String getCatalog() {
		return delegate.getCatalog();
	}

	public ForeignKeyMetadata getForeignKeyMetadata(String keyName) {
		return delegate.getForeignKeyMetadata(keyName);
	}

	public IndexMetadata getIndexMetadata(String indexName) {
		return delegate.getIndexMetadata(indexName);
	}

	// ------------------------------------------------------------------------//
	// -- queries methods --//
	// ------------------------------------------------------------------------//

	public String getInsertQuery() {
		return insertQuery;
	}

	public String getUpdateQuery() {
		return updateQuery;
	}

	public String getSelectPrimaryKeysQuery() {
		return selectPrimaryKeysQuery;
	}

	public String getSelectMaxUpdateDateQuery() {
		return selectMaxUpdateDateQuery;
	}

	public String getSelectDataQueryFromPk() {
		return selectDataQueryFromPk;
	}

	public String getSelectDataToUpdateQuery(Date fromDate) {
		String sql = fromDate == null ?
				selectAllQuery :
				selectDataToUpdateQuery;
		return sql;
	}

	public String getCountQuery() {
		return countQuery;
	}

	public String getCountDataToUpdateQuery(Date fromDate) {
		String sql = fromDate == null ?
				countQuery :
				countDataToUpdateQuery;
		return sql;
	}

	// ------------------------------------------------------------------------//
	// -- PK methods --//
	// ------------------------------------------------------------------------//

	public int[] getPkIndexs() {
		return pkIndexs;
	}

	public boolean isSimpleKey() {
		return pkIndexs.length == 1;
	}

	public List<Object> getPk(ResultSet incomingData) throws SQLException {
		List<Object> result = Lists.newArrayListWithCapacity(pkIndexs.length);
		for (int pkIndex : pkIndexs) {
			Object pk = incomingData.getObject(pkIndex);
			result.add(pk);
		}
		return result;
	}

	public String toPkStr(List<Object> pkList) {
		StringBuilder sb = new StringBuilder();
		for (Object pk : pkList) {
			sb.append(PK_SEPARATOR).append(pk);
		}
		return sb.substring(PK_SEPARATOR.length());
	}

	public List<Object> fromPkStr(String pk) {
		List<Object> pkList = Lists.newArrayList();
		String[] split = pk.split(PK_SEPARATOR);
		for (String s : split) {
			if ("null".equals(s)) {
				s = null;
			}
			pkList.add(s);
		}
		return pkList;
	}

	// ------------------------------------------------------------------------//
	// -- Protected methods --//
	// ------------------------------------------------------------------------//

	protected List<String> initColumnNames(Map<String, ColumnMetadata> columns) {
		List<String> result = Lists.newArrayListWithCapacity(columns.size());
		for (String name : columns.keySet()) {
			result.add(name);
		}
		return Collections.unmodifiableList(result);
	}

	protected Set<String> initPrimaryKeys(DatabaseMetaData meta) throws SQLException {

		Set<String> result = Sets.newLinkedHashSet();
		ResultSet rs = meta.getPrimaryKeys(getCatalog(), getSchema(), getName());
		try {

			while (rs.next()) {
				result.add(rs.getString("COLUMN_NAME"));
			}
			rs.close();
			return ImmutableSet.copyOf(result);
		} finally {
			DaoUtils.closeSilently(rs);
		}
	}

	protected int[] createPkIndex() {

		int[] result = new int[pkNames.size()];

		int pkI = 0;
		for (String pkName : pkNames) {
			String pkColumnName = pkName.toLowerCase();

			int i = 1;

			int index = -1;
			for (String columnName : columnNames) {
				if (pkColumnName.equals(columnName)) {
					index = i;
				} else {
					i++;
				}
			}
			result[pkI++] = index;
		}
		return result;
	}

	protected String createInsertQuery() {
		if (CollectionUtils.isEmpty(columnNames)) {
			return null;
		}

		StringBuilder queryParams = new StringBuilder();
		StringBuilder valueParams = new StringBuilder();

		for (String columnName : columnNames) {
			queryParams.append(", ").append(columnName);
			valueParams.append(", ?");
		}

		String result = String.format("INSERT INTO %s (%s) VALUES (%s)",
				getName(),
				queryParams.substring(2),
				valueParams.substring(2));
		return result;
	}

	protected String createUpdateQuery() {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		StringBuilder updateParams = new StringBuilder();

		for (String columnName : columnNames) {
			updateParams.append(", ").append(columnName).append(" = ?");
		}

		String result = String.format("UPDATE %s SET %s WHERE %s",
				getName(),
				updateParams.substring(2),
				createPkWhereClause());
		return result;
	}

	protected String createSelectDataFromPkQuery() {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		String result = String.format("SELECT %s FROM %s WHERE %s",
				createSelectParams(),
				getName(),
				createPkWhereClause());
		return result;
	}

	protected String createSelectPrimaryKeysQuery() {
		// Could not update, because no PK found
		if (CollectionUtils.isEmpty(pkNames)) {
			return null;
		}

		String prefix = " || '" + PK_SEPARATOR + "' || ";
		StringBuilder pkParams = new StringBuilder();
		for (String columnName : pkNames) {
			pkParams.append(prefix).append(columnName);
		}
		return String.format("SELECT %s FROM %s",
				pkParams.substring(prefix.length()),
				getName());
	}

	protected String createSelectAllQuery() {
		return "SELECT " + createSelectParams() + " FROM " + getName();
	}

	protected String createSelectMaxUpdateDateQuery() {
		return String.format("SELECT max(update_date) FROM %s", getName());
	}

	protected String createSelectDataToUpdateQuery() {
		String whereClause = createWithUpdateDateWhereClause();
		return "SELECT " + createSelectParams() + " FROM " + getName() + whereClause;
	}

	protected String createCountQuery() {
		return String.format("SELECT count(*) FROM %s", getName());
	}

	protected String createCountDataToUpdateQuery() {
		String whereClause = createWithUpdateDateWhereClause();
		return "SELECT count(*) FROM " + getName() + whereClause;
	}

	protected String createPkWhereClause() {
		Preconditions.checkArgument(CollectionUtils.isNotEmpty(pkNames));
		StringBuilder pkParams = new StringBuilder();

		for (String columnName : pkNames) {
			pkParams.append("AND ").append(columnName).append(" = ?");
		}

		return pkParams.substring(4);
	}

	protected String createWithUpdateDateWhereClause() {
		String whereClause;

		if (isWithUpdateDateColumn()) {

			// add a filter
			whereClause = " WHERE (update_date IS NULL OR update_date > ?)";
		} else {
			whereClause = "";
		}
		return whereClause;
	}

	protected String createSelectParams() {

		StringBuilder queryParams = new StringBuilder();

		for (String columnName : columnNames) {
			queryParams.append(", ").append(columnName);
		}

		return queryParams.substring(2);
	}

	public String getTableLogPrefix() {
		return "[" + getName() + "]";
	}

	public static void closeSilently(ResultSet statement) {
		try {
			if (statement != null && !statement.isClosed()) {

				statement.close();
			}
		} catch (AbstractMethodError e) {
			if (log.isDebugEnabled()) {
				log.debug("Fix this linkage error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (IllegalAccessError e) {
			if (log.isDebugEnabled()) {
				log.debug("Fix this IllegalAccessError error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (Exception e) {
			if (log.isErrorEnabled()) {
				log.error("Could not close statement, but do not care", e);
			}
		}
	}
}
