package fr.ifremer.adagio.core.service.technical.synchro.data;

/*
 * #%L
 * Tutti :: Persistence
 * $Id: DataSynchroServiceImpl.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/DataSynchroServiceImpl.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.ServiceLoader;
import java.util.Set;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.cfg.Environment;
import org.hibernate.dialect.Dialect;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.tool.hbm2ddl.ColumnMetadata;
import org.nuiton.util.TimeLog;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.dao.DataRetrievalFailureException;
import org.springframework.jdbc.datasource.DriverManagerDataSource;
import org.springframework.jdbc.support.JdbcUtils;
import org.springframework.stereotype.Service;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Maps;

import fr.ifremer.adagio.core.config.AdagioConfiguration;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;
import fr.ifremer.adagio.core.service.technical.synchro.data.specific.DataSynchroSpecificTableTask;
import fr.ifremer.adagio.core.type.ProgressionModel;

/**
 * Created on 1/14/14.
 * 
 * @author Benoit Lavenier <benoit.lavenier@e-is.pro>
 * @since 3.5.2
 */
@Service("dataSynchroService")
@Lazy
public class DataSynchroServiceImpl implements DataSynchroService {

	/** Logger. */
	private static final Log log =
			LogFactory.getLog(DataSynchroServiceImpl.class);

	private static final TimeLog TIME =
			new TimeLog(DataSynchroServiceImpl.class);

	@Autowired
	protected DriverManagerDataSource dataSource;

	@Autowired
	protected SessionFactory sessionFactory;

	@Autowired
	protected AdagioConfiguration config;

	protected Dialect localDialect;

	protected Properties dbconnexionProperties;

	protected Map<String, DataSynchroSpecificTableTask> extraTasks;

	@Override
	public Properties getLocalConnectionProperties() {
		if (dbconnexionProperties == null) {
			dbconnexionProperties = new Properties();

			dbconnexionProperties.put(Environment.URL, dataSource.getUrl());
			dbconnexionProperties.put(Environment.USER, dataSource.getUsername());
			dbconnexionProperties.put(Environment.PASS, dataSource.getPassword());

			dbconnexionProperties.put(Environment.DIALECT, config.getHibernateDialect());
		}
		return dbconnexionProperties;
	}

	@Override
	public Dialect getLocalDialect() {
		if (localDialect == null) {
			localDialect = ((SessionFactoryImplementor) sessionFactory).getDialect();
		}
		return localDialect;
	}

	public Map<String, DataSynchroSpecificTableTask> getExtraTasks() {
		if (extraTasks == null) {

			extraTasks = Maps.newHashMap();

			ServiceLoader<DataSynchroSpecificTableTask> loader = ServiceLoader.load(DataSynchroSpecificTableTask.class);
			for (DataSynchroSpecificTableTask task : loader) {
				extraTasks.put(task.getTableName(), task);
			}

		}
		return extraTasks;
	}

	@Override
	public void prepare(Properties remoteConnectionProperties, Properties localConnectionProperties, DataSynchroResult result,
			Predicate<String> tableFilter) {
		Preconditions.checkNotNull(result);
		Preconditions.checkNotNull(remoteConnectionProperties);
		Preconditions.checkNotNull(localConnectionProperties);

		result.setLocalUrl(getUrl(localConnectionProperties));
		result.setRemoteUrl(getUrl(remoteConnectionProperties));

		dbconnexionProperties = null;
		localDialect = null;
		extraTasks = null;

		Connection localConnection = null;
		Connection remoteConnection = null;
		try {

			ProgressionModel progressionModel = result.getProgressionModel();
			progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.prepare.step1"));

			// create local connection
			localConnection = createConnection(localConnectionProperties);

			progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.prepare.step2"));

			// create remote Connection
			remoteConnection = createConnection(remoteConnectionProperties);

			// load metas
			DataSynchroDatabaseMetadata localMeta =
					DataSynchroDatabaseMetadata.loadDatabaseMetadata(
							localConnection,
							getDialect(localConnectionProperties),
							getConfiguration(localConnectionProperties),
							tableFilter);

			DataSynchroDatabaseMetadata remoteMeta =
					DataSynchroDatabaseMetadata.loadDatabaseMetadata(
							remoteConnection,
							getDialect(remoteConnectionProperties),
							getConfiguration(remoteConnectionProperties),
							tableFilter);

			progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.prepare.step3"));

			// check schema
			try {
				checkSchemas(localMeta, remoteMeta, tableFilter);
			} catch (DataRetrievalFailureException e) {
				result.setError(e);
			}

			if (result.isSuccess()) {

				// prepare model (compute update date, count rows to update,...)

				for (String tableName : localMeta.getTableNames(tableFilter)) {

					long t0 = TimeLog.getTime();

					progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.prepare.step4", tableName));

					DataSynchroTableMetadata table = remoteMeta.getTable(tableName);

					if (log.isDebugEnabled()) {
						log.debug("Prepare table: " + tableName);
					}
					prepareTable(table,
							localConnection,
							remoteConnection,
							result);

					TIME.log(t0, "prepare table " + tableName);
				}

				long totalRows = result.getTotalRows();
				if (log.isInfoEnabled()) {
					log.info("Total rows to update: " + totalRows);
				}
				localConnection.rollback();
			}
		} catch (SQLException e) {
			try {
				if (localConnection != null) {
					localConnection.rollback();
				}
			} catch (SQLException e1) {

				// ignore the rolback error
			}
			result.setError(e);
		} finally {
			JdbcUtils.closeConnection(remoteConnection);
			JdbcUtils.closeConnection(localConnection);
		}
	}

	@Override
	public void synchronize(Properties remoteConnectionProperties, Properties localConnectionProperties, DataSynchroResult result,
			Predicate<String> tableFilter) {
		Preconditions.checkNotNull(result);
		Preconditions.checkNotNull(remoteConnectionProperties);
		Preconditions.checkNotNull(localConnectionProperties);

		Connection localConnection = null;
		Connection remoteConnection = null;
		try {

			// create local connection
			localConnection = createConnection(localConnectionProperties);

			// create remote Connection
			remoteConnection = createConnection(remoteConnectionProperties);

			// load metas
			DataSynchroDatabaseMetadata dbMetas =
					DataSynchroDatabaseMetadata.loadDatabaseMetadata(
							remoteConnection,
							getDialect(remoteConnectionProperties),
							getConfiguration(remoteConnectionProperties),
							tableFilter);

			// set total in progression model
			ProgressionModel progressionModel = result.getProgressionModel();
			progressionModel.setTotal(result.getTotalRows());

			// prepare target (desactivate constraints)
			prepareSynch(localConnection);

			try {

				for (String tableName : dbMetas.getTableNames(tableFilter)) {

					long t0 = TimeLog.getTime();

					progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.synchronize.step1", tableName));

					DataSynchroTableMetadata table = dbMetas.getTable(tableName);

					if (log.isInfoEnabled()) {
						log.info("Synchronize table: " + tableName);
					}
					long countToUpdate = result.getNbRows(tableName);

					if (countToUpdate > 0) {

						DataSynchroSpecificTableTask extraTask = getExtraTasks().get(tableName);

						synchronizeTable(dbMetas,
								table,
								localConnection,
								remoteConnection,
								extraTask,
								result);
					}

					TIME.log(t0, "synchronize table " + tableName);
				}
				if (log.isInfoEnabled()) {
					long totalInserts = result.getTotalInserts();
					long totalUpdates = result.getTotalUpdates();
					log.info("Total rows to treat: " + result.getTotalRows());
					log.info("Total rows inserted: " + totalInserts);
					log.info("Total rows  updated: " + totalUpdates);
					log.info("Total rows  treated: " + (totalInserts + totalUpdates));
				}
			} finally {
				releaseSynch(localConnection);
			}

			progressionModel.setMessage(t("adagio.persistence.synchronizeReferential.synchronize.step2"));

			localConnection.commit();

		} catch (SQLException e) {
			try {
				if (localConnection != null) {
					localConnection.rollback();
				}
			} catch (SQLException e1) {

				// ignore the rolback error
			}
			result.setError(e);
		} finally {
			JdbcUtils.closeConnection(remoteConnection);
			JdbcUtils.closeConnection(localConnection);
		}
	}

	/**
	 * Check that the tow given datasource shemas are compatible for a
	 * synchronize operation (same tables with same columns).
	 * <p/>
	 * If schemas are incompatible, then a {@link DataRetrievalFailureException} exception will be thrown.
	 * 
	 * @param schema1
	 *            schema 1 to check
	 * @param schema2
	 *            schema 2 to check
	 */
	protected void checkSchemas(DataSynchroDatabaseMetadata schema1,
			DataSynchroDatabaseMetadata schema2, Predicate<String> tableFilter) {
		Set<String> internalSchemaTableNames = schema1.getTableNames(tableFilter);
		Set<String> externalSchemaTableNames = schema2.getTableNames(tableFilter);
		if (!internalSchemaTableNames.equals(externalSchemaTableNames)) {
			throw new DataRetrievalFailureException("Incompatible schemas");
		}
		for (String tableName : internalSchemaTableNames) {
			DataSynchroTableMetadata internalTable = schema1.getTable(tableName);
			DataSynchroTableMetadata externalTable = schema2.getTable(tableName);
			Set<String> internalColumnNames = internalTable.getColumnNames();
			Set<String> externalColumnNames = externalTable.getColumnNames();
			if (!internalColumnNames.equals(externalColumnNames)) {
				throw new DataRetrievalFailureException("Incompatible schema of table: " + tableName);
			}
			for (String columnName : internalColumnNames) {
				ColumnMetadata internalColumn = internalTable.getColumnMetadata(columnName);
				ColumnMetadata externalColumn = externalTable.getColumnMetadata(columnName);
				String internalColumnTypeName = internalColumn.getTypeName();
				String externalColumnTypeName = externalColumn.getTypeName();
				if (!internalColumnTypeName.equals(externalColumnTypeName)) {
					throw new DataRetrievalFailureException("Incompatible column type of table / column: " + tableName + " / " + columnName);
				}
			}
		}
	}

	protected void prepareTable(DataSynchroTableMetadata table,
			Connection localConnection,
			Connection remoteConnection,
			DataSynchroResult result) throws SQLException {

		String tablePrefix = table.getTableLogPrefix();

		String tableName = table.getName();

		DataSynchroTableTool localDao = new DataSynchroTableTool(localConnection, table);

		long localCount = localDao.count();

		// get last updateDate used by local db
		Timestamp updateDate = null;

		if (localCount < 50000) {

			// only use the update date on small table, for big table we will re-insert all the table content
			updateDate = localDao.getLastUpdateDate();

			if (updateDate != null) {

				// just inscrements of 1 milisecond to not having same
				updateDate = new Timestamp(DateUtils.setMilliseconds(updateDate, 0).getTime());
				updateDate = new Timestamp(DateUtils.addSeconds(updateDate, 1).getTime());
			}
		}

		DataSynchroTableTool remoteDao = new DataSynchroTableTool(remoteConnection, table);

		long countToUpdate = remoteDao.countDataToUpdate(updateDate);

		if (log.isInfoEnabled()) {
			log.info(String.format("%s nb rows to update: %s", tablePrefix, countToUpdate));
		}

		result.setUpdateDate(tableName, updateDate);
		result.addRows(tableName, (int) countToUpdate);

		IOUtils.closeQuietly(localDao);
		IOUtils.closeQuietly(remoteDao);

	}

	protected void synchronizeTable(DataSynchroDatabaseMetadata dbMetas,
			DataSynchroTableMetadata table,
			Connection localConnection,
			Connection remoteConnection,
			DataSynchroSpecificTableTask extraTask,
			DataSynchroResult result) throws SQLException {

		String tableName = table.getName();

		result.getProgressionModel().setMessage(t("adagio.persistence.synchronizeReferential.synchronizeTable", tableName));

		String tablePrefix = table.getTableLogPrefix();

		if (extraTask != null && log.isInfoEnabled()) {
			log.info(tablePrefix + " Will use specific task: " + extraTask);
		}
		DataSynchroTableTool localDao = new DataSynchroTableTool(localConnection, table);
		DataSynchroTableTool remoteDao = new DataSynchroTableTool(remoteConnection, table);

		// get last updateDate used by local db
		Date updateDate = result.getUpdateDate(tableName);

		// get table count
		long count = localDao.count();

		// get existing ids in the local db
		Set<String> existingIds = localDao.getExistingPrimaryKeys();

		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " existingIds: " + existingIds.size());
		}

		boolean bigTable = count > 50000;

		// get data to update from remote db
		ResultSet dataToUpdate = remoteDao.getDataToUpdate(
				bigTable ? null : updateDate);

		try {

			if (bigTable) {

				// big table update strategy
				updateBigTable(dbMetas,
						localDao,
						remoteDao,
						dataToUpdate,
						extraTask,
						result);
			} else {

				// small table update strategy
				updateTable(localDao,
						dataToUpdate,
						result);
			}
			dataToUpdate.close();
		} finally {

			IOUtils.closeQuietly(localDao);
			IOUtils.closeQuietly(remoteDao);
			JdbcUtils.closeResultSet(dataToUpdate);
		}
	}

	/**
	 * To update the content of the given {@code table} on the local db,
	 * from the given {@code incomingData} of the remote db.
	 * <p/>
	 * The algorithm is pretty simple, for each row of the {@code incomingData}, if exists on local table, then do an
	 * update, otherwise do a insert.
	 * <p/>
	 * As an update query is more expensive, we won't use this method for table with a lot of rows, we will prefer to
	 * use the {@code updateBigTable} method instead.
	 * 
	 * @param localDao
	 *            connection on the local db
	 * @param incomingData
	 *            data to update from the remote db
	 * @param result
	 *            where to store operation results
	 * @throws SQLException
	 *             if any sql errors
	 */
	protected void updateTable(DataSynchroTableTool localDao,
			ResultSet incomingData,
			DataSynchroResult result) throws SQLException {

		DataSynchroTableMetadata table = localDao.getTable();

		// get existing ids in the local db
		Set<String> existingIds = localDao.getExistingPrimaryKeys();

		String tableName = table.getName();
		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		result.addTableName(tableName);

		int countR = 0;

		while (incomingData.next()) {

			List<Object> pk = table.getPk(incomingData);
			String pkStr = table.toPkStr(pk);

			boolean doUpdate = existingIds.contains(pkStr);

			if (doUpdate) {

				localDao.executeUpdate(pk, incomingData);

			} else {

				localDao.executeInsert(pk, incomingData);
			}

			countR++;

			reportProgress(result, localDao, countR, tablePrefix);
		}

		localDao.flushQueries();

		int insertCount = localDao.getInsertCount();
		int updateCount = localDao.getUpdateCount();

		result.addInserts(tableName, insertCount);
		result.addUpdates(tableName, updateCount);
		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (inserts: %s, updates: %s)", tablePrefix, countR, insertCount, updateCount));
		}

		if (log.isDebugEnabled()) {
			log.debug(String.format("%s INSERT count: %s", tablePrefix, result.getNbInserts(tableName)));
			log.debug(String.format("%s UPDATE count: %s", tablePrefix, result.getNbUpdates(tableName)));
		}

		result.getProgressionModel().increments(countR % 1000);
	}

	/**
	 * To update the content of the given {@code table} (with a lot of rows) on
	 * the local db, from the given {@code incomingData} of the remote db.
	 * <p/>
	 * We can't use the simple algorithm, since update queries cost too much and is not acceptable when talking on huge
	 * numbers of rows.
	 * <p/>
	 * Here is what to do :
	 * <ul>
	 * <li>Get form the local db the data which are not in remote db, keep them</li>
	 * <li>Delete local table content</li>
	 * <li>Insert remote table in local table</li>
	 * <li>Insert the saved extra rows from original table</li>
	 * </ul>
	 * In that way we will only perform some insert queries.
	 * 
	 * @param dbMetas
	 * @param localDao
	 *            connection on the local db
	 * @param remoteDao
	 *            connection on the local db
	 * @param incomingData
	 *            data to update from the remote db
	 * @param extraTask
	 * @param result
	 *            where to store operation results @throws SQLException if any sql errors
	 */
	protected void updateBigTable(DataSynchroDatabaseMetadata dbMetas,
			DataSynchroTableTool localDao,
			DataSynchroTableTool remoteDao,
			ResultSet incomingData,
			DataSynchroSpecificTableTask extraTask,
			DataSynchroResult result) throws SQLException {

		DataSynchroTableMetadata table = localDao.getTable();
		String tableName = localDao.getTable().getName();

		result.addTableName(tableName);

		String tablePrefix = table.getTableLogPrefix() + " - " + result.getNbRows(tableName);

		// get existing ids in the local db
		Set<String> existingIds = localDao.getExistingPrimaryKeys();

		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " local existingIds: " + existingIds.size());
		}

		Set<String> remoteExistingIds = remoteDao.getExistingPrimaryKeys();

		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " remote existingIds: " + existingIds.size());
		}

		existingIds.removeAll(remoteExistingIds);

		if (log.isDebugEnabled()) {
			log.debug(tablePrefix + " local data existingIds not in remote: " + existingIds.size());
		}
		if (log.isTraceEnabled()) {
			for (String existingId : existingIds) {
				log.trace("- " + existingId);
			}
		}

		// copy extra rows from local

		Map<List<Object>, Object[]> extraRows = Maps.newLinkedHashMap();

		for (String pkStr : existingIds) {

			List<Object> pk = table.fromPkStr(pkStr);

			Object[] extraRow = localDao.findByPk(pk);

			extraRows.put(pk, extraRow);
		}

		// remove obsolete extra rows

		if (extraTask != null) {
			extraRows = extraTask.transformExtraLocalData(dbMetas,
					localDao,
					remoteDao,
					extraRows);

			if (log.isDebugEnabled()) {
				log.debug(tablePrefix + " local data existingIds not in remote (after apply task): " + extraRows.size());
			}
		}

		// delete table
		localDao.deleteAll();

		int countR = 0;

		// add all data from remote
		while (incomingData.next()) {

			List<Object> pk = table.getPk(incomingData);

			localDao.executeInsert(pk, incomingData);

			countR++;

			reportProgress(result, localDao, countR, tablePrefix);
		}

		// re-add extra local rows
		for (Map.Entry<List<Object>, Object[]> entry : extraRows.entrySet()) {

			List<Object> pk = entry.getKey();
			Object[] row = entry.getValue();
			localDao.executeInsert(pk, row);

			countR++;

			reportProgress(result, localDao, countR, tablePrefix);
		}

		localDao.flushQueries();

		int insertCount = localDao.getInsertCount();
		result.addInserts(tableName, insertCount);
		if (log.isInfoEnabled()) {
			log.info(String.format("%s done: %s (inserts: %s)", tablePrefix, countR, insertCount));
		}

		if (log.isDebugEnabled()) {
			log.debug(String.format("%s INSERT count: %s", tablePrefix, result.getNbInserts(tableName)));
		}

		result.getProgressionModel().increments(countR % 1000);
	}

	protected void reportProgress(DataSynchroResult result, DataSynchroTableTool dao, int countR, String tablePrefix) {
		if (countR % 1000 == 0) {
			result.getProgressionModel().increments(1000);
		}

		if (countR % 10000 == 0) {
			if (log.isInfoEnabled()) {
				log.info(String.format("%s Done: %s (inserts: %s, updates: %s)", tablePrefix, countR, dao.getInsertCount(), dao.getUpdateCount()));
			}
		}
	}

	Connection createConnection(Properties connectionProperties) throws SQLException {
		return createConnection(
				connectionProperties.getProperty(Environment.URL),
				connectionProperties.getProperty(Environment.USER),
				connectionProperties.getProperty(Environment.PASS));
	}

	String getUrl(Properties connectionProperties) {
		return connectionProperties.getProperty(Environment.URL);
	}

	Dialect getDialect(Properties connectionProperties) {
		return Dialect.getDialect(connectionProperties);
	}

	Configuration getConfiguration(Properties connectionProperties) {
		return new Configuration().setProperties(connectionProperties);
	}

	Connection createConnection(String jdbcUrl,
			String user,
			String password) throws SQLException {
		Connection connection = DriverManager.getConnection(jdbcUrl,
				user,
				password);
		connection.setAutoCommit(false);
		return connection;
	}

	void prepareSynch(Connection connection) throws SQLException {
		PreparedStatement statement = connection.prepareStatement("SET REFERENTIAL_INTEGRITY FALSE;");
		statement.executeUpdate();
	}

	void releaseSynch(Connection connection) throws SQLException {
		PreparedStatement statement = connection.prepareStatement("SET REFERENTIAL_INTEGRITY TRUE;");
		statement.executeUpdate();
	}

	public void prepare(File dbDirectory, DataSynchroResult result, Predicate<String> tableFilter) {
		Properties remoteConnectionProperties = getRemoteProperties(dbDirectory);
		Properties localConnectionProperties = getLocalConnectionProperties();
		prepare(remoteConnectionProperties, localConnectionProperties, result, tableFilter);
	}

	public void synchronize(File dbDirectory,
			DataSynchroResult result, Predicate<String> tableFilter) {
		Properties remoteConnectionProperties = getRemoteProperties(dbDirectory);
		Properties localConnectionProperties = getLocalConnectionProperties();
		synchronize(remoteConnectionProperties, localConnectionProperties, result, tableFilter);
	}

	protected Properties getRemoteProperties(File dbDirectory) {
		Properties remoteConnectionProperties = new Properties();
		AdagioConfiguration config = AdagioConfiguration.getInstance();

		String jdbcUrl = DaoUtils.getJdbcUrl(dbDirectory,
				config.getDbName());

		DaoUtils.fillConnectionProperties(remoteConnectionProperties,
				jdbcUrl,
				config.getJdbcUsername(),
				config.getJdbcPassword());
		return remoteConnectionProperties;
	}

}
