package fr.ifremer.adagio.core.service.technical.synchro.data;

/*
 * #%L
 * Tutti :: Persistence
 * $Id: ReferentialSynchroDatabaseMetadata.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroDatabaseMetadata.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.lang.reflect.Field;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.cfg.Configuration;
import org.hibernate.dialect.Dialect;
import org.hibernate.exception.spi.SQLExceptionConverter;
import org.hibernate.internal.util.config.ConfigurationHelper;
import org.hibernate.mapping.Table;
import org.hibernate.tool.hbm2ddl.DatabaseMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.ifremer.adagio.core.AdagioTechnicalException;
import fr.ifremer.adagio.core.dao.technical.DaoUtils;

/**
 * Created on 1/14/14.
 * 
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 3.5
 */
public class DataSynchroDatabaseMetadata {

	/** Logger. */
	private static final Log log =
			LogFactory.getLog(DataSynchroDatabaseMetadata.class);

	private static final String TABLE_CATALOG_PATTERN = "TABLE_CAT";
	private static final String TABLE_TYPE_PATTERN = "TABLE_TYPE";
	private static final String TABLE_SCHEMA_PATTERN = "TABLE_SCHEM";
	private static final String REMARKS_PATTERN = "REMARKS";
	private static final String TABLE_NAME_PATTERN = "TABLE_NAME";

	/**
	 * Load the datasource schema for the given connection and dialect.
	 * 
	 * @param connection
	 *            connection of the data source
	 * @param dialect
	 *            dialect to use
	 * @return the datasource schema
	 */
	public static DataSynchroDatabaseMetadata loadDatabaseMetadata(Connection connection,
			Dialect dialect, Configuration configuration, Predicate<String> tableFilter) {
		DataSynchroDatabaseMetadata result = new DataSynchroDatabaseMetadata(connection, dialect, configuration);
		result.prepare(tableFilter);

		/*
		 * Map<String, Map<String, ExtractorField>> tablesFieldsFromColumnName = Maps.newHashMap();
		 * Map<String, ExtractorTable> tablesFromTableName = Maps.newHashMap();
		 * List<ExtractorField> fields = Lists.newArrayList();
		 * // Retrieve all fields
		 * for (ExtractorTable table : tables) {
		 * 
		 * Map<String, ExtractorField> fieldsFromColumnName = Maps.newHashMap();
		 * getTableFields(table, fieldsFromColumnName, internalExtractionsFromTableName, false);
		 * // Store in the temp map
		 * tablesFromTableName.put(table.getTableName(), table);
		 * tablesFieldsFromColumnName.put(table.getTableName(), fieldsFromColumnName);
		 * 
		 * fields.addAll(table.getExtractorFields());
		 * }
		 * 
		 * setTablesJoins(tables, tablesFieldsFromColumnName, tablesFromTableName);
		 * 
		 * return tables;
		 */
		return result;
	}

	protected final DatabaseMetadata delegate;

	protected final Map<String, DataSynchroTableMetadata> tables;

	protected final DatabaseMetaData meta;

	protected final Configuration configuration;

	private SQLExceptionConverter sqlExceptionConverter;

	public DataSynchroDatabaseMetadata(Connection connection, Dialect dialect, Configuration configuration) {
		Preconditions.checkNotNull(connection);
		Preconditions.checkNotNull(dialect);
		Preconditions.checkNotNull(configuration);

		this.configuration = configuration;
		this.sqlExceptionConverter = DataSynchroUtils.newSQLExceptionConverter(dialect);

		try {
			this.delegate = new DatabaseMetadata(connection, dialect, configuration, true);
			Field field = DatabaseMetadata.class.getDeclaredField("sqlExceptionConverter");
			field.setAccessible(true);
			field.set(this.delegate, sqlExceptionConverter);

			this.meta = connection.getMetaData();
		} catch (SQLException e) {
			throw new AdagioTechnicalException(t("adagio.persistence.dbMetadata.instanciation.error", connection), e);
		} catch (Exception e) {
			throw new AdagioTechnicalException(t("adagio.persistence.dbMetadata.instanciation.error", connection), e);
		}
		tables = Maps.newTreeMap();
	}

	public int getTableCount() {
		return tables.size();
	}

	public void prepare(Predicate<String> tableFilter) {
		for (String tableName : getTableNames(tableFilter)) {

			if (log.isDebugEnabled()) {
				log.debug("Load metas of table: " + tableName);
			}
			getTable(tableName, null, null, false);
		}

		Map<String, DataSynchroTableMetadata> tablesByNames = Maps.newHashMap();
		for (DataSynchroTableMetadata table : tables.values()) {
			tablesByNames.put(table.getName(), table);
		}
	}

	public DataSynchroTableMetadata getTable(String name) throws HibernateException {
		String defaultSchema = ConfigurationHelper.getString(AvailableSettings.DEFAULT_SCHEMA, configuration.getProperties());
		String defaultCatalog = ConfigurationHelper.getString(AvailableSettings.DEFAULT_CATALOG, configuration.getProperties());

		return getTable(name, defaultSchema, defaultCatalog, false);
	}

	protected DataSynchroTableMetadata getTable(String name,
			String schema,
			String catalog,
			boolean isQuoted) throws HibernateException {
		String key = Table.qualify(catalog, schema, name);
		DataSynchroTableMetadata synchroTableMetadata = tables.get(key);
		if (synchroTableMetadata == null) {

			TableMetadata tableMetadata = delegate.getTableMetadata(
					name, schema, catalog, isQuoted);
			Preconditions.checkNotNull(tableMetadata, "Could not find db table " + name);

			synchroTableMetadata = new DataSynchroTableMetadata(tableMetadata, meta);
			Preconditions.checkNotNull(synchroTableMetadata,
					"Could not find db table " + name);
			tables.put(key, synchroTableMetadata);
		}
		return synchroTableMetadata;
	}

	public Set<String> getTableNames(Predicate<String> tableFilter) {
		Preconditions.checkNotNull(tableFilter);

		Set<String> tablenames = Sets.newHashSet();

		String defaultSchema = ConfigurationHelper.getString(AvailableSettings.DEFAULT_SCHEMA, configuration.getProperties());
		String defaultCatalog = ConfigurationHelper.getString(AvailableSettings.DEFAULT_CATALOG, configuration.getProperties());

		String[] types = null; // available types are: "TABLE", "VIEW", "SYSTEM TABLE", "GLOBAL TEMPORARY",
								// "LOCAL TEMPORARY", "ALIAS", "SYNONYM"
		if (configuration != null
				&& ConfigurationHelper.getBoolean(AvailableSettings.ENABLE_SYNONYMS, configuration.getProperties(), false)) {
			types = new String[] { "TABLE", "VIEW", "SYNONYM" };
		}
		else {
			types = new String[] { "TABLE", "VIEW" };
		}

		ResultSet res = null;
		try {

			// first pass on the main schema
			res = meta.getTables(defaultCatalog, defaultSchema, "%", types);
			while (res.next()) {
				String tableName = res.getString(TABLE_NAME_PATTERN); //$NON-NLS-1$
				if (tableFilter.apply(tableName)) {
					if (log.isDebugEnabled()) {
						log.debug("found: " + TABLE_CATALOG_PATTERN + "=" + res.getString(TABLE_CATALOG_PATTERN)
								+ " " + TABLE_SCHEMA_PATTERN + "=" + res.getString(TABLE_SCHEMA_PATTERN)
								+ " " + TABLE_NAME_PATTERN + "=" + res.getString(TABLE_NAME_PATTERN)
								+ " " + TABLE_TYPE_PATTERN + "=" + res.getString(TABLE_TYPE_PATTERN)
								+ " " + REMARKS_PATTERN + "=" + res.getString(REMARKS_PATTERN));
					}
					tablenames.add(tableName);
				}
			}
			res.close();

		} catch (SQLException e) {
			throw sqlExceptionConverter.convert(e, "Retrieving database table names", "n/a");
		} finally {
			DaoUtils.closeSilently(res);
		}

		return tablenames;
	}

	// protected void setTablesJoins(List<ExtractorTable> tables, Map<String, Map<String, ExtractorField>>
	// tablesFieldsFromColumnName,
	// Map<String, ExtractorTable> tablesFromTableName) {
	// // Retrieve all foreign keys
	// for (ExtractorTable table : tables) {
	// logger.debug("gatherTableImportedJoins for table : " + table.getTableName());
	// gatherTableImportedJoins(table, tablesFromTableName, tablesFieldsFromColumnName);
	//
	// logger.debug("gatherTableExportedJoins for table : " + table.getTableName());
	// gatherTableExportedJoins(table, tablesFromTableName, tablesFieldsFromColumnName);
	// }
	// }
	//
	// /**
	// *
	// * @param table
	// * @param tablesFromTableName
	// * @param fieldsFromColumnName
	// */
	// protected void gatherTableImportedJoins(ExtractorTable table, Map<String, ExtractorTable> tablesFromTableName,
	// Map<String, Map<String, ExtractorField>> fieldsFromColumnName) {
	// // Get foreign key info :
	// Connection conn = null;
	// ResultSet res = null;
	// try {
	// conn = DataSourceUtils.getConnection(dataSource);
	// DatabaseMetaData meta = conn.getMetaData();
	//
	// res = meta.getImportedKeys(table.getCatalog(), table.getSchema(), table.getTableName());
	// while (res.next()) {
	//
	//                String pkTableName = res.getString(PKTABLE_NAME_PATTERN); //$NON-NLS-1$
	//                String pkColumnName = res.getString(PKCOLUMN_NAME_PATTERN); //$NON-NLS-1$
	//                String fkTableName = res.getString(FKTABLE_NAME_PATTERN); //$NON-NLS-1$
	//                String fkColumnName = res.getString(FKCOLUMN_NAME_PATTERN); //$NON-NLS-1$
	//
	//                int deleteRule = res.getInt(DELETE_RULE_PATTERN); //$NON-NLS-1$
	// // Skip if foreign key is delete cascade (skip link to the parent table)
	// if (deleteRule != JDBC_DELETE_RULE_DELETE_CASCADE) {
	// setTableImportedJoins(tablesFromTableName, fieldsFromColumnName, pkTableName, pkColumnName, fkTableName,
	// fkColumnName);
	// }
	// // else : Nothing to do : the fk field will be set as inactive in the "applyDefaultsXXX" methods
	// }
	// } catch (SQLException e) {
	//            throw new DataRetrievalFailureException("Unable to retrieve joins for table : " + table.getTableName(), e); //$NON-NLS-1$
	// } finally {
	// if (res != null) {
	// try {
	// res.close();
	// } catch (SQLException e) {
	// }
	// }
	// DataSourceUtils.releaseConnection(conn, dataSource);
	// }
	// }
	//
	// protected void setTableImportedJoins(Map<String, ExtractorTable> tablesFromTableName,
	// Map<String, Map<String, ExtractorField>> fieldsFromColumnName,
	// String pkTableName, String pkColumnName, String fkTableName, String fkColumnName) {
	//
	// ExtractorJoinType outerJoinType = extractorJoinTypeDao.load(JoinTypeEnum.OuterJoin.getValue());
	// ExtractorJoinType innerJoinType = extractorJoinTypeDao.load(JoinTypeEnum.InnerJoin.getValue());
	// ExtractorJoinType noneJoinType = extractorJoinTypeDao.load(JoinTypeEnum.None.getValue());
	//
	// ExtractorField fkField = fieldsFromColumnName.get(fkTableName).get(fkColumnName);
	//
	// ExtractorTable pkTable = tablesFromTableName.get(pkTableName);
	// ExtractorField pkField = null;
	// if (pkTable != null) {
	// pkField = fieldsFromColumnName.get(pkTableName).get(pkColumnName);
	// }
	//
	// if (pkTable == null || pkField == null) {
	// // Ignoring the join, because referenced table not bound
	// logger.warn(messageSource.getMessage("ExtractorTableDaoImpl.2", new Object[] { fkTableName, fkColumnName,
	// pkTableName }, null));
	// }
	//
	// else if (fkField == null) {
	// // Ignoring the join, because referenced field not bound (mantis #14725)
	// logger.warn(messageSource.getMessage("ExtractorSetupDaoImpl.0", new Object[] { pkTableName, pkColumnName,
	// fkColumnName }, null));
	// }
	//
	// // Configure a join to use with the referenced table
	// else {
	//
	// // Trying to override from database properties file :
	// String joinTypeName = databaseNames.getImportedJoinType(fkTableName, fkColumnName, pkTableName, pkColumnName);
	//
	// // If no join : reset all join stuff
	// if (noneJoinType.getName().equals(joinTypeName)) {
	// fkField.setJoinExtractorTable(null);
	// fkField.setJoinExtractorField(null);
	// fkField.setExtractorJoinType(noneJoinType);
	// fkField.setVisibilityMode(VisibilityModeMask.None.getValue());
	// // BLA passage de isActive à false (mantis#14752)
	// fkField.setIsActive(false);
	// } else {
	// // Use a outer join by default
	// ExtractorJoinType joinTypeToApplied = outerJoinType;
	// // Try to decode join type
	// if (joinTypeName != null) {
	// JoinTypeEnum e = JoinTypeEnum.valueOf(joinTypeName);
	//
	// if (e == null) {
	// // Ignore the joint type found
	// logger.error(messageSource.getMessage("ExtractorTableDaoImpl.4", new Object[] { fkTableName, fkColumnName,
	// joinTypeName },
	// null));
	// } else {
	// joinTypeToApplied = extractorJoinTypeDao.load(e.getValue());
	// }
	// }
	//
	// // If not nullable, set has inner join
	// else if (!fkField.getIsNullable().booleanValue()) {
	// joinTypeToApplied = innerJoinType;
	// }
	//
	// // Affect properties
	// fkField.setExtractorJoinType(joinTypeToApplied);
	// fkField.setJoinExtractorTable(pkTable);
	// fkField.setJoinExtractorField(pkField);
	// pkField.setParentJoinExtractorField(fkField);
	// }
	// }
	// }
	//
	// /**
	// * Manage field from external table that are linked to the current table
	// * Example : SALE.ID => link from PRODUCE.SALE_FK (exported join)
	// *
	// * @param table
	// * @param tablesFromTableName
	// * @param fieldsFromColumnName
	// */
	// protected void gatherTableExportedJoins(ExtractorTable table, Map<String, ExtractorTable> tablesFromTableName,
	// Map<String, Map<String, ExtractorField>> fieldsFromColumnName) {
	// // Get foreign key info :
	// Connection conn = null;
	// ResultSet res = null;
	// try {
	// conn = DataSourceUtils.getConnection(dataSource);
	// DatabaseMetaData meta = conn.getMetaData();
	//
	// res = meta.getExportedKeys(table.getCatalog(), table.getSchema(), table.getTableName());
	// while (res.next()) {
	//                String pkTableName = res.getString(PKTABLE_NAME_PATTERN); //$NON-NLS-1$
	//                String pkColumnName = res.getString(PKCOLUMN_NAME_PATTERN); //$NON-NLS-1$
	//                String fkTableName = res.getString(FKTABLE_NAME_PATTERN); //$NON-NLS-1$
	//                String fkColumnName = res.getString(FKCOLUMN_NAME_PATTERN); //$NON-NLS-1$
	//
	//                int deleteRule = res.getInt(DELETE_RULE_PATTERN); //$NON-NLS-1$
	// // Skip if foreign key is delete cascade (skip link to the parent table)
	// if (deleteRule == JDBC_DELETE_RULE_DELETE_CASCADE) {
	// setTableExportedJoins(table, tablesFromTableName, fieldsFromColumnName, pkTableName, pkColumnName, fkTableName,
	// fkColumnName);
	// }
	// }
	// } catch (SQLException e) {
	//            throw new DataRetrievalFailureException("Unable to retrieve exported joins for table : " + table.getTableName(), e); //$NON-NLS-1$
	// } finally {
	// if (res != null) {
	// try {
	// res.close();
	// } catch (SQLException e) {
	// }
	// }
	// DataSourceUtils.releaseConnection(conn, dataSource);
	// }
	// }
	//
	// protected void setTableExportedJoins(ExtractorTable table, Map<String, ExtractorTable> tablesFromTableName,
	// Map<String, Map<String, ExtractorField>> fieldsFromColumnName,
	// String pkTableName, String pkColumnName, String fkTableName, String fkColumnName) {
	//
	// ExtractorJoinType outerJointType = extractorJoinTypeDao.load(JoinTypeEnum.OuterJoin.getValue());
	// ExtractorJoinType noneJoinType = extractorJoinTypeDao.load(JoinTypeEnum.None.getValue());
	//
	// ExtractorField pkField = fieldsFromColumnName.get(pkTableName).get(pkColumnName);
	// if (pkField == null) {
	// // attention quand table récursive (DENORMALIZED_BATCH ou BATCH : on arrive ici !!
	// logger.debug("Unable to found this column in the map of fields : " + pkTableName + "." + pkColumnName);
	// } else {
	// ExtractorField field = ExtractorField.Factory.newInstance();
	// field.setFieldName(pkField.getFieldName());
	// field.setSqlType(pkField.getSqlType());
	//
	// field.setName(DEFAULT_FK_FIELD_NAME);
	// field.setExtractorJavaType(pkField.getExtractorJavaType());
	// field.setRankOrder((short) 0); // à placé à la fin grace au TreeNodeComparator
	// field.setVisibilityMode(VisibilityModeMask.None.getValue());
	// field.setIsActive(true);
	// field.setIsNullable(pkField.getIsNullable());
	//
	// ExtractorTable fkTable = tablesFromTableName.get(fkTableName);
	// ExtractorField fkField = null;
	// if (fkTable != null) {
	// fkField = fieldsFromColumnName.get(fkTableName).get(fkColumnName);
	// }
	//
	// if (fkTable == null || fkField == null) {
	// // Ignoring the join, because referenced table not bound
	// logger.warn(messageSource.getMessage("ExtractorTableDaoImpl.3", new Object[] { pkTableName, fkTableName,
	// fkColumnName, fkTableName },
	// null));
	// }
	//
	// // Configure a join to use with the referenced table
	// else {
	// // Use a outer join by default
	// ExtractorJoinType joinTypeToApplied = outerJointType;
	//
	// // Trying to override from database properties file
	// String joinTypeName = databaseNames.getExportedJoinType(pkTableName, pkColumnName, fkTableName, fkColumnName);
	// if (joinTypeName != null) {
	// JoinTypeEnum e = JoinTypeEnum.valueOf(joinTypeName);
	//
	// if (e == null) {
	// // Ignore the joint type found
	// logger.error(messageSource.getMessage("ExtractorTableDaoImpl.5", new Object[] { pkTableName, fkTableName,
	// fkColumnName,
	// joinTypeName }, null));
	// } else {
	// joinTypeToApplied = extractorJoinTypeDao.load(e.getValue());
	// }
	// }
	//
	// // If join type = None : do not apply join
	// if (joinTypeToApplied == null || (JoinTypeEnum.None.getValue() == joinTypeToApplied.getId().intValue())) {
	// field.setJoinExtractorTable(null);
	// field.setJoinExtractorField(null);
	// field.setExtractorBaseFilterType(null);
	// field.setExtractorJoinType(noneJoinType);
	// field.setVisibilityMode(VisibilityModeMask.None.getValue());
	// field.setIsActive(true);
	// }
	// // If valid join type, apply this join on field
	// else {
	// field.setJoinExtractorTable(fkTable);
	// field.setExtractorJoinType(joinTypeToApplied);
	// field.setJoinExtractorField(fkField);
	// fkField.setParentJoinExtractorField(field);
	// if (!fkField.getIsActive().booleanValue() && fkField.getExtractorJoinType().getId().intValue() !=
	// JoinTypeEnum.None.getValue()) {
	// // Active ths FK field if setImportedTables() has done a isActive=false
	// fkField.setIsActive(true);
	// }
	// }
	//
	// // Add the field to outer join into the table
	// table.getExtractorFields().add(field);
	// field.setOwnerExtractorTable(table);
	// }
	// }
	// }

}
