package fr.ifremer.adagio.core.service.technical.synchro;

/*
 * #%L
 * Tutti :: Persistence
 * $Id: ReferentialSynchroTableTool.java 1573 2014-02-04 16:41:40Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/trunk/tutti-persistence/src/main/java/fr/ifremer/adagio/core/service/technical/synchro/ReferentialSynchroTableTool.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.Closeable;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * Created on 1/14/14.
 * 
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 3.0
 */
public class ReferentialSynchroTableTool implements Closeable {

	/** Logger. */
	private static final Log log =
			LogFactory.getLog(ReferentialSynchroTableTool.class);

	protected final Connection connection;

	protected final ReferentialSynchroTableMetadata table;

	protected final PreparedStatement insertStatement;

	protected final PreparedStatement updateStatement;

	protected final int columnCount;

	protected final String tableName;

	protected int insertCount = 0;

	protected int updateCount = 0;

	protected boolean debug;

	public ReferentialSynchroTableTool(ReferentialSynchroTableTool tool,
			ReferentialSynchroTableMetadata table) throws SQLException {
		this(tool.connection, table);
	}

	public ReferentialSynchroTableTool(Connection connection,
			ReferentialSynchroTableMetadata table) throws SQLException {
		this.connection = connection;
		this.table = table;
		this.columnCount = table.getColumnsCount();
		this.tableName = table.getName();

		String insertSql = table.getInsertQuery();
		String updateSql = table.getUpdateQuery();

		insertStatement = connection.prepareStatement(insertSql);
		updateStatement = connection.prepareStatement(updateSql);

		debug = log.isTraceEnabled();
	}

	public void deleteAll() throws SQLException {
		PreparedStatement deleteStatement =
				connection.prepareStatement(
						"DELETE FROM " + table.getName());
		deleteStatement.execute();
	}

	public Object[] findByPk(List<Object> pk) throws SQLException {
		String selectDataSql = table.getSelectDataQueryFromPk();

		PreparedStatement selectStatement =
				connection.prepareStatement(selectDataSql);

		int columnCountIndex = 1;

		for (Object pkColumn : pk) {
			selectStatement.setObject(columnCountIndex++, pkColumn);
		}

		int columnsCount = table.getColumnsCount();
		ResultSet resultSet = selectStatement.executeQuery();
		resultSet.next();

		Object[] result = new Object[columnsCount];

		for (int i = 1; i <= columnsCount; i++) {

			result[i - 1] = resultSet.getObject(i);
		}
		return result;
	}

	public Set<String> getExistingPrimaryKeys() throws SQLException {

		String sql = table.getSelectPrimaryKeysQuery();

		PreparedStatement statement = connection.prepareStatement(sql);

		Set<String> result = Sets.newHashSet();
		try {
			ResultSet resultSet = statement.executeQuery();
			while (resultSet.next()) {
				String pk = resultSet.getString(1);
				result.add(pk);
			}
			statement.close();
			return result;
		} finally {
			closeSilently(statement);
		}
	}

	public ResultSet getDataToUpdate(Date fromDate) throws SQLException {

		String sql = table.getSelectDataToUpdateQuery(fromDate);

		PreparedStatement statement = connection.prepareStatement(sql);
		if (table.isWithUpdateDateColumn() &&
				fromDate != null) {
			statement.setTimestamp(1, new Timestamp(fromDate.getTime()));
		}
		statement.setFetchSize(1000);

		ResultSet result = statement.executeQuery();
		return result;
	}

	@Override
	public void close() throws IOException {
		closeSilently(insertStatement);
		closeSilently(updateStatement);
	}

	public void executeInsert(List<Object> pk, ResultSet incomingData) throws SQLException {

		List<Object> params = null;

		if (debug) {
			params = Lists.newArrayList();
		}

		for (int c = 1; c <= columnCount; c++) {
			Object object = incomingData.getObject(c);
			insertStatement.setObject(c, object);
			if (debug) {
				params.add(object);
			}
		}
		insertCount++;

		insertStatement.addBatch();

		if (debug) {
			log.debug(String.format("%s Execute insert query (pk:%s), params: %s", tableName, pk, params));
		}

		if (insertCount > 0 && insertCount % 1000 == 0) {
			insertStatement.executeBatch();
			insertStatement.clearBatch();
		}
	}

	public void executeInsert(List<Object> pk, Object[] incomingData) throws SQLException {

		for (int c = 1; c <= columnCount; c++) {
			Object object = incomingData[c - 1];
			insertStatement.setObject(c, object);
		}
		insertCount++;

		insertStatement.addBatch();

		if (debug) {
			log.debug(String.format("%s Execute insert query (pk:%s), params: %s", tableName, pk, Arrays.toString(incomingData)));
		}

		if (insertCount > 0 && insertCount % 1000 == 0) {
			insertStatement.executeBatch();
			insertStatement.clearBatch();
		}
	}

	public void executeUpdate(List<Object> pk, ResultSet incomingData) throws SQLException {

		List<Object> params = null;

		if (debug) {
			params = Lists.newArrayList();
		}

		for (int c = 1; c <= columnCount; c++) {
			Object object = incomingData.getObject(c);
			updateStatement.setObject(c, object);
			if (debug) {
				params.add(object);
			}
		}

		int columnCountIndex = columnCount + 1;

		for (Object pkColumn : pk) {
			updateStatement.setObject(columnCountIndex++, pkColumn);
		}

		updateCount++;

		updateStatement.addBatch();

		if (debug) {
			log.debug(String.format("%s Execute update query (pk:%s), params: %s", tableName, pk, params));
		}

		if (updateCount > 0 && updateCount % 1000 == 0) {
			updateStatement.executeBatch();
			updateStatement.clearBatch();
		}
	}

	public int getInsertCount() {
		return insertCount;
	}

	public int getUpdateCount() {
		return updateCount;
	}

	public void flushQueries() throws SQLException {

		if (insertCount > 0 && insertCount % 1000 != 0) {
			insertStatement.executeBatch();
		}
		if (updateCount > 0 && updateCount % 1000 != 0) {
			updateStatement.executeBatch();
		}
	}

	/**
	 * Gets the last updateDate for the given {@code table} using
	 * the given datasource.
	 * 
	 * @return the last update date of the given table, or {@code null} if table does not use a updateDate columns or if
	 *         there
	 *         is no data in table.
	 */
	public Timestamp getLastUpdateDate() throws SQLException {
		Timestamp result = null;

		if (table.isWithUpdateDateColumn()) {

			String sql = table.getSelectMaxUpdateDateQuery();

			PreparedStatement statement = connection.prepareStatement(sql);
			try {
				ResultSet resultSet = statement.executeQuery();
				if (resultSet.next()) {
					result = resultSet.getTimestamp(1);
				}
				statement.close();
			} finally {
				closeSilently(statement);
			}
		}
		return result;
	}

	public long countDataToUpdate(Date fromDate) throws SQLException {

		String sql = table.getCountDataToUpdateQuery(fromDate);

		PreparedStatement statement = connection.prepareStatement(sql);
		if (table.isWithUpdateDateColumn() &&
				fromDate != null) {
			statement.setTimestamp(1, new Timestamp(fromDate.getTime()));
		}

		ResultSet queryResult = statement.executeQuery();
		queryResult.next();
		long result = queryResult.getLong(1);
		return result;
	}

	public long count() throws SQLException {

		String sql = table.getCountQuery();

		PreparedStatement statement = connection.prepareStatement(sql);

		try {
			ResultSet resultSet = statement.executeQuery();
			resultSet.next();
			long result = resultSet.getLong(1);
			statement.close();
			return result;
		} finally {
			closeSilently(statement);
		}
	}

	void closeSilently(Statement statement) {
		try {
			if (statement != null && !statement.isClosed()) {

				statement.close();
			}
		} catch (AbstractMethodError e) {
			if (log.isDebugEnabled()) {
				log.debug("Fix this linkage error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (IllegalAccessError e) {
			if (log.isDebugEnabled()) {
				log.debug("Fix this IllegalAccessError error, damned hsqlsb 1.8.0.7:(");
			}
		} catch (Exception e) {
			if (log.isErrorEnabled()) {
				log.error("Could not close statement, but do not care", e);
			}
		}
	}

	public ReferentialSynchroTableMetadata getTable() {
		return table;
	}
}
