package fr.ifremer.adagio.core.dao.referential.taxon;

/*
 * #%L
 * SIH-Adagio Core for Allegro
 * $Id: TaxonNameDaoImpl.java 12039 2014-04-14 13:03:33Z bl05b3e $
 * $HeadURL: https://forge.ifremer.fr/svn/sih-adagio/tags/adagio-3.5.2/core-allegro/src/main/java/fr/ifremer/adagio/core/dao/referential/taxon/TaxonNameDaoImpl.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Query;
import org.hibernate.Session;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataRetrievalFailureException;
import org.springframework.stereotype.Repository;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;

import fr.ifremer.adagio.core.dao.referential.StatusCode;
import fr.ifremer.adagio.core.dao.technical.hibernate.TemporaryDataHelper;

/** @see TaxonName */
@Repository("taxonNameDao")
public class TaxonNameDaoImpl extends TaxonNameDaoBase implements TaxonNameExtendDao {

	private static final String QUERY_ALL_TAXON = "allTaxonNames";

	private static final String QUERY_ALL_TAXON_IS_REFERENT = "allTaxonNamesIsReferent";

	private static final String QUERY_TAXON_NAME_REFERENT = "taxonNameReferent";

	private static final String QUERY_ALL_TAXON_WITH_TRANSCRIBING = "allTaxonNamesWithTranscribing";

	private static final String QUERY_ALL_TAXON_IS_REFERENT_WITH_TRANSCRIBING = "allTaxonNamesIsReferentWithTranscribing";

	private static final String QUERY_TAXON_NAME_REFERENT_WITH_TRANSCRIBING = "taxonNameReferentWithTranscribing";
	private static final String QUERY_ALL_TRANSCRIBING_FOR_A_TYPE = "allTranscribingForAType";

	@Resource(name = "referenceTaxonDao")
	protected ReferenceTaxonDao referenceTaxonDao;

	/** Constructor used by Spring */
	@Autowired
	public TaxonNameDaoImpl(org.hibernate.SessionFactory sessionFactory) {
		super();
		setSessionFactory(sessionFactory);
	}

	@Override
	public TaxonRefVO[] getAllTaxonNames(boolean withSynonyms) {
		return getAllTaxonNames(withSynonyms, null);
	}

	// FIXME-TC Make this works again within a single request
	@Override
	public TaxonRefVO[] getAllTaxonNames(boolean withSynonyms,
			Integer transcribingId) {
		try {

			Session session = getSession();

			Query query;

			boolean withTranscribing = transcribingId != null;

			// if (withTranscribing) {
			//
			// // with transcribing
			// if (withSynonyms) {
			// query = session.getNamedQuery(QUERY_ALL_TAXON_WITH_TRANSCRIBING);
			// } else {
			// query = session.getNamedQuery(QUERY_ALL_TAXON_IS_REFERENT_WITH_TRANSCRIBING);
			// }
			// query.setInteger("transcribingTypeId", transcribingId);
			// } else {
			//
			// // with no transcribing
			// if (withSynonyms) {
			// query = session.getNamedQuery(QUERY_ALL_TAXON);
			// } else {
			// query = session.getNamedQuery(QUERY_ALL_TAXON_IS_REFERENT);
			// }
			// }

			if (withSynonyms) {
				query = session.getNamedQuery(QUERY_ALL_TAXON);
			} else {
				query = session.getNamedQuery(QUERY_ALL_TAXON_IS_REFERENT);
			}

			List<TaxonRefVO> results = new ArrayList<TaxonRefVO>();
			for (Iterator<Object[]> iterator = query.iterate(); iterator.hasNext();) {
				Object[] cols = iterator.next();
				TaxonRefVO taxonNameRefTaxVO = loadTaxon(cols, false);
				results.add(taxonNameRefTaxVO);
			}
			if (results.size() == 0) {
				return null;
			}

			if (withTranscribing) {
				query = session.getNamedQuery(QUERY_ALL_TRANSCRIBING_FOR_A_TYPE);
				query.setInteger("transcribingTypeId", transcribingId);

				Multimap<Integer, TaxonRefVO> r = Multimaps.index(results, new Function<TaxonRefVO, Integer>() {
					@Override
					public Integer apply(fr.ifremer.adagio.core.dao.referential.taxon.TaxonRefVO input) {
						return input.getReferenceTaxonId();
					}
				});

				for (Iterator<Object[]> iterator = query.iterate(); iterator.hasNext();) {
					Object[] cols = iterator.next();
					Integer referencetaxonId = (Integer) cols[0];
					String externalCode = (String) cols[1];
					Collection<TaxonRefVO> taxonRefVOs = r.get(referencetaxonId);
					if (taxonRefVOs != null)
						for (TaxonRefVO taxonRefVO : taxonRefVOs) {
							taxonRefVO.setExternalCode(externalCode);
						}
				}
			}

			// Return the first element in the list
			return results.toArray(new TaxonRefVO[results.size()]);
		} catch (RuntimeException re) {
			logger.error("getAllTaxonNames failed", re); //$NON-NLS-1$
			throw re;
		}
	}

	@Override
	public TaxonRefVO getTaxonNameReferent(Integer referenceTaxonId) {
		return getTaxonNameReferent(referenceTaxonId, null);
	}

	@Override
	public TaxonRefVO getTaxonNameReferent(Integer referenceTaxonId,
			Integer transcribingId) {
		boolean withTranscribing = transcribingId != null;
		try {

			Session session = getSession();

			Query query;
			if (withTranscribing) {

				query = session.getNamedQuery(QUERY_TAXON_NAME_REFERENT_WITH_TRANSCRIBING);
				query.setInteger("transcribingTypeId", transcribingId);
			} else {
				query = session.getNamedQuery(QUERY_TAXON_NAME_REFERENT);
			}
			query.setInteger("referenceTaxonId", referenceTaxonId);

			Object[] cols = (Object[]) query.uniqueResult();
			if (cols == null) {
				throw new DataRetrievalFailureException("TaxonName with referenceTaxonId=" + referenceTaxonId + " could not found.");
			}
			TaxonRefVO result = loadTaxon(cols, withTranscribing);

			return result;
		} catch (RuntimeException re) {
			logger.error("getTaxonNameReferent failed", re); //$NON-NLS-1$
			throw re;
		}
	}

	@Override
	public TaxonRefVO createAsTemporary(TaxonRefVO taxonNameVO, String comments) {
		Preconditions.checkNotNull(taxonNameVO);
		Preconditions.checkArgument(StringUtils.isNotBlank(taxonNameVO.getName()));

		logger.debug("call create(taxonName)"); //$NON-NLS-1$
		try {

			Session session = getSession();

			// Generate a new id for referenceTaxon
			Integer referenceTaxonId = TemporaryDataHelper.getNewNegativeIdForTemporaryData(session, ReferenceTaxonImpl.class);
			Integer taxonNameId = TemporaryDataHelper.getNewNegativeIdForTemporaryData(session, TaxonNameImpl.class);

			ReferenceTaxon referenceTaxon = ReferenceTaxon.Factory.newInstance();
			referenceTaxon.setId(referenceTaxonId);
			referenceTaxon.setName(TemporaryDataHelper.TEMPORARY_NAME_PREFIX + taxonNameVO.getName());

			referenceTaxon = referenceTaxonDao.create(referenceTaxon);

			TaxonName taxonName = TaxonName.Factory.newInstance();
			taxonName.setId(taxonNameId);
			taxonName.setIsTemporary(true);
			taxonName.setIsReferent(true);
			taxonName.setIsObsolete(false);
			taxonName.setReferenceTaxon(referenceTaxon);
			taxonName.setName(TemporaryDataHelper.TEMPORARY_NAME_PREFIX + taxonNameVO.getName());
			taxonName.setIsNaming(false);
			taxonName.setIsVirtual(false);
			taxonName.setStartDate(new Date());
			taxonName.setCreationDate(new Date());
			taxonName.setUpperRank(1);

			// Taxonomic level
			TaxonomicLevel speciesTaxonomicLevel = (TaxonomicLevel) load(TaxonomicLevelImpl.class, TaxonomicLevelCode.SPECIES.getValue());
			taxonName.setTaxonomicLevel(speciesTaxonomicLevel);

			// Compute the comments with RefTaxCode (if present)
			StringBuffer commentsBuffer = new StringBuffer();
			if (taxonNameVO.getExternalCode() != null) {
				commentsBuffer.append(taxonNameVO.getExternalCode());
			}
			if (comments != null && !comments.isEmpty()) {
				if (commentsBuffer.length() > 0) {
					commentsBuffer.append(" - ");
				}
				commentsBuffer.append(comments);
			}
			taxonName.setComments(commentsBuffer.toString());

			// Create the TaxonName
			create(taxonName);

			// Fill the result VO
			TaxonRefVO result = new TaxonRefVO();
			result.setName(taxonNameVO.getName());
			result.setExternalCode(taxonNameVO.getExternalCode());
			result.setReferenceTaxonId(referenceTaxonId);
			result.setTaxonNameId(taxonNameId);
			result.setIsReference(true);
			result.setStatus(StatusCode.TEMPORARY);

			return result;
		} catch (RuntimeException re) {
			logger.error("create(taxonName) failed", re); //$NON-NLS-1$
			throw re;
		}

	}

	// ------------------------------------------------------------------------//
	// -- Internal methods --//
	// ------------------------------------------------------------------------//

	protected TaxonRefVO loadTaxon(Object[] cols, boolean withTranscribing) {
		TaxonRefVO result = new TaxonRefVO();
		int colIndex = 0;

		// ReferenceTaxon.id
		result.setReferenceTaxonId((Integer) cols[colIndex++]);

		// TaxonName.id
		result.setTaxonNameId((Integer) cols[colIndex++]);

		// Is reference
		result.setIsReference((Boolean) cols[colIndex++]);

		// TaxonName.name (remove temp prefix if present)
		String name = (String) cols[colIndex++];
		if (name != null && name.startsWith(TemporaryDataHelper.TEMPORARY_NAME_PREFIX)) {
			name = name.substring(TemporaryDataHelper.TEMPORARY_NAME_PREFIX.length());
		}
		result.setName(name);

		// Status :
		Boolean tempStatus = (Boolean) cols[colIndex++];
		StatusCode status = tempStatus ? StatusCode.TEMPORARY : StatusCode.ENABLE;
		result.setStatus(status);

		if (withTranscribing) {
			String externalCode = (String) cols[colIndex];
			result.setExternalCode(externalCode);
		}
		return result;
	}
}
