package fr.ifremer.adagio.core.dao.data.operation;

/*
 * #%L
 * SIH-Adagio :: Core for Allegro
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import java.util.Collection;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.hibernate.SessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import com.google.common.collect.Sets;

import fr.ifremer.adagio.core.dao.data.batch.CatchBatchExtendDao;
import fr.ifremer.adagio.core.dao.data.fishingArea.FishingArea;
import fr.ifremer.adagio.core.dao.data.fishingArea.FishingArea2RegulationLocationDao;
import fr.ifremer.adagio.core.dao.data.fishingArea.FishingAreaDao;
import fr.ifremer.adagio.core.dao.data.measure.file.MeasurementFileExtendDao;
import fr.ifremer.adagio.core.dao.data.produce.Produce;
import fr.ifremer.adagio.core.dao.data.produce.ProduceExtendDao;
import fr.ifremer.adagio.core.dao.data.vessel.feature.use.GearUseFeatures;
import fr.ifremer.adagio.core.dao.data.vessel.feature.use.VesselUseFeatures;
import fr.ifremer.adagio.core.dao.referential.ObjectTypeCode;

@Repository("fishingOperationDao")
@Lazy
public class FishingOperationDaoImpl
		extends FishingOperationDaoBase {

	@Autowired
	protected CatchBatchExtendDao catchBatchDao;

	@Autowired
	protected OperationVesselAssociationDao operationVesselAssociationDao;

	@Autowired
	protected FishingArea2RegulationLocationDao fishingArea2RegulationLocationDao;

	@Autowired
	protected FishingAreaDao fishingAreaDao;

	@Autowired
	protected MeasurementFileExtendDao measurementFileDao;

	@Autowired
	protected ProduceExtendDao produceDao;

	/**
	 * Constructor used by Spring
	 */
	@Autowired
	public FishingOperationDaoImpl(SessionFactory sessionFactory) {
		super();
		setSessionFactory(sessionFactory);
	}

	@Override
	public void remove(Collection<FishingOperation> entities) {
		if (entities == null) {
			throw new IllegalArgumentException(
					"FishingOperationDao.remove - 'entities' can not be null");
		}
		for (FishingOperation fishingOperation : entities) {
			remove(fishingOperation);
		}
	}

	@Override
	public void remove(FishingOperation fishingOperation) {
		if (fishingOperation == null) {
			throw new IllegalArgumentException(
					"FishingOperationDao.remove - 'fishingOperation' can not be null");
		}

		// delete catch batch
		if (fishingOperation.getCatchBatch() != null && fishingOperation.getCatchBatch().getId() != null) {
			catchBatchDao.remove(fishingOperation.getCatchBatch().getId());
			getSession().flush();
		}

		Set<FishingArea> fishingAreas = Sets.newHashSet();

		// remove gear use features
		if (CollectionUtils.isNotEmpty(fishingOperation.getGearUseFeatures())) {
			for (GearUseFeatures gearUseFeatures : fishingOperation.getGearUseFeatures()) {

				gearUseFeatures.setOperation(null);
				gearUseFeatures.getGearUseMeasurements().clear();

				if (CollectionUtils.isNotEmpty(gearUseFeatures.getFishingAreas())) {
					for (FishingArea fishingArea : gearUseFeatures.getFishingAreas()) {
						fishingArea.setGearUseFeatures(null);
					}
					fishingAreas.addAll(gearUseFeatures.getFishingAreas());
				}
			}
			// must remove all features content before removing them (data integrity will then failed otherwise)
			getSession().flush();
			fishingOperation.getGearUseFeatures().clear();
		}

		// remove vessel use features
		if (CollectionUtils.isNotEmpty(fishingOperation.getVesselUseFeatures())) {
			for (VesselUseFeatures vesselUseFeatures : fishingOperation.getVesselUseFeatures()) {

				vesselUseFeatures.setOperation(null);
				vesselUseFeatures.getVesselUseMeasurements().clear();

				if (CollectionUtils.isNotEmpty(vesselUseFeatures.getFishingAreas())) {
					for (FishingArea fishingArea : vesselUseFeatures.getFishingAreas()) {
						fishingArea.setVesselUseFeatures(null);
					}
					fishingAreas.addAll(vesselUseFeatures.getFishingAreas());
				}
			}

			// must remove all features content before removing them (data integrity will then failed otherwise)
			getSession().flush();
			fishingOperation.getVesselUseFeatures().clear();
		}

		// remove vessel position
		if (CollectionUtils.isNotEmpty(fishingOperation.getVesselPositions())) {
			fishingOperation.getVesselPositions().clear();
		}

		// remove vessel person features
		if (CollectionUtils.isNotEmpty(fishingOperation.getVesselPersonFeatures())) {
			fishingOperation.getVesselPersonFeatures().clear();
		}

		// remove gear physical features
		fishingOperation.setGearPhysicalFeatures(null);

		// remove catch batch
		fishingOperation.setCatchBatch(null);

		// remove samples
		if (CollectionUtils.isNotEmpty(fishingOperation.getSamples())) {
			fishingOperation.getSamples().clear();
		}

		// remove operation vessel associations
		if (CollectionUtils.isNotEmpty(fishingOperation.getOperationVesselAssociations())) {
			for (OperationVesselAssociation asso : fishingOperation.getOperationVesselAssociations()) {
				operationVesselAssociationDao.remove(asso);
			}
			fishingOperation.getOperationVesselAssociations().clear();
		}

		// remove fishing areas
		if (CollectionUtils.isNotEmpty(fishingAreas)) {
			for (FishingArea fishingArea : fishingAreas) {
				fishingArea.setProduce(null);
				fishingArea.setGearUseFeatures(null);
				fishingArea2RegulationLocationDao.remove(fishingArea.getRegulationLocations());
				fishingArea.getRegulationLocations().clear();
			}
			getSession().flush();
			fishingAreaDao.remove(fishingAreas);
		}

		getSession().flush();

		// remove fishing operation produces
		if (CollectionUtils.isNotEmpty(fishingOperation.getProduces())) {
			for (Produce produce : fishingOperation.getProduces()) {
				produce.getProduceQuantificationMeasurements().clear();
				produce.getProduceSortingMeasurements().clear();
				getSession().delete(produce);
			}
			fishingOperation.getProduces().clear();
			getSession().flush();
		}

		// remove fishing operation
		if (fishingOperation.getFishingTrip() != null && fishingOperation.getFishingTrip().getOperations() != null) {
			fishingOperation.getFishingTrip().getOperations().remove(fishingOperation);
			fishingOperation.setFishingTrip(null);
		}
		getSession().delete(fishingOperation);
		getSession().flush();

		measurementFileDao.deleteAllAttachment(
				ObjectTypeCode.OPERATION.getValue(),
				fishingOperation.getId());
	}
}
