/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: ThesaurusTreeHelper.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://tchemit@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3/vradi-swing/src/main/java/com/jurismarches/vradi/ui/offer/thesaurus/helpers/ThesaurusTreeHelper.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.offer.thesaurus.helpers;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.swing.*;
import javax.swing.tree.TreeModel;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService.ServiceListenerType;

import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.helpers.ThesaurusDataHelper;
import com.jurismarches.vradi.ui.helpers.VradiTreeHelper;
import com.jurismarches.vradi.ui.offer.thesaurus.loadors.RootThesaurusNodeLoadors;
import com.jurismarches.vradi.ui.offer.thesaurus.loadors.ThesaurusNodeLoadors;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;
import org.nuiton.wikitty.WikittyServiceEvent;

/**
 * Navigation tree helpers.
 *
 * @author sletellier
 * @see jaxx.runtime.swing.nav.tree.NavTreeHelper
 */
public class ThesaurusTreeHelper extends VradiTreeHelper {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(ThesaurusTreeHelper.class);

    protected RootThesaurus rootThesaurus;

    protected Thesaurus thesaurus;

    protected boolean listening = true;

    /**
     * Create helper with first nodes are the root thesaurus
     *
     * @param dataProvider data provider
     */
    public ThesaurusTreeHelper(VradiDataProvider dataProvider) {
        super();
        init(dataProvider);
    }

    /**
     * Create helper with root thesaurus for root node
     *
     * @param dataProvider  data provider
     * @param rootThesaurus root thesaurus for root node
     */
    public ThesaurusTreeHelper(VradiDataProvider dataProvider, RootThesaurus rootThesaurus) {
        super();
        this.rootThesaurus = rootThesaurus;
        init(dataProvider);
    }

    /**
     * Create helper with thesaurus for root node
     *
     * @param dataProvider data provider
     * @param thesaurus    thesaurus for root node
     */
    public ThesaurusTreeHelper(VradiDataProvider dataProvider, Thesaurus thesaurus) {
        super();
        this.thesaurus = thesaurus;
        init(dataProvider);
    }

    protected void init(VradiDataProvider dataProvider) {
        setDataProvider(dataProvider);

        // register each tree on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, ServiceListenerType.ALL);
    }

    public TreeModel createTreeModel() {
        
        VradiTreeNode root;
        if (thesaurus != null) {
            if (log.isDebugEnabled()) {
                log.debug("[Root creation] thesaurus " + thesaurus.getName());
            }

            root = new VradiTreeNode(
                    Thesaurus.class,
                    thesaurus.getWikittyId(),
                    null,
                    getChildLoador(ThesaurusNodeLoadors.class)
            );
        } else if (rootThesaurus != null) {
            if (log.isDebugEnabled()) {
                log.debug("[Root creation] root thesaurus " + rootThesaurus.getName());
            }
            root = new VradiTreeNode(
                    RootThesaurus.class,
                    rootThesaurus.getWikittyId(),
                    null,
                    getChildLoador(ThesaurusNodeLoadors.class)
            );
        } else {
            if (log.isDebugEnabled()) {
                log.debug("[Root creation] all root thesaurus");
            }
            root = new VradiTreeNode(
                    String.class,
                    "Root node",
                    null,
                    getChildLoador(RootThesaurusNodeLoadors.class)
            );
        }

        TreeModel model = createModel(root);

        // Populate childs nodes
        root.populateChilds(getBridge(), getDataProvider());

        registerLoadedIds(root);

        return model;
    }

    /**
     * Try to select thesaurus pass in param
     *
     * @param id of thesaurus to select
     */
    public void tryToSelect(String id) {
        Thesaurus thesaurus = VradiService.getWikittyProxy().restore(Thesaurus.class, id);
        tryToSelect(thesaurus);
    }

    /**
     * Try to select thesaurus pass in param
     *
     * @param thesaurus to select
     */
    public void tryToSelect(Thesaurus thesaurus) {
        // Skip if root
        if (!thesaurus.getWikittyId().equals(getRootNode().getId())) {

            VradiTreeNode nodeFound = findThesaurusNode(thesaurus);
            if (nodeFound != null) {
                selectNode(nodeFound);
            }

        }
    }

    /**
     * Try to select thesaurus ids pass in param
     *
     * @param thesaurusIds list of thesaurus ids to select
     */
    public void tryToSelect(List<String> thesaurusIds) {
        WikittyProxy proxy = VradiService.getWikittyProxy();
        tryToSelect(proxy.restore(Thesaurus.class, thesaurusIds));
    }

    /**
     * Try to select thesaurus pass in param
     *
     * @param thesauruses list of thesaurus to select
     */
    public void tryToSelect(Collection<Thesaurus> thesauruses) {
        if (thesauruses == null) {
            // Do nothing
            return;
        }
        List<VradiTreeNode> nodes = new ArrayList<VradiTreeNode>();

        // Find recursivly
        for (Thesaurus thesaurus : thesauruses) {

            if (thesaurus != null) {

                String id = thesaurus.getWikittyId();

                // Skip if root
                if (!id.equals(getRootNode().getId())) {

                    VradiTreeNode toSelect = findThesaurusNode(thesaurus);

                    if (toSelect != null) {
                        nodes.add(toSelect);
                    }
                }
            }
        }

        selectNodes(nodes);
    }

    /**
     * Try to unselect thesaurus pass in param
     *
     * @param id of thesaurus to unselect
     */
    public void tryToUnselect(String id) {
        WikittyProxy proxy = VradiService.getWikittyProxy();
        tryToUnselect(proxy.restore(Thesaurus.class, id));
    }

    /**
     * Try to unselect thesaurus pass in param
     *
     * @param thesaurus to unselect
     */
    public void tryToUnselect(Thesaurus thesaurus) {
        tryToUnselect(thesaurus.getWikittyId());

        // Find recursivly
        VradiTreeNode nodeFound = findThesaurusNode(thesaurus);
        if (nodeFound != null) {
            unSelectNode(nodeFound);
        }
    }

    /**
     * Try to unselect thesaurus ids pass in param
     *
     * @param thesaurusIds list of thesaurus ids to select
     */
    public void tryToUnselect(List<String> thesaurusIds) {
        WikittyProxy proxy = VradiService.getWikittyProxy();
        tryToUnselect(proxy.restore(Thesaurus.class, thesaurusIds));
    }

    /**
     * Try to unselect thesaurus pass in param
     *
     * @param thesauruses list of thesaurus to unselect
     */
    public void tryToUnselect(Collection<Thesaurus> thesauruses) {
        if (thesauruses == null) {
            // Do nothing
            return;
        }
        List<VradiTreeNode> nodes = new ArrayList<VradiTreeNode>();

        // Find recursivly
        for (Thesaurus thesaurus : thesauruses) {

            if (thesaurus != null) {

                String id = thesaurus.getWikittyId();

                // Skip if root
                if (!id.equals(getRootNode().getId())) {

                    VradiTreeNode toSelect = findThesaurusNode(thesaurus);

                    if (toSelect != null) {
                        nodes.add(toSelect);
                    }
                }
            }
        }

        unSelectNodes(nodes);
    }

    protected VradiTreeNode findThesaurusNode(String thesaurusId) {
        WikittyProxy proxy = VradiService.getWikittyProxy();
        return findThesaurusNode(proxy.restore(Thesaurus.class, thesaurusId));
    }

    protected VradiTreeNode findThesaurusNode(Thesaurus thesaurus) {

        VradiTreeNode parentNode = getRootNode();
        VradiTreeNode rootNode = getChild(parentNode, thesaurus.getRootThesaurus());

        // If rootNode is rootThesaurus
        if (rootNode != null) {
            parentNode = rootNode;
        }

        WikittyProxy proxy = VradiService.getWikittyProxy();

        // To optimize thesaurus search
        List<Thesaurus> parentList = getParentsPathThesaurus(proxy, thesaurus);
        
        // Find in depth for loading only necessary nodes
        for (Thesaurus parent : parentList) {
            VradiTreeNode nodeFound = getChild(parentNode, parent.getWikittyId());
            if (nodeFound != null) {
                parentNode = nodeFound;
            }
        }
        
        // If not found
        if (!parentNode.getId().equals(thesaurus.getWikittyId())) {
            return null;
        }
        return parentNode;
    }
 
    protected List<Thesaurus> getParentsPathThesaurus(WikittyProxy proxy, Thesaurus thesaurus) {

        List<Thesaurus> parents = new ArrayList<Thesaurus>();
        if (!ThesaurusDataHelper.isFirstChild(thesaurus)) {
            Thesaurus parent = proxy.restore(Thesaurus.class, thesaurus.getParent());

            if (parent != null) {
                List<Thesaurus> thesauruses = getParentsPathThesaurus(proxy, parent);
                parents.addAll(thesauruses);
            }
        }
        parents.add(thesaurus);

        return parents;
    }

    protected List<String> extractIds(List<Thesaurus> beans){
        List<String> ids = new ArrayList<String>();
        if (beans != null){
            for (Thesaurus bean : beans){
                if (bean != null) {
                    ids.add(bean.getWikittyId());
                }
            }
        }
        return ids;
    }

    public static RootThesaurus getRootThesaurus(VradiTreeNode node) {
        if (!node.getInternalClass().equals(RootThesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreRootThesaurus(node.getId());
    }

    public static Thesaurus getThesaurus(VradiTreeNode node) {
        if (!node.getInternalClass().equals(Thesaurus.class)) {
            return null;
        }
        return ThesaurusDataHelper.restoreThesaurus(node.getId());
    }

    protected void refresh(VradiTreeNode node) {

        if (node == null) {
            return;
        }
        
        getBridge().nodeChanged(node);
    }

    @Override
    public void putWikitty(final String wikittyId, Set<String> wikittyExtensions) {

        // Thesaurus
        if (wikittyExtensions.contains(Thesaurus.EXT_THESAURUS)) {

            // Restore
            final Thesaurus thesaurusConcerned = ThesaurusDataHelper.restoreThesaurus(wikittyId);

            if (thesaurusConcerned == null) {
                return;
            }
            
            // Search on parent to creation case
            if (idsLoaded.contains(thesaurusConcerned.getParent())) {
                
                VradiTreeNode existingNode = findThesaurusNode(wikittyId);

                // cas modification
                if (existingNode != null) {

                    // Parent id
                    VradiTreeNode parent = existingNode.getParent();

                    // Move case
                    String newParentId = thesaurusConcerned.getParent();
                    if (parent != null && !parent.getId().equals(newParentId)) {

                        VradiTreeNode newParentNode = findThesaurusNode(newParentId);

                        // Delete old
                        getBridge().removeNodeFromParent(existingNode);

                        // Insert
                        // Verify that node is not already created
                        if (findThesaurusNode(wikittyId) == null) {
                            insertNode(newParentNode, existingNode);
                        }

                    } else {
                        if (log.isDebugEnabled()) {
                            log.debug("Modification case :  " + thesaurusConcerned.getName());
                        }

                        // Modification case
                        refresh(existingNode);
                    }
                }

                // Creation case
                else {

                    if (log.isDebugEnabled()) {
                        log.debug("Creation case :  " + thesaurusConcerned.getName());
                    }

                    // Find parent node
                    VradiTreeNode parentNode = findThesaurusNode(thesaurusConcerned.getParent());

                    // Tri
                    VradiTreeNode newNode = getChildLoador(ThesaurusNodeLoadors.class).createNode(wikittyId, getDataProvider());
                    // Verify that node is not already created
                    if (findThesaurusNode(wikittyId) == null) {
                        insertNode(parentNode, newNode);
                    }
                }
            }
        }

        // Root thesaurus
        if (wikittyExtensions.contains(RootThesaurus.EXT_ROOTTHESAURUS)) {

            // Find existing
            VradiTreeNode existingNode = findThesaurusNode(wikittyId);

            // Modification case
            if (existingNode != null) {
                refresh(existingNode);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Refresh root thesaurus : " + wikittyId);
                }

                // Creation
                VradiTreeNode newNode = getChildLoador(RootThesaurusNodeLoadors.class).createNode(wikittyId, getDataProvider());
                insertNode(getRootNode(), newNode);
            }
        }
    }

    @Override
    public void removeWikitty(WikittyServiceEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("Receive wikitty service remove event : " + event);
        }

        for (String wikittyId : event.getIds()) {

            // If wikitty id it's not loaded, the event dont concerne this tree
            if (isLoadedId(wikittyId)) {

                VradiTreeNode node = findThesaurusNode(wikittyId);

                idsLoaded.remove(wikittyId);
                if (node != null) {

                    if (log.isDebugEnabled()) {
                        log.debug("Removing node " + wikittyId + " from parent");
                    }

                    VradiTreeNode parent = node.getParent();
                    int index = parent.getIndex(node);
                    node.removeFromParent();
                    getBridge().nodesWereRemoved(parent, new int[] {index}, new VradiTreeNode[]{node});
                }
            }
        }
    }
}
