package com.cybelia.sandra.notifier;

import com.cybelia.sandra.entities.notifier.Cron;
import java.util.Date;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.quartz.CronTrigger;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.Scheduler;
import org.quartz.impl.StdSchedulerFactory;

public class NotifierManager {

    protected static final Log log = LogFactory.getLog(NotifierManager.class);

    protected static NotifierManager manager;
    protected Scheduler scheduler;

    protected NotifierManager() {
    }

    /**
     * Singleton
     */
    public static NotifierManager getManager() {
        if (manager == null) {
            manager = new NotifierManager();
        }
        return manager;
    }

    /**
     * Lance toutes les notifications en cours
     */
    public void start(TopiaContext transaction) {
        try {
            if (scheduler == null) {
                scheduler = StdSchedulerFactory.getDefaultScheduler();

                String hql = "SELECT cron from " + Cron.class.getName() + " cron WHERE " +
                        "cron.enable = :enable AND " +
                        "cron.expression IS NOT NULL AND " +
                        "cron.expression != '' AND " +
                        "(cron.dateStart IS NULL OR cron.dateStart >= :date) AND " +
                        "(cron.dateEnd IS NULL OR cron.dateEnd <= :date)";
                List<Cron> crons = transaction.findAll(hql, "enable", true, "date", new Date());

                for (Cron cron : crons) {
                    del(cron);
                    addCron(cron);
                }
                log.info("Notifier started.");
            }
        } catch (Exception eee) {
            log.error("Starting failed", eee);
        } finally {
            if (transaction != null && !transaction.isClosed()) {
                try {
                    transaction.commitTransaction();
                    transaction.closeContext();
                } catch (TopiaException eee) {
                    log.error("Failed to close transaction", eee);
                }
            }
        }
    }

    /**
     * Arrete toutes les notifications en cours
     */
    public void stop() {
        try {
            scheduler.shutdown();
            log.info("Notifier stoped.");
        } catch (Exception e) {
            log.error("Stoping failed", e);
        }
    }

    /**
     * Ajout d'une notification
     */
    public void add(Cron cron) {
        try {
            boolean enable = cron.getEnable();
            String expression = cron.getExpression();
            if (enable && expression != null && !"".equals(expression)) {
                addCron(cron);
            }
        } catch (Exception e) {
            log.error("Adding failed", e);
        }
    }

    /**
     * Suppression d'une notification
     */
    public void del(Cron cron) {
        try {
            boolean result = scheduler.deleteJob(cron.getLabel(), Scheduler.DEFAULT_GROUP);
            if (result) {
                log.info("Deleted cron : " + cron.getLabel());
            } else {
                log.info("Failed delete cron : " + cron.getLabel());
            }
        } catch (Exception e) {
            log.error("Deleting failed", e);
        }
    }

    /**
     * Ajout dans le Scheduler d'un notification
     */
    protected void addCron(Cron cron) throws Exception {
        CronTrigger cronTrigger = new CronTrigger(cron.getLabel(),
                Scheduler.DEFAULT_GROUP,
                cron.getLabel(),
                Scheduler.DEFAULT_GROUP,
                cron.getDateStart(),
                cron.getDateEnd(),
                cron.getExpression());

        JobDetail jobDetail = new JobDetail(cron.getLabel(), Scheduler.DEFAULT_GROUP, NotifierJob.class);
        JobDataMap jobDataMap = new JobDataMap();
        jobDataMap.put("cronTopiaId", cron.getTopiaId());
        jobDetail.setJobDataMap(jobDataMap);

        scheduler.scheduleJob(jobDetail, cronTrigger);
        log.info("Added cron : " + cron.getLabel());
    }

    /**
     * Job de notification
     */
    public static class NotifierJob implements Job {

        @Override
        public void execute(JobExecutionContext context) throws JobExecutionException {
            JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
            String cronTopiaId = jobDataMap.getString("cronTopiaId");
            try {
                log.info("Notify cron " + cronTopiaId);
                new SandraNotifier().notifyCron(cronTopiaId);
            } catch (TopiaException te) {
                log.error("Error in notification", te);
                new JobExecutionException(te, true);
            }
        }
    }
}