package com.cybelia.sandra.ibu;

import com.cybelia.sandra.SandraConfig;
import com.cybelia.sandra.SandraHelper;
import com.cybelia.sandra.SandraSchedulerConfigHelper;
import com.cybelia.sandra.ibu.csv.CSVReaderGeneric;
import com.cybelia.sandra.ibu.csv.bean.Ibu;
import com.cybelia.sandra.ibu.csv.bean.IbuAutorisation;
import com.cybelia.sandra.ibu.csv.bean.IbuCamion;
import com.cybelia.sandra.ibu.csv.bean.IbuChauffeur;
import com.cybelia.sandra.ibu.csv.bean.IbuEleveurDuplicated;
import com.cybelia.sandra.ibu.csv.bean.IbuEleveurGPS;
import com.cybelia.sandra.ibu.csv.bean.IbuEvent;
import com.cybelia.sandra.ibu.csv.bean.IbuLabel;
import com.cybelia.sandra.ibu.csv.bean.IbuTransporteur;
import com.cybelia.sandra.ibu.csv.bean.IbuUser;
import com.cybelia.sandra.ibu.csv.bean.IbuUsine;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderAutorisation;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderCamion;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderChauffeur;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderEleveurDuplicated;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderEleveurGPS;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderEvent;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderIbu;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderLabel;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderTransporteur;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderUser;
import com.cybelia.sandra.ibu.csv.reader.CSVReaderUsine;
import com.cybelia.sandra.ibu.manager.ManagerInjector;
import com.cybelia.sandra.ibu.notification.SandraIbuMailer;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import org.apache.commons.io.FileExistsException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaNotFoundException;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.DateUtil;
import org.nuiton.util.FileUtil;

/**
 * @author letellier
 */
public class IBU {

    protected static final Log log = LogFactory.getLog(IBU.class);

    protected String codeSociete;

    public IBU() {

        // Must find all files
        this.codeSociete = ".*";
    }

    public IBU(String codeSociete) {
        this.codeSociete = codeSociete;
    }

    public void injectIBU() throws TopiaNotFoundException, IOException {

        // Recuperation des repertoires de travails
        ApplicationConfig config = SandraConfig.getConfig();
        String ibuPath = SandraSchedulerConfigHelper.getIbuPath(config);
        String ibuWorkPath = SandraSchedulerConfigHelper.getIbuWorkPath(config);
        String ibuDonePath = SandraSchedulerConfigHelper.getIbuDonePath(config);

        String ibuPatternFile = "^" + codeSociete + SandraSchedulerConfigHelper.getPatternFile(config);

        // Deplacement des fichiers qui n'ont pas ete integre (rep work)
        File ibuWorkFile = new File(ibuWorkPath);
        List<File> workFiles = FileUtil.find(ibuWorkFile, ibuPatternFile, false);
        for (File workFile : workFiles) {
            String ibuFilePath = ibuPath + File.separator + workFile.getName();
            File ibuFile = new File(ibuFilePath);

            try {
                FileUtils.moveFile(workFile, ibuFile);
            } catch (FileExistsException eee) {
                FileUtils.deleteQuietly(workFile);
            }
        }

        // Recupertation des fichiers
        if (log.isDebugEnabled()) {
            log.debug("Injecting path files with template " + ibuPatternFile + " : " + ibuPath);
        }
        File ibuPathFile = new File(ibuPath);
        List<File> files = FileUtil.find(ibuPathFile, ibuPatternFile, false);
        Collections.sort(files, new IbuFileComparator(ibuPatternFile));

        Date start = new Date();

        // Initialisation du context
        ManagerInjector manager = new ManagerInjector();

        List<Integer> statsByLine = new ArrayList<Integer>();
        for (File f : files) {

            String fileName = f.getName();

            // Deplacement du fichier vers work
            String work = ibuWorkPath + File.separator + fileName;
            File workFile = new File(work);
            try {
                FileUtils.moveFile(f, workFile);
            } catch (FileExistsException eee) {
                FileUtils.deleteQuietly(f);
            }

            log.info("Injecting : " + fileName);
            try {
                // Injection IBU
                FileInjector<Ibu> fileInjector = new FileInjector<Ibu>(new CSVReaderIbu(), workFile, manager) {

                    @Override
                    public void inject(ManagerInjector manager, List<Ibu> toInject) {
                        manager.injectIbus(contextMigration, toInject);
                    }
                };

                Date startFile = new Date();

                fileInjector.doInject();

                Date endDate = new Date();
                int s = DateUtil.getDifferenceInSeconds(startFile, endDate);

                int ms = SandraHelper.getDifferenceInMilliseconds(startFile, endDate);
                int size = fileInjector.getNbLines();
                int msByLines = 0;
                if (size > 0) {
                    msByLines = ms/size;
                }
                statsByLine.add(msByLines);
                log.info("File '" + fileName + "' with " + size + " lines was integrated in " + s + "s : " + msByLines + "ms/lines");
            } finally {

                // Deplacement du fichier vers old
                String done = ibuDonePath + File.separator + fileName;
                File doneFile = new File(done);
                try {
                    FileUtils.moveFile(workFile, doneFile);
                } catch (FileExistsException eee) {
                    log.warn("File '" + fileName + "' already in done, delete this one");
                    FileUtils.deleteQuietly(workFile);
                }
            }
        }

        int size = files.size();
        if (log.isInfoEnabled() && size > 0) {
            Date end = new Date();
            int min = DateUtil.getDifferenceInMinutes(start, end);

            // Calculate average
            int sum = 0;
            for (int stat : statsByLine) {
                sum += stat;
            }

            int average = sum / size;

            log.info("Injection of " + size + " files executed in " + min + "min : " + average + "ms/lines");
        }
    }

    public void injectIBUContent(String content) throws TopiaNotFoundException, IOException {
        injectIBUContent(new ManagerInjector(), content);
    }

    public void injectIBUContent(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {
        StringInjector<Ibu> injector = new StringInjector<Ibu>(managerInjector, new CSVReaderIbu(), content) {

            @Override
            public void inject(ManagerInjector manager, List<Ibu> toInject) {
                manager.injectIbus(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    public void injectLabels(String content) throws TopiaNotFoundException, IOException {
        injectLabels(new ManagerInjector(), content);
    }

    public void injectLabels(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {
        StringInjector<IbuLabel> injector = new StringInjector<IbuLabel>(managerInjector, new CSVReaderLabel(), content) {

            @Override
            public void inject(ManagerInjector manager, List<IbuLabel> toInject) {
                manager.injectLabels(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    public void injectUsines(String content) throws TopiaNotFoundException, IOException {
        injectUsines(new ManagerInjector(), content);
    }

    public void injectUsines(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {
        StringInjector<IbuUsine> injector = new StringInjector<IbuUsine>(managerInjector, new CSVReaderUsine(), content) {

            @Override
            public void inject(ManagerInjector manager, List<IbuUsine> toInject) {
                manager.injectUsines(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    public void injectCamions(String content) throws TopiaNotFoundException, IOException {
        injectCamions(new ManagerInjector(), content);
    }

    public void injectCamions(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {
        StringInjector<IbuCamion> injector = new StringInjector<IbuCamion>(managerInjector, new CSVReaderCamion(), content) {

            @Override
            public void inject(ManagerInjector manager, List<IbuCamion> toInject) {
                manager.injectCamions(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    public void injectChauffeurs(String content) throws TopiaNotFoundException, IOException {
        injectChauffeurs(new ManagerInjector(), content);
    }

    public void injectChauffeurs(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {
        StringInjector<IbuChauffeur> injector = new StringInjector<IbuChauffeur>(managerInjector, new CSVReaderChauffeur(), content) {

            @Override
            public void inject(ManagerInjector manager, List<IbuChauffeur> toInject) {
                manager.injectChauffeurs(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    public void injectTransporteurs(String content) throws TopiaNotFoundException, IOException {
        injectTransporteurs(new ManagerInjector(), content);
    }

    public void injectTransporteurs(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {
        StringInjector<IbuTransporteur> injector = new StringInjector<IbuTransporteur>(managerInjector, new CSVReaderTransporteur(), content) {

            @Override
            public void inject(ManagerInjector manager, List<IbuTransporteur> toInject) {
                manager.injectTransporteurs(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    public void injectAutorisations(String content) throws TopiaNotFoundException, IOException {
        injectAutorisations(new ManagerInjector(), content);
    }

    public void injectAutorisations(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {

        StringInjector<IbuAutorisation> injector = new StringInjector<IbuAutorisation>(managerInjector, new CSVReaderAutorisation(), content) {

            @Override
            public void inject(ManagerInjector manager, List<IbuAutorisation> toInject) {
                manager.injectAutorisations(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    public void injectUsers(String content) throws TopiaNotFoundException, IOException {
        injectUsers(new ManagerInjector(), content);
    }

    public void injectUsers(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {
        StringInjector<IbuUser> injector = new StringInjector<IbuUser>(managerInjector, new CSVReaderUser(), content) {

            @Override
            public void inject(ManagerInjector manager, List<IbuUser> toInject) {
                manager.injectUsers(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    public void injectEvents(String content) throws TopiaNotFoundException, IOException {
        injectEvents(new ManagerInjector(), content);
    }

    public void injectEvents(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {
        StringInjector<IbuEvent> injector = new StringInjector<IbuEvent>(managerInjector, new CSVReaderEvent(), content) {

            @Override
            public void inject(ManagerInjector manager, List<IbuEvent> toInject) {
                manager.injectEvents(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    public void injectEleveurGPS(String content) throws TopiaNotFoundException, IOException {
        injectEleveurGPS(new ManagerInjector(), content);
    }

    public void injectEleveurGPS(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {
        StringInjector<IbuEleveurGPS> injector = new StringInjector<IbuEleveurGPS>(managerInjector, new CSVReaderEleveurGPS(), content) {

            @Override
            public void inject(ManagerInjector manager, List<IbuEleveurGPS> toInject) {
                manager.injectEleveurGPS(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    public void injectEleveurDuplicated(String content) throws TopiaNotFoundException, IOException {
        injectEleveurDuplicated(new ManagerInjector(), content);
    }

    public void injectEleveurDuplicated(ManagerInjector managerInjector, String content) throws TopiaNotFoundException, IOException {
        StringInjector<IbuEleveurDuplicated> injector = new StringInjector<IbuEleveurDuplicated>(managerInjector, new CSVReaderEleveurDuplicated(), content) {

            @Override
            public void inject(ManagerInjector manager, List<IbuEleveurDuplicated> toInject) {
                manager.injectEleveurDuplicated(contextMigration, toInject);
            }
        };
        injector.doInject();
    }

    protected void endInjection(MigrationContext contextMigration) {
        if (contextMigration.getErrorBaseFileName() == null) {
            contextMigration.setErrorFilePath(SandraSchedulerConfigHelper.getIbuErrorPath(SandraConfig.getConfig()));
        }

        try {
            int returnCode = 0;
            returnCode |= createFilesAndGetReturnCode(contextMigration);

            if (log.isDebugEnabled()) {
                log.debug("Return code is : " + returnCode);
            }

            SandraIbuMailer.checkMailsToSend(contextMigration, returnCode);

        } catch (Exception ex) {
            log.error("Mails are not send caused by : ", ex);
        }
    }

    protected int createFilesAndGetReturnCode(MigrationContext contextMigration) throws Exception {
        boolean hasParserError = createParserErrorFile(contextMigration);
        boolean hasInjectorError = createInjectorErrorFile(contextMigration);
        return ReturnCodeHelper.getReturnCode(hasParserError, hasInjectorError);
    }

    protected boolean createParserErrorFile(MigrationContext contextMigration) throws Exception {
        if (log.isDebugEnabled()) {
            log.debug("Checking for a parser error file to write...");
        }
        String errorFileName = contextMigration.getParserErrorFilePath();
        List<Integer> errorLineNumbers = contextMigration.getParserErrorLineNumbers();
        return createErrorFile(contextMigration, errorLineNumbers, errorFileName);
    }

    protected boolean createInjectorErrorFile(MigrationContext contextMigration) throws Exception {
        if (log.isDebugEnabled()) {
            log.debug("Checking for an injector error file to write...");
        }
        String errorFileName = contextMigration.getInjectorErrorFilePath();
        List<Integer> errorLineNumbers = contextMigration.getInjectorErrorLineNumbers();
        return createErrorFile(contextMigration, errorLineNumbers, errorFileName);
    }

    protected boolean createErrorFile(MigrationContext contextMigration, List<Integer> errorLineNumbers, String errorFileName) throws Exception {
        if (!errorLineNumbers.isEmpty() && errorFileName != null && !"".equals(errorFileName)) {
            if (log.isInfoEnabled()) {
                log.info("... writing to file: " + errorFileName);
            }
            List<String> lines = contextMigration.getLines();
            String fileContent = lines.get(0);
            for (Integer i : errorLineNumbers) {
                fileContent += lines.get(i);
            }
            File errorFile = new File(errorFileName);
            FileUtils.writeStringToFile(errorFile, fileContent);
            return true;
        } else {
            if (log.isDebugEnabled()) {
                log.debug("... Nothing to do.");
            }
            return false;
        }
    }

    protected interface Injector<E> {

        public void inject(ManagerInjector manager, List<E> toInject);

        public abstract void doInject();
    }

    protected abstract class AbstractInjector<E> implements Injector<E> {
        protected ManagerInjector manager;
        protected MigrationContext contextMigration;
        protected CSVReaderGeneric<E> reader;
        protected List<E> toInject;

        protected AbstractInjector(CSVReaderGeneric<E> reader, ManagerInjector manager) {
            this.reader = reader;
            this.contextMigration = new MigrationContext();

            this.manager = manager;
        }

        @Override
        public void doInject() {

            try {

                toInject = reader.parse(contextMigration, getReader());
                inject(manager, toInject);

            } catch (Exception ex) {
                try {
                    log.error("Injection failed : ", ex);
                    createInjectorErrorFile(contextMigration);
                } catch (Exception ex1) {
                    log.error("Creating mail error failed : ", ex1);
                }
            } finally {
                if (manager != null) {
                    try {
                        manager.finalize();
                    } catch (Throwable t) {
                        log.fatal("Cannot close manager", t);
                    }
                }
                endInjection(contextMigration);
            }
        }

        public int getNbLines() {
            return toInject.size();
        }

        public abstract Reader getReader();
    }

    protected abstract class FileInjector<E> extends AbstractInjector<E> {

        protected File file;

        protected FileInjector(CSVReaderGeneric<E> reader, File file, ManagerInjector manager) {
            super(reader, manager);
            this.file = file;
            contextMigration.setSourceFileName(file.getName());
        }

        @Override
        public Reader getReader() {
            try {
                return new FileReader(file);
            } catch (FileNotFoundException eee) {
                log.error("Failed to read file " + file.getName(), eee);
            }
            return null;
        }

    }

    protected abstract class StringInjector<E> extends AbstractInjector<E> {

        protected String content;

        protected StringInjector(ManagerInjector manager, CSVReaderGeneric<E> reader, String content) throws TopiaNotFoundException, IOException {
            super(reader, manager);
            this.content = content;
            if (contextMigration.getErrorBaseFileName() == null) {
                contextMigration.setFilePath("installEvents");
                contextMigration.setErrorFilePath("log");
            }
        }

        @Override
        public Reader getReader() {

            InputStream stream;

            stream = new ByteArrayInputStream(content.getBytes());

            return new InputStreamReader(stream);
        }
    }
}
