package com.cybelia.sandra;

import com.cybelia.sandra.entities.Societe;
import com.cybelia.sandra.entities.SocieteDAO;
import com.cybelia.sandra.ibu.IBU;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.List;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.topia.TopiaContext;
import org.nuiton.util.ApplicationConfig;
import org.quartz.CronTrigger;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SimpleTrigger;
import org.quartz.StatefulJob;
import org.quartz.Trigger;
import org.quartz.impl.StdSchedulerFactory;

import static org.nuiton.i18n.I18n._;

/**
 * @author sletellier
 */
public class SandraScheduler {

    /**
     * Logger.
     */
    public static final Log log = LogFactory.getLog(SandraScheduler.class);

    public static final String LOGGER_JOB = "loggerJob";
    public static final String CODE_SOCIETE = "codeSociete";

    public static void main(String[] args) throws Exception {

        // Initialise config
        ApplicationConfig config = SandraConfig.getConfig(args);
        I18n.init(new DefaultI18nInitializer("sandra-scheduler-i18n"), SandraSchedulerConfigHelper.getLocale(config));

        // Start all schedulers
        Scheduler scheduler = StdSchedulerFactory.getDefaultScheduler();
        for (SchedulerJobFactory.SchedulerJobClass sandraScheduler : SchedulerJobFactory.SchedulerJobClass.values()) {

            String pattern = sandraScheduler.getPattern();
            String cronExpression = SandraSchedulerConfigHelper.getCronfromProperies(config, pattern);

            if (StringUtils.isEmpty(cronExpression)) {
                log.warn("No cron found for notification : " + _(pattern));
            } else {

                CronTrigger cronTrigger = new CronTrigger(
                        pattern,
                        Scheduler.DEFAULT_GROUP,
                        cronExpression);

                Class<? extends SchedulerJobFactory.SchedulerJob> schedulerClass = sandraScheduler.getSchedulerClass();

                JobDetail jobDetail = new JobDetail(pattern, Scheduler.DEFAULT_GROUP, schedulerClass, false, true, true);

                JobDataMap jobDataMap = new JobDataMap();
                jobDataMap.put("sandraScheduler", pattern);
                jobDetail.setJobDataMap(jobDataMap);

                log.info(_(pattern) + " will start at : " + SandraSchedulerConfigHelper.getNextDatefromProperies(config, pattern));
                scheduler.scheduleJob(jobDetail, cronTrigger);
            }
        }

        // Lunch one ibu per societe
        List<Societe> societes = getAllSociete();
        for (Societe societe : societes) {
            String pattern = societe.getCode() + "-" + IbuSchedulerJob.PATTERN;
            String cronExpression = SandraSchedulerConfigHelper.getCronfromProperies(config, IbuSchedulerJob.PATTERN);

            if (cronExpression != null) {
                CronTrigger cronTrigger = new CronTrigger(
                        pattern,
                        Scheduler.DEFAULT_GROUP,
                        cronExpression);

                JobDetail jobDetail = new JobDetail(pattern, Scheduler.DEFAULT_GROUP, IbuSchedulerJob.class, false, true, true);

                JobDataMap jobDataMap = new JobDataMap();
                jobDataMap.put("sandraScheduler", pattern);
                jobDataMap.put(CODE_SOCIETE, societe.getCode());
                jobDetail.setJobDataMap(jobDataMap);

                log.info(_(pattern) + " will start at for societe " + societe.getCode() + " : " + SandraSchedulerConfigHelper.getNextDatefromProperies(config, IbuSchedulerJob.PATTERN));
                scheduler.scheduleJob(jobDetail, cronTrigger);
            }
        }

        if (log.isDebugEnabled()) {
            // Add job to log activity
            JobDetail loggerJob = new JobDetail(LOGGER_JOB, Scheduler.DEFAULT_GROUP, LogJob.class);

            // Repeat every 10 minutes
            Trigger loggerTrigger = new SimpleTrigger(LOGGER_JOB, SimpleTrigger.REPEAT_INDEFINITELY, 10 * 60 * 1000);

            // Tell quartz to schedule the job using our trigger
            scheduler.scheduleJob(loggerJob, loggerTrigger);
        }

        // Start all schedulers
        scheduler.start();
    }

    public static List<Societe> getAllSociete() {
        try {
            TopiaContext ibuRootContext = SandraSchedulerConfigHelper.getSchedulerRootContext();
            TopiaContext transaction = ibuRootContext.beginTransaction();
            SocieteDAO societeDAO = null;
            societeDAO = SandraDAOHelper.getSocieteDAO(transaction);
            return societeDAO.findAll();
        } catch (Exception eee) {
            log.error("Failed to get all societe", eee);
        }
        return null;
    }

    public static class IbuSchedulerJob implements StatefulJob {

        public static final String PATTERN = "sandra.scheduler.ibu";

        @Override
        public void execute(JobExecutionContext jobExecutionContext) throws JobExecutionException {

            String codeSociete = (String)jobExecutionContext.getJobDetail().getJobDataMap().get(CODE_SOCIETE);

            // Starting ibu
            IBU ibu = new IBU(codeSociete);
            try {
                ibu.injectIBU();
            } catch (Exception eee) {
                log.error("Failed to run ibu for societe " + codeSociete, eee);
            }
        }

    }

    public static class LogJob implements StatefulJob {

        @Override
        public void execute(JobExecutionContext jobExecutionContext) throws JobExecutionException {
            try {
                Scheduler scheduler = jobExecutionContext.getScheduler();
                String[] groups = scheduler.getJobGroupNames();
                for (String group : groups) {
                    String[] jobNames = scheduler.getJobNames(group);
                    for (String jobName : jobNames) {
                        if (!jobName.equals(LOGGER_JOB)) {
                            Trigger trigger = scheduler.getTrigger(jobName, group);
                            DateFormat dateFormat = new SimpleDateFormat("HH:mm:ss");
                            String msg = "[" + jobName + "] " + dateFormat.format(trigger.getNextFireTime());
                            int triggerState = scheduler.getTriggerState(jobName, group);
                            switch (triggerState) {
                                case Trigger.STATE_BLOCKED : msg += " is blocked"; break;
                                case Trigger.STATE_COMPLETE : msg += " is completed"; break;
                                case Trigger.STATE_ERROR : msg += " is on error"; break;
                                case Trigger.STATE_NONE : msg += " is none"; break;
                                case Trigger.STATE_NORMAL : msg += " is normal"; break;
                                case Trigger.STATE_PAUSED : msg += " is in pause"; break;
                            }
                            log.info(msg);
                        }
                    }
                }
            } catch (SchedulerException e) {
                log.info("Failled to get all jobs");
            }
        }
    }
}
