/* *##%
 * Copyright (c) 2009 poussin. All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.text.CharacterIterator;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.text.StringCharacterIterator;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * 
 * Util static method for wikitty
 * 
 * @author poussin
 * @version $Revision: 1 $
 * 
 *          Last update: $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $ by : $Author: echatellier $
 */
public class WikittyUtil {

    public static final String DEFAULT_VERSION = "0.0";

    public static final String DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyUtil.class);

    /** used to format date for solr */
    protected static TimeZone CANONICAL_TZ = TimeZone.getTimeZone("UTC");
    protected static final Locale CANONICAL_LOCALE = Locale.US;

    static final public SimpleDateFormat solrDateFormat = new SolrDateFormat();

    public static class SolrDateFormat extends SimpleDateFormat {
        public SolrDateFormat() {
            super(DATE_FORMAT, CANONICAL_LOCALE);
            setTimeZone(CANONICAL_TZ);
        }
    }

    /** All date format parser used to convert string to date */
    static final protected DateFormat[] parserDateFormats = new DateFormat[] {
            solrDateFormat, DateFormat.getInstance(),
    // TODO poussin 20090813: add other date syntax
    };

    // TODO poussin 20090902 use spring configuration to add mapping in this
    // variable
    /** contains mapping between interface and concret class that must be used */
    static public Map<Class, Class> interfaceToClass = new HashMap<Class, Class>();

    /**
     * Pattern for tag value: tag="value" or tag=value. value can contains '"'
     */
    static protected String tagValuesPatternString = "(\\w*)=(\".*?(?<!\\\\)\"|[^(\\p{Space})]+)";
    static protected Pattern tagValuesPattern = Pattern.compile(
            tagValuesPatternString, Pattern.DOTALL);
    /**
     * Field pattern parser <li>group 1: type (string) <li>group 2: name
     * (string) <li>group 3: lower bound (number) can be null <li>group 4: upper
     * bound (number) can be null, mutualy exclusif with group 5 <li>group 5:
     * upper bound (n or *) can be null <li>group 6: unique can be null <li>
     * group 7: not null can be null <li>group 8: all tag/value (toto=titi
     * tutu=tata;lala tata="truc bidulle")
     */
    static protected Pattern fieldPattern = Pattern
            .compile(
                    "\\s*(\\w+)\\s+(\\w+)(?:\\s*\\[(\\d+)-(?:(\\d+)|([\\*n]))\\])?(?:\\s+(unique))?(?:\\s+(not null))?((?:\\s+"
                            + tagValuesPatternString + ")*)", Pattern.DOTALL);

    /**
     * parse FieldType definition and return field name. All field information
     * are stored in fieldType object passed in argument
     * 
     * @param def
     *            string field definition
     * @param fieldType
     *            object used to put parsed information
     * @return field name parsed in definition
     */
    static public String parseField(String def, FieldType fieldType) {
        Matcher match = fieldPattern.matcher(def);
        if (match.matches()) {
            fieldType.type = FieldType.TYPE.parse(match.group(1));
            String name = match.group(2);
            String lower = match.group(3);
            if (lower != null) {
                fieldType.lowerBound = Integer.parseInt(lower);
            }

            String upper = match.group(4);
            if (upper != null) {
                fieldType.upperBound = Integer.parseInt(upper);
            }
            String noupper = match.group(5);
            if (noupper != null) {
                fieldType.upperBound = FieldType.NOLIMIT;
            }
            String uniqueString = match.group(6);
            fieldType.unique = uniqueString != null;

            String notNullString = match.group(7);
            fieldType.notNull = notNullString != null;

            String tagValues = match.group(8);
            Map<String, String> tagValuesMap = tagValuesToMap(tagValues);
            fieldType.setTagValues(tagValuesMap);

            return name;
        } else {
            throw new WikittyException(String.format(
                    "Bad FieldType definition '%s'", def));
        }
    }


    /**
     * Serialize tagValues to string
     * 
     * @param tagValues tagValues as map
     * @return string represent tagValues
     */
    public static String tagValuesToString(Map<String, String> tagValues) {
        String result = "";
        if(tagValues != null) {
            for (String tag : tagValues.keySet()) {
                String value = tagValues.get(tag);
                // replace " in string with \"
                value = value.replaceAll("\"", "\\\\\"");
                // quote value with "..."
                result += " " + tag + "=\"" + tagValues.get(tag) +"\"";
            }
        }
        return result;
    }

    /**
     * Deserialize tagValues to map
     *
     * @param tagValues tagValues as string
     * @return map represent tagValues
     */
    public static Map<String, String> tagValuesToMap(String tagValues) {
        Map<String, String> result = new HashMap<String, String>();
        if (tagValues != null) {
            Matcher matchTagValues = tagValuesPattern.matcher(tagValues);
            while (matchTagValues.find()) {
                String tag = matchTagValues.group(1);
                String value = matchTagValues.group(2);
                if (value.startsWith("\"") && value.endsWith("\"")) {
                    // delete start and end "
                    value = value.substring(1, value.length() - 1);
                    // if value is between ", then inners " are quoted
                    value = value.replaceAll("\\\\\"", "\"");
                }
                result.put(tag, value);
            }
        }
        return result;
    }

    /**
     * Create map from string representation
     * 
     * ex: "String name", "Wikitty children[0-*]"
     * 
     * @param definitions
     * @return
     */
    public static LinkedHashMap<String, FieldType> buildFieldMapExtension(
            String... definitions) {
        LinkedHashMap<String, FieldType> result = new LinkedHashMap<String, FieldType>();
        for (String def : definitions) {
            FieldType fieldType = new FieldType();
            String name = WikittyUtil.parseField(def, fieldType);
            log.debug("parse " + def + " => " + fieldType.toDefinition(name));
            result.put(name, fieldType);
        }
        return result;
    }

    /**
     * if version if null return 0 else version If version is not in format
     * <major>.<minor>, ".0" is added to the version
     * 
     * @param version
     * @return the normalized version
     */
    public static String normalizeVersion(String version) {
        if (version == null || "".equals(version)) {
            version = "0";
        }
        if (version.indexOf(".") == -1) {
            version += ".0";
        }
        return version;
    }

    /**
     * return true if v1 and v2 are egals 1.2.0 et 1.2 ne sont pas egaux
     */
    public static boolean versionEquals(String v1, String v2) {
        if (v1 == null || v2 == null) {
            return false;
        }
        return normalizeVersion(v1).equals(normalizeVersion(v2));
    }

    /**
     * return true if v1 greater than v2
     * 
     * @param v1
     * @param v2
     * @return
     */
    public static boolean versionGreaterThan(String v1, String v2) {
        if (v1 != null && v2 == null) {
            return true;
        }
        if (v1 == null) {
            return false;
        }
        String[] v1s = normalizeVersion(v1).split("\\.");
        String[] v2s = normalizeVersion(v2).split("\\.");
        int minlen = Math.min(v1s.length, v2s.length);
        for (int i = 0; i < minlen; i++) {
            if (!v1s[i].equals(v2s[i])) {
                return Integer.parseInt(v1s[i]) > Integer.parseInt(v2s[i]);
            }
        }
        // si on est ici c que tout les nombres sont v1[i] = v2[i]
        return v1s.length > v2s.length;
    }

    /**
     * increment minor version.
     * 
     * @param version
     *            version as 3.1 where 1 is minor and 3 major
     * @return incremented minor number (3.1 -> 3.2)
     */
    static public String incrementMinorRevision(String v) {
        String result;

        if (v == null || "".equals(v)) {
            result = "0.1";
        } else {
            v = v.trim();
            String[] mm = v.split("\\.");
            if (mm.length == 1) {
                result = v + ".1";
            } else {
                int i = Integer.parseInt(mm[1]) + 1;
                result = mm[0] + "." + i;
            }
        }
        return result;
    }

    /**
     * increment major version.
     * 
     * @param version
     *            version as 3.2 where 2 is minor and 3 major
     * @return incremented major number and reset minor number (3.2 -> 4.0)
     */
    static public String incrementMajorRevision(String v) {
        String result;

        if (v == null || "".equals(v)) {
            result = "1.0";
        } else {
            v = v.trim();
            String[] mm = v.split("\\.");
            int i = Integer.parseInt(mm[0]) + 1;
            result = i + ".0";
        }
        return result;
    }

    // /**
    // *
    // * @param value null and empty string are casted to '0' int value.
    // * @throws WikittyException on NumberFormatException or if value object
    // can't be casted to int.
    // */
    // static public int toInt(Object value) throws WikittyException {
    // int result = 0;
    // if (value == null || value.equals("") ) {
    // result = 0; // default to 0
    // } else if (value instanceof Number) {
    // result = ((Number) value).intValue();
    // } else {
    // // try to convert to int
    // try {
    // result = Integer.parseInt(value.toString());
    // } catch (NumberFormatException eee) {
    // throw new WikittyException(String.format(
    // "Can't convert value '%s' to int", getClass(value)), eee);
    // }
    // }
    // return result;
    // }
    //
    // static public float toFloat(Object value) throws WikittyException {
    // float result = 0;
    // if (value == null) {
    // result = 0; // default to 0
    // } else if (value instanceof Number) {
    // result = ((Number) value).floatValue();
    // } else {
    // // try to convert to float
    // try {
    // result = Float.parseFloat(value.toString());
    // } catch (NumberFormatException eee) {
    // throw new WikittyException(String.format(
    // "Can't convert value '%s' to float", getClass(value)), eee);
    // }
    // }
    // return result;
    // }

    /**
     * 
     * @param value
     *            null and empty string are casted to '0' value.
     * @throws WikittyException
     *             on NumberFormatException or if value object can't be casted
     *             to number.
     */
    static public BigDecimal toBigDecimal(Object value) {
        BigDecimal result = null;
        if (value == null) {
            result = new BigDecimal(0); // default to 0
        } else if (value instanceof BigDecimal) {
            result = (BigDecimal) value;
        } else {
            try {
                result = new BigDecimal(value.toString());
            } catch (NumberFormatException eee) {
                throw new WikittyException(
                        String.format("Can't convert value '%s' to numeric",
                                getClass(value)), eee);
            }
        }
        return result;
    }

    /**
     * Convert object to boolean: - null => false - 0 => false - numeric => true
     * - object.toString() == false => false - other => true
     * 
     * @param value
     * @return
     */
    static public boolean toBoolean(Object value) {
        boolean result = false;
        if (value != null) {
            if (value instanceof Boolean) {
                result = (Boolean) value;
            } else if (value instanceof Number) {
                result = !((Number) value).equals(0);
            } else {
                // try to convert to Boolean
                result = !"false".equalsIgnoreCase(value.toString());
            }
        }
        return result;
    }

    static public String toString(Object value) {
        String result = null;
        if (value != null) {
            if (value instanceof String) {
                result = (String) value;
            } else if (value instanceof Wikitty) {
                result = ((Wikitty) value).getId();
            } else if (value instanceof BusinessEntity) {
                result = ((BusinessEntity) value).getWikittyId();
            } else if (value instanceof Date) {
                result = solrDateFormat.format((Date) value);
            } else {
                // try to convert to String
                result = value.toString();
            }
        }
        return result;
    }

    static public Date toDate(Object value) {
        Date result = null;
        if (value != null) {
            if (value instanceof Date) {
                result = (Date) value;
            } else {
                // try to convert to Date
                try {
                    result = solrDateFormat.parse(value.toString());
                } catch (ParseException eee) {
                    log.debug("Can't parse date, i try with next parser", eee);
                }

                if (result == null) {
                    throw new WikittyException(String.format(
                            "Can't convert value '%s' of type '%s' to Date",
                            value, getClass(value)));
                }
            }
        }
        return result;
    }

    /**
     * return wikitty id and not wikitty objet because this method can be call
     * on server or client side and it's better to keep conversion between id
     * and objet to the caller
     * 
     * @param value
     * @return id of wikitty object or null
     * @throws org.sharengo.wikitty.WikittyException
     */
    static public String toWikitty(Object value) {
        String result = null;
        if (value != null) {
            if (value instanceof String) {
                result = (String) value;
            } else if (value instanceof Wikitty) {
                result = ((Wikitty) value).getId();
            } else if (value instanceof BusinessEntity) {
                result = ((BusinessEntity) value).getWikittyId();
            } else {
                // try to convert to String
                result = value.toString();
            }
        }
        return result;
    }

    /**
     * 
     * @param <E>
     * @param clazz
     * @return unmodifiable list
     */
    static public <E> List<E> toList(Object value, Class<E> clazz) {
        try {
            List<E> result = (List<E>) value;
            if (result != null) {
                result = Collections.unmodifiableList(result);
            }
            return result;
        } catch (Exception eee) {
            throw new WikittyException(String.format(
                    "Can't convert value '%s' to list", getClass(value)), eee);
        }
    }

    /**
     * Convert object o for indexation
     * 
     * @param field
     *            field description
     * @param o
     *            field value
     * @return solr representation
     */
    static public String toString(FieldType field, Object o) {
        String result = null;
        if (o != null) {
            switch (field.getType()) {
            case DATE:
                // Date date = (Date)o;
                result = (o instanceof String) ? (String) o
                        : WikittyUtil.solrDateFormat.format((Date) o);
                break;
            default:
                result = WikittyUtil.toString(o);
                break;
            }
        }
        return result;
    }

    
    /**
     * convert string field representation to correct value type
     * 
     * @param field
     *            field description
     * @param s
     *            string value
     * @return object in type of field
     */
    static public Object fromString(FieldType field, String s) {
        Object result = null;
        switch (field.getType()) {
        case BOOLEAN:
            result = WikittyUtil.toBoolean(s);
            break;
        case DATE:
            result = WikittyUtil.toDate(s);
            break;
        case NUMERIC:
            result = WikittyUtil.toBigDecimal(s);
            break;
        default:
            result = s;
            break;
        }
        return result;
    }

    /**
     * return class of argument, if argument is null, return null
     * 
     * @param value
     * @return class of value or null
     */
    static public Class getClass(Object value) {
        Class result = null;
        if (value != null) {
            result = value.getClass();
        }
        return result;
    }

    /**
     * Create new instance of WikittyDto without Wikitty object passed in
     * argument. If arguement is Interface try to add 'Impl' to find
     * instanciable class.
     * <p>
     * clazz parameter must be child of WikittyDto or business interface
     * 
     * @param clazz
     *            class of the new instance
     * @return
     */
    static public <E extends BusinessEntity> E newInstance(Class<E> clazz) {
        try {
            Class clazzInstanciable = clazz;
            if (clazzInstanciable.isInterface()) {
                log
                        .debug(String
                                .format(
                                        "Argument '%s' is interface looking for implementation",
                                        clazzInstanciable.getName()));
                // looking for implementation of this interface
                if (interfaceToClass.containsKey(clazz)) {
                    clazzInstanciable = interfaceToClass.get(clazz);
                } else {
                    // default use xxxImpl.class
                    clazzInstanciable = clazz.forName(clazz.getName() + "Impl");
                }
            }

            if (!BusinessEntityWikitty.class
                    .isAssignableFrom(clazzInstanciable)) {
                throw new WikittyException(String.format(
                        "Your class '%s' don't extends WikittyDto", clazz
                                .getName()));
            }

            E result = (E) clazzInstanciable.newInstance();
            return result;

        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    /**
     * Create new instance of WikittyDto with Wikitty object passed in argument.
     * If arguement is Interface try to add 'Impl' to find instanciable class.
     * <p>
     * clazz parameter must be child of WikittyDto or business interface
     * 
     * @param clazz
     *            class of the new instance
     * @param w
     *            wikitty object to use internaly for in new instance
     * @return
     */
    // E extends BeanDto to permit business interface as parameter
    static public <E extends BusinessEntity> E newInstance(
            WikittyService wikittyService, Class<E> clazz, Wikitty w) {
        try {
            Class clazzInstanciable = clazz;
            if (clazzInstanciable.isInterface()) {
                log
                        .debug(String
                                .format(
                                        "Argument '%s' is interface looking for implementation",
                                        clazzInstanciable.getName()));
                // looking for implementation of this interface
                if (interfaceToClass.containsKey(clazz)) {
                    clazzInstanciable = interfaceToClass.get(clazz);
                } else {
                    // default use xxxImpl.class
                    clazzInstanciable = clazz.forName(clazz.getName() + "Impl");
                }
            }

            if (!BusinessEntityWikitty.class
                    .isAssignableFrom(clazzInstanciable)) {
                throw new WikittyException(String.format(
                        "Your class '%s' don't extends WikittyDto", clazz
                                .getName()));
            }

            E result = null;
            if (w != null) {
                try {
                    // try to find constructor with wikitty argument
                    Constructor cons = clazzInstanciable
                            .getConstructor(Wikitty.class);
                    Object[] parms = { null };
                    result = (E) cons.newInstance(parms);

                    BusinessEntityWikitty bean = (BusinessEntityWikitty) result;
                    checkExtensionVersion(wikittyService, w, bean);
                    bean.setWikitty(w);

                } catch (NoSuchMethodException noerror) {
                    log.debug(String.format(
                            "Can't find constructor with wikitty arguement in '%s',"
                                    + "we try with setWikitty method",
                            clazzInstanciable.getName()), noerror);
                }
            }

            return result;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    /**
     * Check extension default, i.e. if bean contain a extension with a great
     * version as in wikitty. In this case the extension is stored in last
     * version and wikitty it is restored again. The upgrade data is during the
     * restoration.
     * 
     * @param wikittyService
     * @param wikitty
     * @param entity
     * @return
     */
    static public Wikitty checkExtensionVersion(WikittyService wikittyService,
            Wikitty wikitty, BusinessEntityWikitty entity) {

        Wikitty result = wikitty;
        boolean upgradeData = false;

        Collection<WikittyExtension> extensions = entity.getStaticExtensions();
        for (WikittyExtension ext : extensions) {

            String extName = ext.getName();
            if (wikitty.hasExtension(extName)) {

                WikittyExtension oldExt = wikitty.getExtension(extName);
                String newVersion = ext.getVersion();
                String oldVersion = oldExt.getVersion();

                if (WikittyUtil.versionGreaterThan(newVersion, oldVersion)) {
                    wikittyService.storeExtension(Arrays.asList(ext));
                    upgradeData = true;
                }
            }
        }

        if (upgradeData) {
            String wikittyId = wikitty.getId();
            result = wikittyService.restore(wikittyId);
        }

        return result;
    }

    /**
     * Try to cast obj to class passed in arguement
     * 
     * @param obj
     *            object to cast
     * @param clazz
     *            new type of object
     * @return the same object but casted to class wanted, except for primitif
     *         where is new object if obj BigDecimal
     */
    public static <E> E cast(Object obj, Class<E> clazz) {
        E result = null;
        if (obj != null) {
            if (clazz.isAssignableFrom(obj.getClass())) {
                result = clazz.cast(obj);
            } else if (obj instanceof BigDecimal) {
                BigDecimal bd = (BigDecimal) obj;
                if (clazz == int.class || clazz == Integer.class) {
                    result = clazz.cast(bd.intValue());
                } else if (clazz == float.class || clazz == Float.class) {
                    result = clazz.cast(bd.floatValue());
                }
            }
            if (result == null) {
                throw new WikittyException("Unable to cast from '"
                        + obj.getClass().getName() + "' to " + clazz.getName());
            }
        }
        return result;
    }

    public static Wikitty beanToWikitty(BusinessEntity bean) {
        Wikitty result;
        if (bean instanceof BusinessEntityWikitty) {
            BusinessEntityWikitty b = (BusinessEntityWikitty) bean;
            result = b.getWikitty();
        } else if (bean instanceof BusinessEntityBean) {
            BusinessEntityBean b = (BusinessEntityBean) bean;
            result = WikittyUtil.beanToWikitty(b);
        } else {
            throw new IllegalArgumentException(String.format(
                    "This BusinessEntity implementation is not supported: %s",
                    bean.getClass().getName()));
        }
        return result;
    }

    public static Wikitty beanToWikitty(BusinessEntityBean bean) {
        try {
            // TODO poussin 20090910 for now, we force version change, but it's
            // better if we modify version in bean we field is set
            Wikitty result = new Wikitty(bean.getWikittyId());
            // prevent BusinessEntity with null id (during store call for
            // creation)
            bean.id = result.id;
            result.version = bean.getWikittyVersion();

            // add execution defined extension in wikitty
            for (String extName : bean.getExtensionNames()) {
                for (String fieldName : bean.getExtensionFields(extName)) {
                    Object value = bean.getField(extName, fieldName);
                    result.setField(extName, fieldName, value);
                }
            }

            // add development time defined extension in wikitty
            Field[] fields = bean.getClass().getFields();
            for (Field field : fields) {
                Object value = field.get(bean);
                String fqfieldName = field.getName();
                // fieldName use $ as separator between extension name and field
                // name
                fqfieldName = fqfieldName.replace('$', '.');
                result.setFqField(fqfieldName, value);
            }
            return result;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    public static <E extends BusinessEntityBean, F extends BusinessEntityWikitty> E wikittyToBean(
            Class<E> clazz, F dto) {
        E result = wikittyToBean(clazz, dto.getWikitty());
        return result;
    }

    /**
     * Convert WikittyDto (dto that encapsulate Wikitty) to BeanDto (dto without
     * internaly wikitty)
     * 
     * @param clazz
     *            target object clazz
     * @param w
     *            source object
     * @return
     */
    public static <E extends BusinessEntityBean> E wikittyToBean(
            Class<E> clazz, Wikitty w) {
        try {
            E result = clazz.newInstance();
            result.id = w.id;
            result.version = w.version;
            result.extensions = new LinkedHashMap<String, WikittyExtension>(
                    w.extensions);

            Field[] fields = clazz.getFields();
            Map<String, Field> allFields = new HashMap<String, Field>();
            for (Field field : fields) {
                allFields.put(field.getName(), field);
            }

            for (String extName : result.extensions.keySet()) {
                WikittyExtension ext = result.extensions.get(extName);
                for (String fieldName : ext.getFieldNames()) {
                    String key = extName + "$" + fieldName;
                    Field field = allFields.get(key);
                    if (field != null) {
                        Class type = field.getType();
                        if (Integer.TYPE.isAssignableFrom(type)) {
                            int value = w.getFieldAsInt(extName, fieldName);
                            field.setInt(result, value);
                        } else if (Boolean.TYPE.isAssignableFrom(type)) {
                            boolean value = w.getFieldAsBoolean(extName,
                                    fieldName);
                            field.setBoolean(result, value);
                        } else if (Date.class.isAssignableFrom(type)) {
                            Date value = w.getFieldAsDate(extName, fieldName);
                            field.set(result, value);
                        } else if (Double.TYPE.isAssignableFrom(type)) {
                            double value = w.getFieldAsDouble(extName,
                                    fieldName);
                            field.setDouble(result, value);
                        } else if (Float.TYPE.isAssignableFrom(type)) {
                            float value = w.getFieldAsFloat(extName, fieldName);
                            field.setFloat(result, value);
                        } else if (List.class.isAssignableFrom(type)) {
                            List value = w.getFieldAsList(extName, fieldName,
                                    clazz);
                            field.set(result, value);
                        } else if (Long.TYPE.isAssignableFrom(type)) {
                            long value = w.getFieldAsLong(extName, fieldName);
                            field.setLong(result, value);
                        } else if (Set.class.isAssignableFrom(type)) {
                            Set value = w.getFieldAsSet(extName, fieldName,
                                    clazz);
                            field.set(result, value);
                        } else if (String.class.isAssignableFrom(type)) {
                            String value = w.getFieldAsString(extName,
                                    fieldName);
                            field.set(result, value);
                        } else {
                            throw new WikittyException(String.format(
                                    "Can't convert field for '%s' because"
                                            + " unkonw field type '%s'", clazz,
                                    field));
                        }
                    } else {
                        Object value = w.getFieldAsObject(extName, fieldName);
                        result.setField(extName, fieldName, value);
                    }
                }
            }

            return result;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    // private static BASE64Encoder enc = new BASE64Encoder();
    public static String genUID() {
        return UUID.randomUUID().toString();
        /*
         * we can gain 10 chars per ID on applying a base64 on the UID. long
         * mostSignificant = uid.getMostSignificantBits(); long leastSignificant
         * = uid.getLeastSignificantBits(); long current = mostSignificant;
         * byte[] b = new byte[16]; for ( int i = 0; i < 16; i++ ) { b[i] =
         * (byte) (current & 0xff); current = current >> 8; if ( i == 7 )
         * current = leastSignificant; } return enc.encode(b);
         */
    }
}
