/* *##%
 * Copyright (c) 2009 poussin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.AbstractMap.SimpleEntry;
import java.util.Map.Entry;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.LabelImpl;
import org.sharengo.wikitty.TreeNodeImpl;
import org.sharengo.wikitty.search.Search;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserFactory;

/**
 * Abstract class that new implementation must extends.
 * New implementation only have three method to implement:
 * <li>getSearchEngin
 * <li>getExtensionStorage
 * <li>getWikittyStorage
 *
 * @author poussin
 * @version $Revision: 27 $
 *
 * Last update: $Date: 2010-04-28 17:56:01 +0200 (mer., 28 avril 2010) $
 * by : $Author: bpoussin $
 */
public abstract class AbstractWikittyService implements WikittyService {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(AbstractWikittyService.class);

    // FIXME poussin 20090902 next 3 variables must be read from configuration file
    /** number of thread used to import/export task */
    protected int MAX_IMPORT_EXPORT_THREAD = 1;
    /** directory path where export asynchronous file are stored */
    protected String EXPORT_DIRECTORY = "/tmp/";
    /** url used by client to retrieve export file when job is ended */
    protected String EXPORT_URL = "file:///tmp/";

    /** Executor that do import export task */
    protected ExecutorService importExportExecutor =
            // TODO poussin 20090902 do thread number configurable
            Executors.newFixedThreadPool(MAX_IMPORT_EXPORT_THREAD);
    /** contains all import or export task, key is job id send to client */
    protected Map<String, Future<String>> importExportTask =
            new HashMap<String, Future<String>>();

    /** Default migration use to migrate a wikitty in last extension version */
    protected WikittyExtensionMigration defaultExtensionMigration =
            new WikittyExtensionMigrationRename();

    abstract protected WikittySearchEngin getSearchEngin();
    abstract protected WikittyExtensionStorage getExtensionStorage();
    abstract protected WikittyStorage getWikittyStorage();

    protected UpdateResponse store(WikittyTransaction transaction,
            Collection<Wikitty> wikitties, boolean disableAutoVersionIncrement) {
        // update/store extension if necessary
        Set<WikittyExtension> allExtensions = new HashSet<WikittyExtension>();
        for (Wikitty w : wikitties) {
            // collect all extensions used by all wikitties
            allExtensions.addAll(w.getExtensions());
        }

        // try to commit command
        UpdateResponse extUpdate = getExtensionStorage().store(transaction, allExtensions);
        UpdateResponse wikUpdate = getWikittyStorage().store(transaction, wikitties, disableAutoVersionIncrement);
        UpdateResponse indexUpdate = getSearchEngin().store(transaction, wikitties);

        UpdateResponse result = new UpdateResponse();
        // prepare update client response
        result.add(extUpdate);
        result.add(wikUpdate);
        result.add(indexUpdate);

        return result;
    }

    /**
     * Store and index wikitty object
     * @param wikitty
     */
    @Override
    public UpdateResponse store(Wikitty wikitty) {
        if (wikitty != null) {
            WikittyTransaction transaction = new WikittyTransaction();
            try {
                transaction.begin();

                List<Wikitty> wikitties = Arrays.asList(wikitty);
                UpdateResponse result = store(transaction, wikitties, false);
                
                transaction.commit();
                return result;
            } catch (Exception eee) {
                transaction.rollback();
                throw new WikittyException(eee);
            }
        } else {
            throw new WikittyException("You can't store null wikitty object");
        }
    }

    /**
     * Store and index wikitties object
     * @param wikitty
     */
    @Override
    public UpdateResponse store(Collection<Wikitty> wikitties) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            UpdateResponse result = store(transaction, wikitties, false);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    /**
     * Store and index wikitties object
     * @param wikitty
     */
    @Override
    public UpdateResponse store(Collection<Wikitty> wikitties, boolean disableAutoVersionIncrement) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();
            
            UpdateResponse result = store(transaction, wikitties, disableAutoVersionIncrement);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public List<String> getAllExtensionIds() {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            List<String> result = getExtensionStorage().getAllExtensionIds(transaction);
            
            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public List<String> getAllExtensionsRequires(String extensionName) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            List<String> result = getExtensionStorage()
                    .getAllExtensionsRequires(transaction, extensionName);
            
            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }
    
    protected UpdateResponse storeExtension(WikittyTransaction transaction, Collection<WikittyExtension> exts) {
        UpdateResponse result = getExtensionStorage().store(transaction, exts);
        return result;
    }

    /**
     * Save just one extension
     * @param ext
     * @throws java.io.IOException
     */
    @Override
    public UpdateResponse storeExtension(Collection<WikittyExtension> exts) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            UpdateResponse result = storeExtension(transaction, exts);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public UpdateResponse storeExtension(WikittyExtension ext) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            UpdateResponse result = storeExtension(transaction, Arrays.asList(ext));

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    protected WikittyExtension restoreExtension(WikittyTransaction transaction, String id) {
        //split the id to ensure that version is normalized
        String name = WikittyExtension.computeName(id);
        String version = WikittyExtension.computeVersion(id);

        WikittyExtension result = getExtensionStorage().restore(transaction, name, version);
        return result;
    }
    /**
     * Load extension from id. Id is 'name[version]'
     * @param id
     * @return
     */
    @Override
    public WikittyExtension restoreExtension(String id) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            WikittyExtension result = restoreExtension(transaction, id);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }

    }

    protected WikittyExtension restoreExtensionLastVersion(WikittyTransaction transaction, String name) {
        String version = getExtensionStorage().getLastVersion(transaction, name);
        if(version == null) {
            return null;
        }

        WikittyExtension result = getExtensionStorage().restore(transaction, name, version);
        return result;
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(String name) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            WikittyExtension result = restoreExtensionLastVersion(transaction, name);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    protected Wikitty restore(WikittyTransaction transaction, String id) {
        if (!getWikittyStorage().exists(transaction, id)) {
            // object doesn't exist, we return null
            return null;
        }

        if (getWikittyStorage().isDeleted(transaction, id)) {
            // object deleted, we return null
            return null;
        }
        Wikitty result = getWikittyStorage().restore(transaction, id);
        if(result != null) {
            result = upgradeData(transaction, result);
        }
        return result;
    }

    protected List<Wikitty> restore(WikittyTransaction transaction, List<String> ids) {
        List<Wikitty> result = new ArrayList<Wikitty>();
        for(String id : ids) {
            Wikitty w = restore(transaction, id);
            if (w != null) {
                result.add(w);
            }
        }
        return result;
    }

    @Override
    public List<Wikitty> restore(List<String> ids) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            List<Wikitty> result = restore(transaction, ids);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public Wikitty restore(String id) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty result = restore(transaction, id);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    protected Wikitty upgradeData(WikittyTransaction transaction, Wikitty wikitty) {
        Wikitty result = wikitty;

        Collection<WikittyExtension> extensions = wikitty.getExtensions();
        for (WikittyExtension extension : extensions) {
            String extensionName = extension.getName();
            log.debug("extensionName="  + extensionName);
            
            WikittyExtension currentExtension = extension;
            String currentExtensionVersion = currentExtension.getVersion();

            WikittyExtension lastExtension = restoreExtensionLastVersion(transaction, extensionName);
            String lastExtensionVersion = lastExtension.getVersion();
            log.debug("lastExtensionVersion="  + lastExtensionVersion);

            WikittyExtensionMigration migration = WikittyExtensionMigration.migrationRegistry.get(extensionName);
            if (migration == null) {
                migration = defaultExtensionMigration;
            }

            // Loop on between extension in wikitty and last version
            while(WikittyUtil.versionGreaterThan(lastExtensionVersion, currentExtensionVersion)) {

                // Get extension after the current version
                String nextExtensionVersion = WikittyUtil.incrementMajorRevision(currentExtensionVersion);
                String nextExtensionId = WikittyExtension.computeId(extensionName, nextExtensionVersion);
                WikittyExtension nextExtension = restoreExtension(transaction, nextExtensionId);

                log.debug("currentExtensionVersion="  + currentExtensionVersion);
                log.debug("nextExtensionVersion="  + nextExtensionVersion);

                // Test if extension is never use in this version
                if(nextExtension != null) {
                    result = migration.migrate(this, transaction, result, currentExtension, nextExtension);
                    currentExtension = nextExtension;
                }

                // Follow
                currentExtensionVersion = nextExtensionVersion;
            }
        }
        
        return result;
    }

    protected void delete(WikittyTransaction transaction, Collection<String> ids) throws WikittyException {
        // work only on valid id
        Collection<Wikitty> storedWikitties = new ArrayList<Wikitty>();
        List<String> idList = new LinkedList<String>(ids);
        for (Iterator<String> i = idList.iterator(); i.hasNext();) {
            String id = i.next();
            // test if wikitty exists
            if (!getWikittyStorage().exists(transaction, id)) {
                // don't exist, remove this id in id list
                i.remove();
            }
            if (getWikittyStorage().isDeleted(transaction, id)) {
                // already deleted, remove this id in id list
                i.remove();
            }

            // Store node with have deleted node as parent
            Criteria criteria = Search.query().eq(TreeNode.FQ_FIELD_PARENT, id).criteria();
            List<Wikitty> wikittyNodes = findAllByCriteria(transaction, criteria).getAll();
            for (Wikitty wikittyNode : wikittyNodes) {
                String wikittyNodeId = wikittyNode.getId();
                if(!ids.contains(wikittyNodeId)) {
                    TreeNode treeNode = new TreeNodeImpl(wikittyNode);
                    treeNode.setParent(null);
                    storedWikitties.add(wikittyNode);
                }
            }

            // Store node with have deleted child
            criteria = Search.query().eq(TreeNode.FQ_FIELD_CHILDREN, id).criteria();
            wikittyNodes = findAllByCriteria(transaction, criteria).getAll();
            for (Wikitty wikittyNode : wikittyNodes) {
                String wikittyNodeId = wikittyNode.getId();
                if(!ids.contains(wikittyNodeId)) {
                    TreeNode treeNode = new TreeNodeImpl(wikittyNode);
                    treeNode.removeChildren(id);
                    storedWikitties.add(wikittyNode);
                }
            }
        }

        getWikittyStorage().delete(transaction, ids);
        getSearchEngin().delete(transaction, ids);

        store(transaction, storedWikitties, false);
    }

    @Override
    public void delete(String id) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            List<String> ids = Arrays.asList(id);
            delete(transaction, ids);

            transaction.commit();
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public void delete(Collection<String> ids){
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();
            
            delete(transaction, ids);

            transaction.commit();
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    /**
     * Use with caution : It will delete ALL indexes from search engine !
     * This operation should be disabled in production environment.
     */
    @Override
    public void clear() {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            getSearchEngin().clear(transaction);
            getWikittyStorage().clear(transaction);
            getExtensionStorage().clear(transaction);

            transaction.commit();
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    /**
     * Assume that this PagedResult contains wikitty id as result and
     * return new PagedResult with Wikitty instance
     */
    protected PagedResult<Wikitty> findAllByCriteria(WikittyTransaction transaction, Criteria criteria) {
        PagedResult<String> resultIds = getSearchEngin().findAllByCriteria(transaction, criteria);
        List<String> ids = resultIds.getAll();
        List<Wikitty> wikitties = restore(transaction, ids);
        PagedResult<Wikitty> result = new PagedResult<Wikitty>(
                resultIds.getFirstIndice(),
                resultIds.getNumFound(),
                resultIds.getQueryString(),
                resultIds.getFacets(),
                wikitties);
        return result;
    }
    
    @Override
    public PagedResult<Wikitty> findAllByCriteria(Criteria criteria) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            PagedResult<Wikitty> result = findAllByCriteria(transaction, criteria);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    protected Wikitty findByCriteria(WikittyTransaction transaction, Criteria criteria) {
        criteria.setFirstIndex(0).setEndIndex(1);
        PagedResult<Wikitty> pages = findAllByCriteria(transaction, criteria);

        Wikitty result = null;
        if (pages.size() > 0) {
            result = pages.getFirst();
        }

        return result;
    }

    @Override
    public Wikitty findByCriteria(Criteria criteria) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty result = findByCriteria(transaction, criteria);
            
            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }
    
    @Override
    public void addLabel(String wikittyId, String label) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty w = restore(transaction, wikittyId);
            w.addExtension(LabelImpl.extensions);
            LabelImpl l = new LabelImpl(w);
            l.addLabels(label);
            store(transaction, Arrays.asList(w), false);
            
            transaction.commit();
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    
    @Override
    public PagedResult<Wikitty> findAllByLabel(String label, int firstIndex, int endIndex) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            LabelImpl l = new LabelImpl();
            l.addLabels(label);
            Criteria criteria = Search.query(l.getWikitty()).criteria()
                    .setFirstIndex(firstIndex).setEndIndex(endIndex);
            PagedResult<Wikitty> result = findAllByCriteria(transaction, criteria);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    
    @Override
    public Wikitty findByLabel(String label) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            LabelImpl l = new LabelImpl();
            l.addLabels(label);
            Criteria criteria = Search.query(l.getWikitty()).criteria();
            Wikitty result = findByCriteria(transaction, criteria);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    
    @Override
    public Set<String> findAllAppliedLabels(String wikittyId) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty w = restore(transaction, wikittyId);
            LabelImpl l = new LabelImpl(w);
            Set<String> result = l.getLabels();

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    
    protected Tree restoreTree(WikittyTransaction transaction, String wikittyId) {
        Wikitty w = restore(transaction, wikittyId);
        if(w == null) {
            return null;
        }

        if ( !w.hasExtension(TreeNode.EXT_TREENODE) ) {
            throw new WikittyException(String.format(
                    "Wikitty '%s' do not handle extension %s",
                    wikittyId, TreeNode.EXT_TREENODE ));
        }
        Tree tree = new Tree();
        TreeNode node = AbstractWikittyService.toBean(new TreeNodeImpl(w));
        tree.setNode(node);

        TreeNodeImpl exempleNode = new TreeNodeImpl();
        exempleNode.setParent(wikittyId);

        Criteria criteria = Search.query(exempleNode.getWikitty()).criteria()
                .setFirstIndex(0).setEndIndex(Criteria.ALL_ELEMENTS);
        PagedResult<Wikitty> childNodes = findAllByCriteria(transaction, criteria);
        for( Wikitty childNode : childNodes.getAll() ) {
            tree.addChild(restoreTree(transaction, childNode.getId()));
        }

        return tree;
    }

    @Override
    public Tree restoreTree(String wikittyId) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();
            
            Tree tree = restoreTree(transaction, wikittyId);

            transaction.commit();
            return tree;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public Map.Entry<TreeNode, Integer> restoreNode(String wikittyId, Criteria filter) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty w = restore(transaction, wikittyId);
            if(w == null) {
                transaction.commit();
                return null;
            }

            if ( !w.hasExtension(TreeNode.EXT_TREENODE) ) {
                throw new WikittyException(String.format(
                        "Wikitty '%s' do not handle extension %s",
                        wikittyId, TreeNode.EXT_TREENODE ));
            }

            TreeNode node = AbstractWikittyService.toBean(new TreeNodeImpl(w));
            Integer count = getSearchEngin().findNodeCount(transaction, w, filter);

            HashMap.SimpleEntry<TreeNode, Integer> result =
                    new SimpleEntry<TreeNode, Integer>(node, count);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public Map<TreeNode, Integer> restoreChildren(String wikittyId, Criteria filter) {
        WikittyTransaction transaction = new WikittyTransaction();
        try {
            transaction.begin();

            Wikitty w = restore(transaction, wikittyId);
            if(w == null) {
                transaction.commit();
                return null;
            }

            if ( !w.hasExtension(TreeNode.EXT_TREENODE) ) {
                throw new WikittyException(String.format(
                        "Wikitty '%s' do not handle extension %s",
                        wikittyId, TreeNode.EXT_TREENODE ));
            }

            Map<TreeNode, Integer> result = new LinkedHashMap<TreeNode, Integer>();

            Map<String, Integer> search = getSearchEngin().findAllChildrenCount(transaction, w, filter);
            Set<Entry<String, Integer>> children = search.entrySet();
            for (Entry<String, Integer> child : children) {
                Integer count = child.getValue();

                String id = child.getKey();
                Wikitty wikitty = restore(transaction, id);
                TreeNode node = AbstractWikittyService.toBean(new TreeNodeImpl(wikitty));

                result.put(node, count);
            }

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    @Override
    public Wikitty restoreVersion(String wikittyId, String version) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public UpdateResponse syncEngin() {
        final WikittyTransaction transaction = new WikittyTransaction();
        try {
            final int numberForCommit = 1000;
            final WikittySearchEngin searchEngin = getSearchEngin();
            final UpdateResponse result = new UpdateResponse();
            final List<Wikitty> wikitties = new ArrayList<Wikitty>(numberForCommit);

            transaction.begin();
            searchEngin.clear(transaction);
            transaction.commit();
            transaction.begin();

            getWikittyStorage().scanWikitties(transaction, new WikittyStorage.Scanner() {
                int count = 0;

                @Override
                public void scan(Wikitty wikitty) {
                    Date deleteDate = wikitty.getDeleteDate();
                    if(deleteDate == null) {
                        count ++;
                        wikitties.add(wikitty);

                        if(count == numberForCommit) {
                            // Reindex
                            UpdateResponse response = searchEngin.store(transaction, wikitties);
                            result.add(response);
                            transaction.commit();
                            // Reinit
                            count = 0;
                            wikitties.clear();
                            transaction.begin();
                        }
                    }
                }
            });

            // Last wikitties
            UpdateResponse response = searchEngin.store(transaction, wikitties);
            result.add(response);

            transaction.commit();
            return result;
        } catch (Exception eee) {
            transaction.rollback();
            throw new WikittyException(eee);
        }
    }

    /**
     * Class used for import process, this class retain numberForCommit object
     * before to send it to storage.
     */
    static protected class WikittyBatchUpdate {
        // TODO poussin 20090902 do configurable numberForCommit
        protected int numberForCommit = 1000;
        protected int currentAdded = 0;
        protected Map<String, WikittyExtension> exts = new HashMap<String, WikittyExtension>();
        protected List<Wikitty> wikitties = new LinkedList<Wikitty>();

        protected AbstractWikittyService ws;
        protected WikittyTransaction transaction;

        public WikittyBatchUpdate(AbstractWikittyService ws, WikittyTransaction transaction) {
            this.ws = ws;
            this.transaction = transaction;
        }

        public void addExtension(WikittyExtension ext) {
            exts.put(ext.getId(), ext);
            inc();
        }

        public void addWikitty(Wikitty w) {
            wikitties.add(w);
            inc();
        }

        /**
         * search extension in local extension list and if missed restore
         * extension from internal WikittyService
         * @param id
         * @return
         */
        public WikittyExtension getExtension(WikittyTransaction transaction, String id) {
            WikittyExtension result = exts.get(id);
            if (result == null) {
                result = ws.restoreExtension(transaction, id);
            }
            return result;
        }

        public void flush() {
            ws.storeExtension(transaction, exts.values());
            ws.store(transaction, wikitties, true);

            exts.clear();
            wikitties.clear();
            currentAdded = 0;
        }

        protected void inc() {
            currentAdded++;
            if (currentAdded >= numberForCommit) {
                flush();
            }
        }
    }

    
    @Override
    public void syncImportFromXml(String xml) {
        Reader reader = new StringReader(xml);
        ImportTask task = new ImportTask(this, reader);
        task.run();
    }


    @Override
    public void syncImportFromUri(String uri) {
        try {
            URL url = new URL(uri);
            Reader reader = new InputStreamReader(url.openStream());
            ImportTask task = new ImportTask(this, reader);
            task.run();
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    
    @Override
    public String asyncImportFromUri(String uri) {
        try {
            URL url = new URL(uri);
            Reader reader = new InputStreamReader(url.openStream());
            ImportTask task = new ImportTask(this, reader);
            FutureTask<String> future = new FutureTask<String>(task, null);
            importExportExecutor.submit(future);
            
            String jobId = UUID.randomUUID().toString();
            importExportTask.put(jobId, future);
            return jobId;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }


    static public class ImportTask implements Runnable {
        protected AbstractWikittyService ws;
        protected WikittyTransaction transaction;
        protected Reader reader;

        public ImportTask(AbstractWikittyService ws, Reader reader) {
            this.ws = ws;
            this.reader = reader;
            this.transaction = new WikittyTransaction();
        }
        
        @Override
        public void run() {
            try {
                transaction.begin();
                XmlPullParserFactory factory = XmlPullParserFactory.newInstance(
                        System.getProperty(XmlPullParserFactory.PROPERTY_NAME), null);
                factory.setNamespaceAware(true);
                XmlPullParser xpp = factory.newPullParser();
                xpp.setInput(reader);

                WikittyExtension ext = null;
                Wikitty w = null;
                String CDATA = null;

                WikittyBatchUpdate batchUpdate = new WikittyBatchUpdate(ws, transaction);

                long time = System.currentTimeMillis();

                int eventType = xpp.getEventType();
                do {
                    String objectVersion = null;
                    if (eventType == xpp.START_DOCUMENT) {
                        log.info("start XML import at " + new Date());
                    } else if (eventType == xpp.END_DOCUMENT) {
                        time = System.currentTimeMillis() - time;
                        log.info("XML import in (ms)" + time);
                    } else if (eventType == xpp.START_TAG) {
                        String name = xpp.getName();
                        if ("extension".equals(name)) {
                            String extName = xpp.getAttributeValue(null, "name");
                            String version = xpp.getAttributeValue(null, "version");
                            String requires = xpp.getAttributeValue(null, "requires");
                            ext = new WikittyExtension(extName, version, requires, new LinkedHashMap<String, FieldType>());
                        } else if ("object".equals(name)) {
                            String id = xpp.getAttributeValue(null, "id");
                            objectVersion = xpp.getAttributeValue(null, "version");
                            String extensions = xpp.getAttributeValue(null, "extensions");
                            w = new Wikitty(id);
                            String[] extensionList = extensions.split(",");
                            for (String extId : extensionList) {
                                String extName = WikittyExtension.computeName(extId);
                                String extVersion = WikittyExtension.computeVersion(extId);
                                extId = WikittyExtension.computeId(extName, extVersion);
                                WikittyExtension e = batchUpdate.getExtension(transaction, extId);
                                if(e == null) {
                                    throw new WikittyException("Extension not found : " + extId);
                                }
                                w.addExtension(e);
                            }
                        }
                    } else if (eventType == xpp.END_TAG) {
                        String name = xpp.getName();
                        if ("extension".equals(name)) {
                            batchUpdate.addExtension(ext);
                            ext = null;
                        } else if ("object".equals(name)) {
                            w.setVersion(objectVersion);
                            batchUpdate.addWikitty(w);
                            w = null;
                        } else if (ext != null && "field".equals(name)) {
                            FieldType type = new FieldType();
                            String fieldName = WikittyUtil.parseField(CDATA, type);
                            ext.addField(fieldName, type);
                        } else if (ext != null && "tagvalues".equals(name)) {
                            Map<String, String> tagValues = WikittyUtil.tagValuesToMap(CDATA);
                            ext.setTagValues(tagValues);
                        } else if (w != null) {
                            String[] fq = name.split("\\.");
                            String extensionName = fq[0];
                            String fieldName = fq[1];
                            FieldType fieldType = w.getFieldType(name);
                            if(fieldType.isCollection()) {
                                w.addToField(extensionName, fieldName, CDATA);
                            } else {
                                w.setField(extensionName, fieldName, CDATA);
                            }
                        }
                    } else if (eventType == xpp.TEXT) {
                        CDATA = xpp.getText();
                    }
                    eventType = xpp.next();
                } while (eventType != xpp.END_DOCUMENT);

                // don't forget to flush batchUpdate :)
                batchUpdate.flush();
                transaction.commit();
            } catch (Exception eee) {
                transaction.rollback();
                throw new WikittyException(eee);
            }
        }
    } // end ImportTask
    
    @Override
    public String asyncExportAllByCriteria(Criteria criteria) {
        try {
            String jobId = UUID.randomUUID().toString();

            File file = new File(EXPORT_DIRECTORY, jobId);
            String url = EXPORT_URL + jobId;
            Writer result = new FileWriter(file);
            ExportTask task = new ExportTask(this, criteria, result);
            FutureTask<String> future = new FutureTask<String>(task, url);
            importExportExecutor.submit(future);

            importExportTask.put(jobId, future);
            return jobId;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    
    @Override
    public String syncExportAllByCriteria(Criteria criteria) {
        StringWriter result = new StringWriter();
        ExportTask task = new ExportTask(this, criteria, result);
        task.run();
        return result.toString();
    }
    

    static public class ExportTask implements Runnable {
        protected AbstractWikittyService ws;
        protected WikittyTransaction transaction;
        
        protected Criteria criteria;
        protected Writer result;

        public ExportTask(AbstractWikittyService ws, Criteria criteria, Writer result) {
            this.ws = ws;
            this.transaction = new WikittyTransaction();
            this.criteria = criteria;
            this.result = result;
        }
        
       @Override
        public void run() {
            try {
                transaction.begin();
                PagedResult<Wikitty> pageResult = ws.findAllByCriteria(transaction, criteria);

                // keep extension already done
                Set<String> extDone = new HashSet<String>();
                result.write("<wikengo>\n");
                for (Wikitty w : pageResult.getAll()) {
                    String extensionList = "";
                    for (WikittyExtension ext : w.getExtensions()) {
                        String id = ext.getId();
                        extensionList += "," + id;
                        if (!extDone.contains(id)) {
                            extDone.add(id);
                            result.write("  <extension name='" + ext.getName()
                                    + "' version='" + ext.getVersion()
                                    + (ext.getRequires() != null ? "' requires='" + ext.getRequires() : "")
                                    + "'>\n");
                            Map<String, String> tagValues = ext.getTagValues();
                            result.write("    <tagvalues>" + WikittyUtil.tagValuesToString(tagValues) + "</tagvalues>\n");
                            for (String fieldName : ext.getFieldNames()) {
                                String def = ext.getFieldType(fieldName).toDefinition(fieldName);
                                result.write("    <field>" + def + "</field>\n");
                            }
                            result.write("  </extension>\n");
                        }
                    }
                    if (!"".equals(extensionList)) {
                        // delete first ','
                        extensionList = extensionList.substring(1);
                    }
                    result.write("  <object id='" + w.getId() + "' version='" + w.getVersion() + "' extensions='" + extensionList + "'>\n");
                    for (String fieldName : w.fieldNames()) {
                        FieldType type = w.getFieldType(fieldName);
                        if (type.isCollection()) {
                            Object fqField = w.getFqField(fieldName);
                            if (fqField != null) {
                                for (Object o : (Collection) fqField) {
                                    String fqFieldValue = WikittyUtil.toString(type, o);
                                    if (fqFieldValue != null) {
                                        fqFieldValue = StringEscapeUtils.escapeXml(fqFieldValue);
                                        result.write("    <" + fieldName + ">" + fqFieldValue + "</" + fieldName + ">\n");
                                    }
                                }
                            }
                        } else {
                            String fqFieldValue = WikittyUtil.toString(type, w.getFqField(fieldName));
                            if (fqFieldValue != null) {
                                fqFieldValue = StringEscapeUtils.escapeXml(fqFieldValue);
                                result.write("    <" + fieldName + ">" + fqFieldValue + "</" + fieldName + ">\n");
                            }
                        }
                    }
                    result.write("  </object>\n");
                }
                result.write("</wikengo>\n");
                transaction.commit();
            } catch (IOException eee) {
                transaction.rollback();
                throw new WikittyException(eee);
            }
        }
    } // end ExportTask

    
    @Override
    public JobState infoJob(String jobId) {
        try {
            Future<String> future = importExportTask.get(jobId);
            JobState result = new JobState();
            if (future.isDone()) {
                result.status = "done";
                result.resourceUri = future.get();
            } else if (future.isCancelled()) {
                result.status = "cancelled";
            } else {
                result.status = "inProgress";
            }
            return result;
        } catch (Exception eee) {
            throw new WikittyException(eee);
        }
    }

    
    @Override
    public void cancelJob(String jobId) {
        Future future = importExportTask.get(jobId);
        future.cancel(true); // true to kill process, perhaps to strong ?
    }

    
    @Override
    public void freeJobResource(String jobId) {
        Future<String> future = importExportTask.remove(jobId);
        if (future != null) {
            File file = new File(EXPORT_DIRECTORY, jobId);
            file.delete();
        }
    }

    /**
     * Method copied from eugengo-0.7 generators
     * 
     * @param bean
     * @return
     */
    /*public static TreeNodeImpl toImpl(TreeNode bean) {
        if (bean == null) return null;
        TreeNodeImpl impl = new TreeNodeImpl(bean);
        return impl;
    }*/

    /**
     * Method copied from eugengo-0.7 generators
     * 
     * @param bean
     * @return
     */
    protected static void fillBeanAttributes(TreeNode fromBean, TreeNodeBean toBean) {
        String beanId = fromBean.getWikittyId();
        toBean.id = beanId;

        String beanVersion = fromBean.getWikittyVersion();
        toBean.setWikittyVersion(beanVersion);

        toBean.setName(fromBean.getName());
        toBean.setParent(fromBean.getParent());
        // WARNING: Copy collection to other collection to not manipulate storage collection directly
        Collection<String> treeNodeChildren = fromBean.getChildren();
        if(treeNodeChildren != null) {
            toBean.TreeNode$children = new HashSet<String>(treeNodeChildren);
        }
    }

    /**
     * Method copied from eugengo-0.7 generators
     * 
     * @param impl
     * @return
     */
    public static TreeNode toBean(TreeNodeImpl impl) {
        if (impl == null) return null;
        TreeNodeBean bean = new TreeNodeBean();
        AbstractWikittyService.fillBeanAttributes(impl, bean);
        return bean;
    }
}
