/* *##% ToPIA - Migration service
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*/
package org.nuiton.topia.migration;

import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaRuntimeException;
import org.nuiton.topia.event.TopiaContextEvent;
import org.nuiton.topia.event.TopiaTransactionEvent;
import org.nuiton.topia.framework.TopiaContextImplementor;


import java.io.File;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.Version;
import org.nuiton.util.Resource;
import org.hibernate.cfg.Configuration;
import org.hibernate.tool.hbm2ddl.SchemaExport;
import org.nuiton.topia.framework.TopiaContextImpl;
import org.nuiton.topia.framework.TopiaUtil;
import org.nuiton.topia.migration.ManualMigrationCallback.MigrationChoice;
import org.nuiton.util.VersionUtil;

/**
 * TopiaMigrationServiceImpl.java
 *
 * Classe principale du projet.
 * 
 * @author tchemit
 * @version $Revision: 1459 $
 *
 * Last update : $Date: 2009-05-16 09:56:47 +0200 (Sat, 16 May 2009) $
 */
public class ManualMigrationEngine //extends MigrationServiceImpl
        implements TopiaMigrationService {

    // log
    private final static Log log = LogFactory.getLog(ManualMigrationEngine.class);
    /**
     * La version de référence
     */
    static public final String MIGRATION_APPLICATION_VERSION = "topia.service.migration.version";
    /**
     * L'emplacement de tous les mappings
     */
    static public final String MIGRATION_MAPPING_DIRECTORY = "topia.service.migration.mappingdir";
    /**
     * Les noms des modèles connus
     */
    static public final String MIGRATION_MODEL_NAME = "topia.service.migration.modelname";
    /**
     * L'unique handler a utiliser
     */
    static public final String MIGRATION_CALLBACK = "topia.service.migration.callback";
    /**
     * Un drapeau pour indiquer si on ne doit pas lancer le service au demarrage
     */
    static public final String MIGRATION_NO_MIGRATE_ON_INIT = "topia.service.migration.no.migrate.on.init";
    /**
     * Configuration hibernate ne mappant que l'entite version (initialise en pre-init)
     */
    protected Configuration versionConfiguration;
    /**
     * Chemin du dossier contenant les schemas de toutes les versions (initialise en pre-init)
     */
    protected String mappingsDirectory;
    /**
     * ensemble des versions connues par le service
     */
    protected SortedSet<Version> versions;
    /**
     * Version courante de l'application (initialise en pre-init)
     */
    protected Version applicationVersion;
    /**
     * Un drapeau pour savoir si la table version existe en base (initialise en pre-init)
     */
    protected boolean versionTableExist;
    /**
     * Version courante de la base (initialise en pre-init)
     */
    protected Version dbVersion;
    /**
     * Un drapeau pour ne pas effectuer de migration au demarrage (initialise en pre-init)
     */
    protected boolean noMigrateOnInit;
    /**
     * CallbackHandler list (initialise en pre-init)
     */
    protected ManualMigrationCallback callback;
    /**
     * topia root context (initialise en pre-init)
     */
    protected TopiaContextImplementor rootContext;
    /**
     * Un drapeau pour savoir si le service a bien ete initialise (i.e a bien fini la methode preInit)
     */
    protected boolean init = false;

    @Override
    public Class<?>[] getPersistenceClasses() {
        return new Class<?>[]{TMSVersionImpl.class};
    }

    @Override
    public String getServiceName() {
        return TopiaMigrationService.SERVICE_NAME;
    }

    @Override
    public boolean preInit(TopiaContextImplementor context) {
        rootContext = context;

        Properties config = context.getConfig();

        String mappingDirectory = config.getProperty(MIGRATION_MAPPING_DIRECTORY, null);

        String version = config.getProperty(MIGRATION_APPLICATION_VERSION, null);
        String callbackStr = config.getProperty(MIGRATION_CALLBACK, "");
        String[] dirs = config.getProperty(TopiaContextImpl.TOPIA_PERSISTENCE_DIRECTORIES, "").split(",");

        this.noMigrateOnInit = Boolean.valueOf(config.getProperty(MIGRATION_NO_MIGRATE_ON_INIT, "true"));
        String modelName = config.getProperty(MIGRATION_MODEL_NAME, null);

        if (version == null || version.trim().isEmpty()) {
            throw new IllegalStateException("'" + MIGRATION_APPLICATION_VERSION + "' not set.");
        }
        if (modelName == null || modelName.trim().isEmpty()) {
            throw new IllegalStateException("'" + MIGRATION_MODEL_NAME + "' not set.");
        }
        if (callbackStr == null || callbackStr.trim().isEmpty()) {
            throw new IllegalStateException("'" + MIGRATION_CALLBACK + "' not set.");
        }
        if (mappingDirectory == null || mappingDirectory.trim().isEmpty()) {
            throw new IllegalStateException("'" + MIGRATION_MAPPING_DIRECTORY + "' not set.");
        }

        this.applicationVersion = VersionUtil.valueOf(version.trim());
        this.mappingsDirectory = mappingDirectory.trim() + "/" + modelName.trim();

        // enregistrement du callback
        try {
            Class<?> clazz = (Class<?>) Class.forName(callbackStr);
            this.callback = (ManualMigrationCallback) clazz.newInstance();

        } catch (ClassNotFoundException e) {
            log.error("CallbackHandler Class " + callbackStr + " not found", e);
        } catch (InstantiationException e) {
            log.error("CallbackHandler class " + callbackStr + " cannot be instanciated", e);
        } catch (IllegalAccessException e) {
            log.error("CallbackHandler class " + callbackStr + " cannot be accessed", e);
        }

        // creation de la configuration hibernate ne concernant que l'entite Version
        // afin de pouvoir creer la table via un schemaExport si necessaire

        versionConfiguration = new Configuration();

        // ajout des repertoires contenant les mappings hibernate

        for (String dir : dirs) {
            dir = dir.trim();
            if (!dir.isEmpty()) {
                log.debug("addDirectory " + dir);
                versionConfiguration.addDirectory(new File(dir));
            }
        }

        for (Class<?> clazz : getPersistenceClasses()) {
            log.debug("addClass " + clazz);
            versionConfiguration.addClass(clazz);
        }

        Properties prop = new Properties();
        prop.putAll(versionConfiguration.getProperties());
        prop.putAll(config);

        versionConfiguration.setProperties(prop);

        this.versionTableExist = TopiaUtil.isSchemaExist(versionConfiguration, TMSVersionImpl.class.getName());

        // recuperation de la version de la base
        try {

            dbVersion = detectDbVersion();
        } catch (MigrationServiceException e) {
            throw new TopiaRuntimeException("Can't obtain dbVersion for reason " + e.getMessage(), e);
        }

        init = true;

        // add topia context listener
        context.addTopiaContextListener(this);
        context.addTopiaTransactionVetoable(this);

        if (!noMigrateOnInit) {

            // lancement de la migration

            try {

                doMigrateSchema();

            } catch (MigrationServiceException e) {
                throw new TopiaRuntimeException("Can't migrate schema for reason " + e.getMessage(), e);
            }
        }

        return true;
    }

    @Override
    public boolean postInit(TopiaContextImplementor context) {
        return true;
    }

    @Override
    public void preCreateSchema(TopiaContextEvent event) {
    }

    @Override
    public void preRestoreSchema(TopiaContextEvent event) {
    }

    @Override
    public void preUpdateSchema(TopiaContextEvent event) {
    }

    @Override
    public void postCreateSchema(TopiaContextEvent event) {

        if (log.isInfoEnabled()) {
            log.info("postCreateSchema event called : put version in database");
        }

        saveVersion(applicationVersion);
    }

    @Override
    public void postUpdateSchema(TopiaContextEvent event) {
        if (log.isInfoEnabled()) {
            log.info("postUpdateSchema event called : put version in database");
        }

        saveVersion(applicationVersion);
    }

    @Override
    public void postRestoreSchema(TopiaContextEvent event) {

        if (log.isInfoEnabled()) {
            log.info("postRestoreSchema event detected, redo, schema migration");
        }
        try {

            doMigrateSchema();

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("postRestoreSchema schema migration failed for reason " + e.getMessage(), e);
            }
        }
    }

    @Override
    public void beginTransaction(TopiaTransactionEvent event) {

        TopiaContextImplementor context = (TopiaContextImplementor) event.getSource();

        // add topia context listener
        context.addTopiaContextListener(this);
    }

    public void doMigrateSchema() throws MigrationServiceException {
        // migration
        boolean complete = migrateSchema();
        if (!complete) {
            if (log.isErrorEnabled()) {
                log.error("Database migration not complete");
            }
            throw new TopiaRuntimeException("Database migration not succesfully ended !");
        }
    }

    @Override
    public boolean migrateSchema() throws MigrationServiceException {

        checkInit();

        log.info("Starting Topia Migration Service  - Application version : " + applicationVersion.getVersion() + ", database version : " + dbVersion.getVersion());

        // tel if migration is needed
        boolean bMigrationNeeded = false;

        if (versionTableExist && dbVersion.equals(applicationVersion)) {
            log.info("Database is up to date, no migration needed.");
            // la base est a jour
            return true;
        }

        if (dbVersion.before(applicationVersion)) {

            SortedSet<Version> allVersions = getVersions();

            log.info("Database need update, available versions : " + allVersions);

            // on filtre les versions a appliquer
            List<Version> versionsToApply = detectVersions(allVersions, dbVersion, applicationVersion);

            if (versionsToApply.isEmpty()) {
                log.info("no version to apply, no migration needed.");
            } else {
                bMigrationNeeded = true;
                log.info("will migrate versions " + versionsToApply);
                // ask handler for migration
                MigrationChoice bMigrationWanted = callback.doMigration(rootContext,
                        dbVersion,
                        applicationVersion,
                        versionsToApply);

                log.info("Handler choose : " + bMigrationWanted);
                if (bMigrationWanted == MigrationChoice.NO_MIGRATION) {
                    // l'utilisateur a annule la migration
                    return false;
                }
            }
        }

        // on sauvegarde la version si necessaire (base non versionnee ou migration realisee)
        if (!versionTableExist || bMigrationNeeded) {

            log.info("Set application version in database to " + applicationVersion);

            // put version in database and create table if required
            saveVersion(applicationVersion);
        }

        // return succes flag
        // - no migration needed
        // - or migration needed and accepted
        return true;
    }

    /**
     * Enregistre la version donnee en base avec creation de la table
     * si elle n'existe pas.
     *
     * @param version la nouvelle version de la base
     */
    public void saveVersion(Version version) {
        checkInit();
        try {

            boolean createTable = !versionTableExist;
            // update version even if database has not been migrated
            // only case that database doesn't exist match this
            if (createTable) {
                // si la base n'etait pas versionnee, la table version n'existe pas
                // creation
                log.debug("Adding table to put version");

                // creer le schema en base
                // dans la configuration versionConfiguration, il n'y a que la table version
                SchemaExport schemaExport = new SchemaExport(versionConfiguration);
                schemaExport.create(log.isDebugEnabled(), true);

                log.debug("Table for " + TMSVersion.class.getSimpleName() + " created");

            }
            // Changement de la version en base
            TopiaContext tx = null;

            try {
                tx = rootContext.beginTransaction();

                TMSVersionDAO dao = MigrationServiceDAOHelper.getTMSVersionDAO(tx);

                //FIXME on supprime toues les versions precedentes ???
                //FIXME il serait mieux de conserver toutes les versions je pense...
                //FIXME on pourrait conserver l'information sur les date de mise a jour
                List<TMSVersion> toDelete = dao.findAll();
                for (TMSVersion v : toDelete) {
                    v.delete();
                }

                log.info("Database version : " + version);
                dao.create(TMSVersion.VERSION, version.getVersion());

                tx.commitTransaction();
            } catch (TopiaException e) {
                if (tx != null) {
                    tx.rollbackTransaction();
                }
                throw e;
            } finally {
                if (tx != null) {
                    tx.closeContext();
                }
            }
        } catch (TopiaException e) {
            throw new TopiaRuntimeException(e);
        }

        // on change les etats internes du service
        // ainsi cela empechera le redeclanchement de la migration
        // suite a une creation de schema
        versionTableExist = true;
        dbVersion = version;
    }

    /**
     * Recupere toutes les versions prises en charge par le service de migration.
     *
     * @return l'ensemble triee (par ordre croissant) de toutes les versions
     * detectees a partir du repertoire des mappings pour le modele donne.
     * @throws MigrationServiceException pour tout probleme
     */
    protected SortedSet<Version> getVersions() throws MigrationServiceException {
        if (versions == null) {
            checkInit();

            // schema des noms de dossier de version
            final Pattern MAPPING_PATTERN = Pattern.compile(mappingsDirectory + File.separator + "([0-9]+(\\.[0-9]+)*)");
            ClassLoader classLoader = getClass().getClassLoader();

            List<URL> urls = Resource.getURLs(".*" + mappingsDirectory + "/.*", classLoader instanceof URLClassLoader ? (URLClassLoader) classLoader : null);

            // ensemble ordonnee des version a charger apres
            versions = new TreeSet<Version>();

            if (urls != null && !urls.isEmpty()) {

                for (URL url : urls) {
                    if (log.isDebugEnabled()) {
                        log.debug("url to scan " + url);
                    }
                    Matcher matcher = MAPPING_PATTERN.matcher(url.getFile());
                    if (matcher.find()) {
                        // group(1) est ce qui match entre le premier niveau de parentheses
                        String sVersion = matcher.group(1);

                        versions.add(VersionUtil.valueOf(sVersion));
                    }
                }
            }
        }
        return versions;
    }

    /**
     * Filtre l'ensemble des versions, pour obtenir toutes les versions a migrer.
     * 
     * @param versions les versions a filtrer
     * @param dbVersion la version actuel
     * @param applicationVersion la version a atteindre
     * @return la liste des versions a migrer
     */
    protected List<Version> detectVersions(SortedSet<Version> versions, Version dbVersion, Version applicationVersion) {
        List<Version> toApply = new ArrayList<Version>();
        for (Version v : versions) {
            log.debug("detected version " + v);
            if (v.compareTo(dbVersion) <= 0) {
                // version trop ancienne
                continue;
            }
            if (v.compareTo(applicationVersion) > 0) {
                // version trop recente
                continue;
            }
            // la version est a appliquer
            log.info("version to migrate : " + v);
            toApply.add(v);
        }
        return toApply;
    }

    /**
     * Detecte la version de la base.
     *
     * Note : si la base n'est pas versionnee, on considere alors qu'elle est en
     * en version 0.
     * 
     * @return la version de la base
     * @throws MigrationServiceException
     */
    protected Version detectDbVersion() throws MigrationServiceException {

        Version v = null;
        try {
            if (versionTableExist) {
                TopiaContext tx = null;
                try {
                    tx = rootContext.beginTransaction();
                    TMSVersionDAO dao = MigrationServiceDAOHelper.getTMSVersionDAO(tx);
                    List<TMSVersion> versionsInDB = dao.findAll();
                    if (!versionsInDB.isEmpty()) {
                        v = VersionUtil.valueOf(versionsInDB.get(0).getVersion());
                    }
                } finally {
                    if (tx != null) {
                        tx.closeContext();
                    }
                }
            }
        } catch (TopiaException e) {
            throw new MigrationServiceException(e);
        }

        if (v == null) {
            // la base dans ce cas n'est pas versionee.
            // On dit que la version de la base est 0
            // et les schema de cette version 0 doivent
            // etre detenu en local
            v = Version.VZERO;
            log.info("Database version not found, so database schema is considered as V0");
        } else {
            log.info("Database version : " + v);
        }

        return v;
    }

    protected void checkInit() {
        if (!init) {
            throw new IllegalStateException("le service n'est pas initialisé!");
        }
    }
}
