/**
 * *##% ToPIA - Migration service
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.topia.migration;

import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.jdbc.Work;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.util.Version;

/**
 * ManualMigrationCallback
 *
 * @author chemit
 * @version $Revision: 81 $
 *
 * Last update : $Date: 2008-08-21 18:21:52 +0200 (jeu 21 aoû 2008) $
 * By : $Author: chemit $
 */
public abstract class ManualMigrationCallback {

    /** to use log facility, just put in your code: log.info("..."); */
    static private Log log = LogFactory.getLog(ManualMigrationCallback.class);

    /**
     * CallbackHandler return type.
     */
    public static enum MigrationChoice {
        // no migration

        NO_MIGRATION,
        // migration
        MIGRATION,
        // migration done by application, only version managed
        CUSTOM_MIGRATION
    }

    public abstract boolean askUser(Version dbVersion, Version applicationVersion, List<Version> versions);

    /**
     * Tentative de migration depuis la version de la base version la version souhaitee.
     *
     * On applique toutes les migrations de version indiquee dans le parametre <code>version</code>.
     *
     * Pour chaque version, on cherche la methode migrateTo_XXX ou XXX est la
     * version transforme en identifiant java via la methode {@link Version#getValidName()}.
     * et on l'execute.
     *
     * Note: pour chaque version a appliquer, on ouvre une nouvelle transaction.
     *
     * @param ctxt topia context de la transaction en cours
     * @param dbVersion database version
     * @param applicationVersion application version
     * @param versions  all versions knwon by service
     * @return migration aggrement
     */
    public MigrationChoice doMigration(TopiaContext ctxt, Version dbVersion, Version applicationVersion, List<Version> versions) {

        log.info(versions);

        MigrationChoice result = MigrationChoice.NO_MIGRATION;

        boolean doMigrate = askUser(dbVersion, applicationVersion, versions);

        if (doMigrate) {
            TopiaContextImplementor tx = null;

            for (Version v : versions) {
                // ouverture d'une connexion direct JDBC sur la base
                try {

                    tx = (TopiaContextImplementor) ctxt.beginTransaction();

                    try {

                        String methodName = "migrateTo_" + v.getValidName();

                        Method m = getClass().getMethod(methodName, TopiaContextImplementor.class);
                        m.setAccessible(true);

                        log.info("start migration of version " + v);
                        
                        if (log.isDebugEnabled()) {
                            log.debug("launch method " + methodName);
                        }

                        m.invoke(this, tx);

                        // commit des modifs
                        tx.commitTransaction();

                        // la migration a reussi, on dit que c fait
                        result = MigrationChoice.CUSTOM_MIGRATION;
                    } catch (Exception eee) {
                        // en cas d'erreur
                        log.error("Migration impossible de la base", eee);
                        // rollback du travail en cours
                        tx.rollbackTransaction();
                    } finally {
                        // close database connexion
                        if (tx != null) {
                            tx.closeContext();
                        }
                    }

                } catch (Exception eee) {
                    log.error("Error lors de la tentative de migration", eee);
                }
            }
        }
        return result;
    }

    public void executeSQL(TopiaContextImplementor tx, final String... sqls) throws TopiaException {

        tx.getHibernate().doWork(new Work() {

            @Override
            public void execute(Connection connection) throws SQLException {
                for (final String sql : sqls) {
                    log.info(sql);
                    PreparedStatement sta = connection.prepareStatement(sql);
                    sta.executeUpdate();
                    log.debug("done " + sql);
                    sta.close();
                }
            }
        });

    }
}
