/*
 * *##% NuitonMatrix
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*/
package org.nuiton.math.matrix.gui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;

import org.nuiton.i18n.I18n;
import org.nuiton.math.matrix.MatrixException;
import org.nuiton.math.matrix.MatrixFactory;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.util.ListenerSet;

/**
 * JPanel contenant une JTable pour afficher une Matrice a une ou deux
 * dimension.
 *
 * TODO: Une methode permettant de retourne la sous matrice de la selection que
 * la matrice soit reprensentée en lineaire ou non. (avoir un mapping cellule de
 * table vers element de matrice
 *
 * Created: 29 oct. 2004
 *
 * @author Benjamin Poussin <poussin@codelutin.com>
 * @version $Revision: 168 $
 *
 * Mise a jour: $Date: 2009-07-17 18:18:45 +0200 (ven., 17 juil. 2009) $
 * par : $Author: echatellier $
 */
public class MatrixPanelEditor extends MatrixEditor implements TableModelListener { // MatrixPanelEditor

    /** serialVersionUID */
    private static final long serialVersionUID = 2097859265435050946L;
    private final static int DEFAULT_WIDTH = 150;
    private final static int DEFAULT_HEIGHT = 150;
    protected ListenerSet<MatrixPanelListener> listeners = new ListenerSet<MatrixPanelListener>();
    protected JTable table;
    protected MatrixND m;
    protected MatrixTableModel tableModel;
    protected JScrollPane editArea;
    protected MatrixPopupMenu popupMenu;
    // protected JTextArea text;
    /** if true, use linear representation of matrix */
    protected boolean linearModel = false;
    /** if false don't show default value in matrix (ex: 0) */
    protected boolean linearModelShowDefault = false;
    /** Boolean to authorize table editing.
     * @deprecated since 2.0.0 : duplicate field from parent for nothing!
     */
    @Deprecated
    protected boolean enabled = true;
    /** Boolean to show matrix.      
     * @deprecated since 2.0.0 : duplicate field from parent for nothing!
     */
    @Deprecated
    protected boolean visible = true;
    /** Boolean to autorize matrice dimension changes. */
    protected boolean dimensionEdit;

    /**
     * Construct a new JPanel to edit matrix.
     *
     * @param m the matrix to edit.
     * @param dimensionEdit to enabled matrix dimension changes.
     */
    public MatrixPanelEditor(MatrixND m, boolean dimensionEdit) {
        this(dimensionEdit, DEFAULT_WIDTH, DEFAULT_HEIGHT);
        setMatrix(m);
    }

    /**
     * Construct a new JPanel to edit matrix.
     *
     * @param dimensionEdit to enabled matrix dimension changes.
     * @param width width prefered for the component
     * @param height height prefered for the component
     */
    public MatrixPanelEditor(boolean dimensionEdit, int width, int height) {
        this.dimensionEdit = dimensionEdit;
        setPreferredSize(new Dimension(width, height));
        initObjet();
    }

    /**
     * Construct a new JPanel to edit matrix.
     *
     * @param dimensionEdit to enabled matrix dimension changes.
     */
    public MatrixPanelEditor(boolean dimensionEdit) {
        this(dimensionEdit, DEFAULT_WIDTH, DEFAULT_HEIGHT);
    }

    /**
     * Construct a new JPanel to edit matrix. Matrix dimension can not change.
     */
    public MatrixPanelEditor() {
        this(false);
    }

    protected MatrixFactory getFactory() {
        return MatrixFactory.getInstance();
    }

    /**
     *
     * @param l listener to add
     * @deprecated  since 2.0.0 : this is not a valid listener adder
     */
    @Deprecated
    public void addMatrixListener(MatrixPanelListener l) {
        listeners.add(l);
    }

    /**
     *
     * @param l listener to add
     * @since 2.0.0
     */
    public void addMatrixPanelListener(MatrixPanelListener l) {
        listeners.add(l);
    }

    /**
     *
     * @param l listener to remove
     * @since 2.0.0
     */
    public void removeMatrixPanelListener(MatrixPanelListener l) {
        listeners.remove(l);
    }

    /**
     *
     * @return the listeners registred
     * @since 2.0.0
     */
    public MatrixPanelListener[] getMatrixPanelListeners() {
        java.util.List<MatrixPanelListener> r = new java.util.ArrayList<MatrixPanelListener>();
        for (java.util.Iterator<MatrixPanelListener> itr = listeners.iterator(); itr.hasNext();) {
            r.add(itr.next());
        }
        return r.toArray(new MatrixPanelListener[r.size()]);
    }

    protected void initObjet() {
        setLayout(new BorderLayout());
        editArea = new JScrollPane();
        add(editArea, BorderLayout.CENTER);
        initDimensionEdit();
    }
    protected JButton bEdit = null;

    @Override
    public JButton getButtonEdit() {
        if (bEdit == null) {
            bEdit = new JButton(I18n._("lutinmatrix.create.matrix.button"));
            bEdit.addActionListener(new ActionListener() {

                @Override
                public void actionPerformed(ActionEvent e) {
                    String dim;
                    dim = JOptionPane.showInputDialog(null, I18n._("lutinmatrix.create.matrix.message"), I18n._("lutinmatrix.create.matrix.title"),
                            JOptionPane.DEFAULT_OPTION);

                    if (dim != null) {
                        String[] sdim = dim.split(";");
                        int[] idim = new int[sdim.length];
                        for (int i = 0; i < idim.length; i++) {
                            idim[i] = Integer.parseInt(sdim[i]);
                        }
                        setMatrix(getFactory().create(idim));
                    }
                }
            });
            add(bEdit, BorderLayout.SOUTH);
        }
        return bEdit;
    }

    /**
     * @return Returns the linearModel.
     */
    @Override
    public Boolean isLinearModel() {
        return this.linearModel;
    }

    /**
     * @param linearModel The linearModel to set.
     */
    @Override
    public void setLinearModel(Boolean linearModel) {
        this.linearModel = linearModel;
        initObject(m);
    }

    /**
     * @return Returns the linearModelShowDefault.
     */
    @Override
    public Boolean isLinearModelShowDefault() {
        return this.linearModelShowDefault;
    }

    /**
     * @param linearModelShowDefault The linearModelShowDefault to set.
     */
    @Override
    public void setLinearModelShowDefault(Boolean linearModelShowDefault) {
        this.linearModelShowDefault = linearModelShowDefault;
        initObject(m);
    }

    /**
     * Get the value of dimensionEdit.
     *
     * @return value of dimensionEdit.
     */
    public boolean isDimensionEdit() {
        return dimensionEdit;
    }

    /**
     * Set the value of dimensionEdit.
     *
     * @param v Value to assign to dimensionEdit.
     */
    public void setDimensionEdit(boolean v) {
        this.dimensionEdit = v;
        initDimensionEdit();
    }

    protected void initDimensionEdit() {
        getButtonEdit().setVisible(dimensionEdit);
    }

    protected void initObject(MatrixND m) {
        if (m == null) {
            editArea.setViewportView(null);
        } else { // if (m.getNbDim() <= 2) {
            // pour les matrices 1D et 2D
            JTable t = getTable();
            if (isLinearModel()) {
                tableModel = new MatrixTableModelLinear(m,
                        isLinearModelShowDefault());
            } else {
                tableModel = new MatrixTableModelND(m);
            }
            t.getModel().removeTableModelListener(this);
            tableModel.addTableModelListener(this);
            t.setModel(tableModel);
            t.setDefaultRenderer(String.class, tableModel.getMatrixCellRenderer());
            editArea.setViewportView(t);
            // next line is needed otherwize matrix doesn't appear
            add(editArea, BorderLayout.CENTER);
            // if (table.getColumnCount() > 0 && m.getNbDim() > 1) {
            // table.getColumnModel().getColumn(0).setCellRenderer(
            // tableModel.getMatrixCellRenderer());
            // }
            // text = null;
        }
        // else {
        // // pour les matrices 3D et plus
        // text = new JTextArea();
        // editArea.setViewportView(text);
        // text.setText(MatrixHelper.encodeToXML(m));
        // table = null;
        // editArea.setColumnHeaderView(null);
        // }
        setEnabled(enabled);
        setVisible(visible);
        repaint();
    }

    @Override
    public JTable getTable() {
        if (table == null) {
            popupMenu = new MatrixPopupMenu(this);
            table = new JTable() {

                private final static long serialVersionUID = 1l;

                @Override
                public void processMouseEvent(MouseEvent event) {
                    if (event.isPopupTrigger()) {
                        popupMenu.show(event.getComponent(), event.getX(),
                                event.getY());
                    }
                    super.processMouseEvent(event);
                }
            };

            table.getInputMap().put(
                    KeyStroke.getKeyStroke(KeyEvent.VK_C, Event.CTRL_MASK),
                    "copy");
            table.getActionMap().put("copy",
                    popupMenu.getSendToClipBoardSelectionCopyAction());

            table.getInputMap().put(
                    KeyStroke.getKeyStroke(KeyEvent.VK_V, Event.CTRL_MASK),
                    "paste");
            table.getActionMap().put("paste",
                    popupMenu.getSendToClipBoardCurrentPasteAction());

            /*
             * table.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_A,
             * Event.CTRL_MASK), "selectAll");
             * table.getActionMap().put("selectAll", new AbstractAction(){
             * public void actionPerformed(ActionEvent e) { table.selectAll();
             * }});
             */

            table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
            table.setSelectionMode(ListSelectionModel.SINGLE_INTERVAL_SELECTION);
            table.setCellSelectionEnabled(true);

        }
        return table;
    }

    @Override
    public void setMatrix(MatrixND m) throws MatrixException {
        initObject(m);
        this.m = m;
        fireEvent();
    }

    @Override
    public MatrixND getMatrix() {
        // if (m == null) {
        return m;
        // } else if (m.getNbDim() <= 2) {
        // return m;
        // } else {
        // return MatrixHelper.decodeFromXML(text.getText());
        // }
    }

    /**
     * Enable the matrix to be edited. By default, the matrix is editable.
     */
    @Override
    public void setEnabled(boolean enabled) {
        if (tableModel != null) {
            tableModel.setEnabled(enabled);
        }
        if (table != null) {
            table.setEnabled(enabled);
        }
        super.setEnabled(enabled);
        // if (text != null) {
        // text.setEditable(enabled);
        // }
        // si la table n'est pas editable, inutile de laisser le bouton
        // de creation de matrice.
        if (!enabled && dimensionEdit) {
            dimensionEdit = false;
            initDimensionEdit();
        }
        if (!enabled) {
            editArea.setViewportView(null);
        }
        this.enabled = enabled;
    }

    @Override
    public boolean isEnabled() {
        return enabled;
    }

    /**
     * Set the matrix visible. By default, the matrix is visible.
     * @param visible
     */
    @Override
    public void setVisible(boolean visible) {
        if (table != null) {
            table.setVisible(visible);
        }
        super.setVisible(visible);
        this.visible = visible;
    }

    @Override
    public boolean isVisible() {
        return visible;
    }
    /*
     * @see javax.swing.event.TableModelListener#tableChanged(javax.swing.event.TableModelEvent)
     */

    @Override
    public void tableChanged(TableModelEvent e) {
        fireEvent();
    }

    @Override
    protected void fireEvent() {
        MatrixPanelEvent e = new MatrixPanelEvent(this);
        for (Iterator<MatrixPanelListener> i = listeners.iterator(); i.hasNext();) {
            MatrixPanelListener l = i.next();
            l.matrixChanged(e);
        }
    }

    /**
     * Une petite fonction main pour le test...
     * @param args 
     */
    public static void main(String[] args) {
        I18n.init("fr", "FR");
        // I18n.init("en","EN");

        JFrame frame = new JFrame();
        MatrixPanelEditor ed = null;

        try {
            ed = new MatrixPanelEditor(true);
            ed.setLinearModel(false);
            frame.getContentPane().add(ed);

            // MatriceND m = new MatriceNDImpl(new int[]{4,4});
            // m.set(new int[]{0,0}, new Const(0));
            // m.set(new int[]{0,1}, new Const(1));
            // m.set(new int[]{0,2}, new Const(2));
            // m.set(new int[]{1,0}, new Const(3));
            // m.set(new int[]{1,1}, new Const(4));
            // m.set(new int[]{1,2}, new Const(5));

            List<String> sem1 = Arrays.asList(new String[]{"toto", "titi", "tutu"});
            List<String> sem2 = Arrays.asList(new String[]{"tata", "tete", "tyty"});
            List<String> sem3 = Arrays.asList(new String[]{"riri", "fifi", "loulou"});

            /*
             * MatrixND m = MatrixFactory.getInstance().create(new
             * int[]{100,100});
             */

            /*
             * MatrixND m = MatrixFactory.getInstance().create("name", new
             * List[] { sem1, sem2 }, new String[]{"dim1", "dim2"});
             * m.setValue(0, 0, 1); m.setValue(0, 1, 2); m.setValue(0, 2, 3);
             */

            MatrixND m = MatrixFactory.getInstance().create("name",
                    new List[]{sem1, sem2, sem3},
                    new String[]{"dim1", "dim2", "dim3"});

            m.setValue(0, 0, 0, 1);
            m.setValue(0, 1, 0, 2);
            m.setValue(0, 0, 1, 3);

            ed.setMatrix(m);
            // ed.setEnabled(false);
        } catch (MatrixException e) {
            e.printStackTrace();
            System.exit(0);
        }

        // final MatrixPanelEditor mp = ed;
        frame.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosing(WindowEvent e) {
                System.exit(0);
            }
        });
        frame.pack();
        frame.setVisible(true);
    }
} // MatrixPanelEditor

