/*
 * #%L
 * JRedmine :: Client
 * 
 * $Id: RedmineXpp3HelperTest.java 238 2012-03-20 21:10:19Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.3/jredmine-client/src/test/java/org/nuiton/jredmine/model/io/xpp3/RedmineXpp3HelperTest.java $
 * %%
 * Copyright (C) 2009 - 2010 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.model.io.xpp3;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.codehaus.plexus.util.IOUtil;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.nuiton.jredmine.TestHelper;
import org.nuiton.jredmine.model.Attachment;
import org.nuiton.jredmine.model.Issue;
import org.nuiton.jredmine.model.IssueCategory;
import org.nuiton.jredmine.model.IssuePriority;
import org.nuiton.jredmine.model.IssueStatus;
import org.nuiton.jredmine.model.News;
import org.nuiton.jredmine.model.Project;
import org.nuiton.jredmine.model.RedmineModelEnum;
import org.nuiton.jredmine.model.TimeEntry;
import org.nuiton.jredmine.model.Tracker;
import org.nuiton.jredmine.model.User;
import org.nuiton.jredmine.model.Version;

import java.beans.Introspector;
import java.io.File;
import java.io.FileInputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0.0
 */
public class RedmineXpp3HelperTest {

    protected static final Log log =
            LogFactory.getLog(RedmineXpp3HelperTest.class);

    //TODO use the RedmineModelEnum and finish the model mock (miss some types)
    private static final RedmineModelEnum[] TYPE_ORDER = new RedmineModelEnum[]{
            RedmineModelEnum.attachment,
            RedmineModelEnum.issue,
            RedmineModelEnum.project,
            RedmineModelEnum.tracker,
            RedmineModelEnum.user,
            RedmineModelEnum.version,
            RedmineModelEnum.issueStatus,
            RedmineModelEnum.issuePriority,
            RedmineModelEnum.issueCategory,
            RedmineModelEnum.news,
            RedmineModelEnum.timeEntry
    };

    protected static File testDir;

    protected static RedmineXpp3Helper builder;

    File file;

    protected boolean verbose;

    protected static int universeSize;

    @BeforeClass
    public static void setUpClass() throws Exception {

        testDir = TestHelper.getTestDir(RedmineXpp3HelperTest.class.getPackage().getName(), "target" + File.separator + "test-classes");

        builder = new RedmineXpp3Helper();

        log.info("test dir : " + TestHelper.getRelativePath(TestHelper.getBasedir(), testDir));

        // setup memory model
        TestHelper.loadMemoryModel();

        universeSize = TYPE_ORDER.length;

        if (universeSize < RedmineModelEnum.values().length) {
            log.warn("There is some types of the model which are not tested :");
            Set<RedmineModelEnum> keySet = TestHelper.getMemoryModel().keySet();
            for (RedmineModelEnum t : RedmineModelEnum.values()) {
                if (!keySet.contains(t)) {
                    log.warn(t);
                }
            }
        }

    }

    @AfterClass
    public static void tearDownClass() throws Exception {
        testDir = null;
        builder = null;
    }

    @Before
    public void setUp() {
        verbose = TestHelper.isVerbose();
    }

    @After
    public void tearDown() {
        file = null;
    }

    /**
     * Test the method {@code read} of class ModelBuilder.
     *
     * @throws Exception for any exceptions
     */
    @Test
    public void testRead() throws Exception {

        TestAction action = new TestAction() {

            @Override
            protected <T> void runForType(File rootDir, Class<T> type, List<?> expected) throws Exception {

                File file = new File(rootDir, Introspector.decapitalize(type.getSimpleName()) + ".xml");

                if (verbose) {
                    log.info("will test " + TestHelper.getRelativePath(TestHelper.getBasedir(), file));
                }

                // test from file

                T actual = builder.readObject(type, file, true);

                TestHelper.assertMyEquals(type, expected.get(0), actual);

                // test from string

                FileInputStream input = new FileInputStream(file);
                try {
                    String txt = IOUtil.toString(input);

                    actual = builder.readObject(type, txt, true);

                    TestHelper.assertMyEquals(type, expected.get(0), actual);
                } finally {
                    input.close();
                }
            }
        };

        action.run("single",
                   Arrays.asList(TestHelper.getModel(Attachment.class).get(0)),
                   Arrays.asList(TestHelper.getModel(Issue.class).get(0)),
                   Arrays.asList(TestHelper.getModel(Project.class).get(0)),
                   Arrays.asList(TestHelper.getModel(Tracker.class).get(0)),
                   Arrays.asList(TestHelper.getModel(User.class).get(0)),
                   Arrays.asList(TestHelper.getModel(Version.class).get(0)),
                   Arrays.asList(TestHelper.getModel(IssueStatus.class).get(0)),
                   Arrays.asList(TestHelper.getModel(IssuePriority.class).get(0)),
                   Arrays.asList(TestHelper.getModel(IssueCategory.class).get(0)),
                   Arrays.asList(TestHelper.getModel(News.class).get(0)),
                   Arrays.asList(TestHelper.getModel(TimeEntry.class).get(0))
        );

        Issue issue = TestHelper.getModel(Issue.class).get(0);
        User user = TestHelper.getModel(User.class).get(0);
        issue.setIsPrivate(true);
        user.setSalt("38006729a049cd820aafd6c2bb3b193f");
        try {
            action.run("single2",
                       null,
                       Arrays.asList(issue),
                       null,
                       null,
                       Arrays.asList(user),
                       null,
                       null,
                       null,
                       null,
                       null,
                       null
            );
        } finally {
            issue.setIsPrivate(false);
            user.setSalt(null);
        }
    }

    /**
     * Test the method {@code readArray} of class ModelBuilder.
     *
     * @throws Exception for any exceptions
     */
    @Test
    @SuppressWarnings("unchecked")
    public void testReadArray() throws Exception {

        TestAction action = new TestAction() {

            @Override
            protected <T> void runForType(File rootDir, Class<T> type, List<?> expected) throws Exception {

                File file = new File(rootDir, Introspector.decapitalize(type.getSimpleName()) + "s.xml");

                if (verbose) {
                    log.info("will test " + TestHelper.getRelativePath(TestHelper.getBasedir(), file));
                }

                // test from file

                T[] actual = builder.readObjects(type, file, true);

                TestHelper.assertMyListEquals(type, (List<T>) expected, actual);

                FileInputStream input = new FileInputStream(file);
                try {
                    // test from text
                    String txt = IOUtil.toString(input);

                    actual = builder.readObjects(type, txt, true);

                    TestHelper.assertMyListEquals(type, (List<T>) expected, actual);
                } finally {
                    input.close();
                }
            }
        };

        action.run("array-empty",
                   Collections.emptyList(),
                   Collections.emptyList(),
                   Collections.emptyList(),
                   Collections.emptyList(),
                   Collections.emptyList(),
                   Collections.emptyList(),
                   Collections.emptyList(),
                   Collections.emptyList(),
                   Collections.emptyList(),
                   Collections.emptyList(),
                   Collections.emptyList());


        action.run("array-singleton",
                   Arrays.asList(TestHelper.getModel(Attachment.class).get(0)),
                   Arrays.asList(TestHelper.getModel(Issue.class).get(0)),
                   Arrays.asList(TestHelper.getModel(Project.class).get(0)),
                   Arrays.asList(TestHelper.getModel(Tracker.class).get(0)),
                   Arrays.asList(TestHelper.getModel(User.class).get(0)),
                   Arrays.asList(TestHelper.getModel(Version.class).get(0)),
                   Arrays.asList(TestHelper.getModel(IssueStatus.class).get(0)),
                   Arrays.asList(TestHelper.getModel(IssuePriority.class).get(0)),
                   Arrays.asList(TestHelper.getModel(IssueCategory.class).get(0)),
                   Arrays.asList(TestHelper.getModel(News.class).get(0)),
                   Arrays.asList(TestHelper.getModel(TimeEntry.class).get(0))
        );

        action.run("array-multi",
                   TestHelper.getModel(Attachment.class),
                   TestHelper.getModel(Issue.class),
                   TestHelper.getModel(Project.class),
                   TestHelper.getModel(Tracker.class),
                   TestHelper.getModel(User.class),
                   TestHelper.getModel(Version.class),
                   TestHelper.getModel(IssueStatus.class),
                   TestHelper.getModel(IssuePriority.class),
                   TestHelper.getModel(IssueCategory.class),
                   TestHelper.getModel(News.class),
                   TestHelper.getModel(TimeEntry.class)
        );

    }

    /** A little action wrapper to simplify this test class :) */
    public static abstract class TestAction {

        /**
         * Implements the logic of action for the given {@code type} and
         * the {@code expected} datas.
         *
         * @param <T>      the type of data used
         * @param rootDir  the rootdir file of the local cache
         * @param type     the type of data used
         * @param expected the list of expected datas
         * @throws Exception if any pb
         */
        protected abstract <T> void runForType(File rootDir, Class<T> type, List<?> expected) throws Exception;

        public void run(String name, List<?>... expected) throws Exception {
            Assert.assertEquals(universeSize, expected.length);
            File rootDir = new File(testDir, name);

            for (int i = 0; i < universeSize; i++) {

                RedmineModelEnum next = TYPE_ORDER[i];

                if (expected[i] != null) {
                    runForType(rootDir, next.getModelType(), expected[i]);
                }
            }
        }
    }
}
