/*
 * #%L
 * JAXX :: Widgets
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package jaxx.runtime.swing.model;

import com.google.common.collect.Lists;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.Collection;
import java.util.List;
import javax.swing.ComboBoxModel;
import javax.swing.DefaultListModel;
import javax.swing.DefaultListSelectionModel;
import javax.swing.event.EventListenerList;
import javax.swing.event.ListDataListener;

/**
 * @author sletellier <letellier@codelutin.com>
 */
public class GenericListModel<B> extends DefaultListSelectionModel implements ComboBoxModel {

    protected EventListenerList listenerList = new EventListenerList();
    protected final PropertyChangeSupport pcs = new PropertyChangeSupport(this);

    public static final String PROPERTY_SELECTED_VALUE = "selectedValues";

    protected DefaultListModel delegate = new DefaultListModel();

    protected List<B> selectedValues;

    public GenericListModel() {
        this.selectedValues = Lists.newArrayList();
    }

    public GenericListModel(Collection<B> values) {
        this();
        setElements(values);
    }

    public void setElements(Collection<B> values) {
        Collection<B> oldValues = getElements();
        Collection<B> oldSelectedValues = getSelectedValues();
        clearSelection();
        fireSelectionRemoved(oldSelectedValues);

        clearElements();
        fireValuesRemoved(oldValues);

        for (B value : values) {
            delegate.addElement(value);
        }

        fireSelectionAdded(values);
    }

    public void clearElements() {
        Collection<B> elements = getElements();
        delegate.clear();

        fireValuesRemoved(elements);
    }

    public Collection<B> getElements() {
        int size = delegate.getSize();
        Collection<B> result = Lists.newArrayList();
        for (int i=0;i<size;i++) {
            result.add((B) delegate.get(i));
        }
        return result;
    }

    public void addElement(int index, B valueToAdd) {
        delegate.add(index, valueToAdd);

        fireValuesAdded(Lists.newArrayList(valueToAdd));
    }

    public void addElement(B valueToAdd) {
        delegate.addElement(valueToAdd);

        fireValuesAdded(Lists.newArrayList(valueToAdd));
    }

    public void addElements(Collection<B> valuesToAdd) {
        for (B value : valuesToAdd) {
            delegate.addElement(value);
        }

        fireValuesAdded(valuesToAdd);
    }

    public B getSelectedValue() {
        return selectedValues.get(0);
    }

    public Collection<B> getSelectedValues() {
        return Lists.newArrayList(selectedValues);
    }

    public void removeElements(Collection<B> values) {
        for (B value : values) {
            delegate.removeElement(value);
        }
        unSelectItems(values);

        fireValuesRemoved(values);
    }

    protected void unSelectItems(Collection<B> values) {
        Collection<B> oldValue = Lists.newArrayList(selectedValues);
        for (B value : values) {
            int index = selectedValues.indexOf(value);
            removeSelectionIntervalWithoutFire(index, index);
        }
        fireSelectionRemoved(values);
        firePropertyChange(PROPERTY_SELECTED_VALUE, oldValue, selectedValues);
    }

    protected void unSelectItem(B value) {
        Collection<B> oldValue = Lists.newArrayList(selectedValues);
        int index = selectedValues.indexOf(value);
        removeSelectionIntervalWithoutFire(index, index);

        fireSelectionRemoved(Lists.newArrayList(value));
        firePropertyChange(PROPERTY_SELECTED_VALUE, oldValue, selectedValues);
    }

    public void addSelectedItem(B toSelect) {
        Collection<B> oldValue = Lists.newArrayList(selectedValues);
        selectedValues.add(toSelect);
        int index = selectedValues.indexOf(toSelect);
        super.addSelectionInterval(index, index);

        fireSelectionAdded(Lists.newArrayList(toSelect));
        firePropertyChange(PROPERTY_SELECTED_VALUE, oldValue, selectedValues);
    }

    public boolean hasSelectedIndex() {
        return !selectedValues.isEmpty();
    }

    @Override
    public void addSelectionInterval(int index0, int index1) {
        Collection<B> oldValue = Lists.newArrayList(selectedValues);

        addSelectionIntervalWithFire(index0, index1);
        super.addSelectionInterval(index0, index1);

        Collection<B> newValue = Lists.newArrayList(selectedValues);
        newValue.removeAll(oldValue);
        fireSelectionAdded(newValue);
        firePropertyChange(PROPERTY_SELECTED_VALUE, oldValue, selectedValues);
    }

    @Override
    public void setSelectionInterval(int index0, int index1) {
        Collection<B> oldValue = Lists.newArrayList(selectedValues);
        selectedValues.clear();
        addSelectionIntervalWithFire(index0, index1);
        super.setSelectionInterval(index0, index1);

        Collection<B> newValue = Lists.newArrayList(selectedValues);
        newValue.removeAll(oldValue);
        fireSelectionAdded(newValue);
        firePropertyChange(PROPERTY_SELECTED_VALUE, oldValue, selectedValues);
    }

    @Override
    public void removeSelectionInterval(int index0, int index1) {
        Collection<B> oldValue = Lists.newArrayList(selectedValues);
        removeSelectionIntervalWithoutFire(index0, index1);

        Collection<B> newValue = Lists.newArrayList(selectedValues);
        newValue.removeAll(oldValue);
        fireSelectionRemoved(newValue);
        firePropertyChange(PROPERTY_SELECTED_VALUE, oldValue, selectedValues);
    }

    protected void removeSelectionIntervalWithoutFire(int index0, int index1) {
        if (index0 > index1) {
            int tmp = index1;
            index1 = index0;
            index0 = tmp;
        }
        for (int i=index0;i<=index1;i++) {
            if (selectedValues.size() > i && i != -1) {
                selectedValues.remove(i);
            }
        }
        super.removeSelectionInterval(index0, index1);
    }

    protected void addSelectionIntervalWithFire(int index0, int index1) {
        if (index0 > index1) {
            int tmp = index1;
            index1 = index0;
            index0 = tmp;
        }
        for (int i=index0;i<=index1;i++) {

            if (delegate.size() > i && i != -1) {
                B value = (B)delegate.getElementAt(i);
                selectedValues.add(value);
            }
        }
    }

    @Override
    public void clearSelection() {
        Collection<B> oldValue = Lists.newArrayList(selectedValues);
        selectedValues.clear();
        super.clearSelection();

        fireSelectionRemoved(oldValue);
        firePropertyChange(PROPERTY_SELECTED_VALUE, oldValue, selectedValues);
    }

    @Override
    public int getSelectionMode() {
        return MULTIPLE_INTERVAL_SELECTION;
    }

    @Override
    public void setSelectedItem(Object anItem) {
        Collection<B> oldValue = this.selectedValues;
        fireSelectionRemoved(selectedValues);

        selectedValues = Lists.newArrayList((B)anItem);

        fireSelectionAdded(selectedValues);
        firePropertyChange(PROPERTY_SELECTED_VALUE, oldValue, selectedValues);
    }

    @Override
    public Object getSelectedItem() {
        if (selectedValues.isEmpty()) {
            return null;
        }
        return selectedValues.get(0);
    }

    @Override
    public int getSize() {
        return delegate.size();
    }

    @Override
    public Object getElementAt(int index) {
        return delegate.get(index);
    }

    @Override
    public void addListDataListener(ListDataListener l) {
        delegate.addListDataListener(l);
    }

    @Override
    public void removeListDataListener(ListDataListener l) {
        delegate.removeListDataListener(l);
    }

    public void addGenericListListener(GenericListListener l) {
        listenerList.add(GenericListListener.class, l);
    }

    public void removeGenericListListener(GenericListListener l) {
        listenerList.remove(GenericListListener.class, l);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        pcs.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        pcs.removePropertyChangeListener(propertyName, listener);
    }

    protected void fireValuesAdded(Collection<B> values) {
        if (values.isEmpty()) {
            return;
        }
        Object[] listeners = listenerList.getListenerList();
        GenericListEvent<B> e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == GenericListListener.class) {
                if (e == null) {
                    e = new GenericListEvent<B>(this, values);
                }
                ((GenericListListener)listeners[i+1]).valuesAdded(e);
            }
        }
    }

    protected void fireValuesRemoved(Collection<B> values) {
        if (values.isEmpty()) {
            return;
        }
        Object[] listeners = listenerList.getListenerList();
        GenericListEvent<B> e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == GenericListListener.class) {
                if (e == null) {
                    e = new GenericListEvent<B>(this, values);
                }
                ((GenericListListener)listeners[i+1]).valuesRemoved(e);
            }
        }
    }

    protected void fireSelectionAdded(Collection<B> selectedValues) {
        if (selectedValues.isEmpty()) {
            return;
        }
        Object[] listeners = listenerList.getListenerList();
        GenericListEvent<B> e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == GenericListListener.class) {
                if (e == null) {
                    e = new GenericListEvent<B>(this, selectedValues);
                }
                ((GenericListListener)listeners[i+1]).selectionAdded(e);
            }
        }
    }

    protected void fireSelectionRemoved(Collection<B> selectedValues) {
        if (selectedValues.isEmpty()) {
            return;
        }
        Object[] listeners = listenerList.getListenerList();
        GenericListEvent<B> e = null;

        for (int i = listeners.length - 2; i >= 0; i -= 2) {
            if (listeners[i] == GenericListListener.class) {
                if (e == null) {
                    e = new GenericListEvent<B>(this, selectedValues);
                }
                ((GenericListListener)listeners[i+1]).selectionAdded(e);
            }
        }
    }

    protected void firePropertyChange(String propertyName, Object oldValue, Object newValue) {
        pcs.firePropertyChange(propertyName, oldValue, newValue);
    }
}