/*
 * *##%
 * JAXX Runtime
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.runtime.swing.navigation;

import jaxx.runtime.JAXXAction;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.decorator.Decorator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Enumeration;

/**
 * This object is design to build a {@link NavigationTreeTableModel}.
 *
 * @see NavigationModelBuilder
 *
 * @author sletellier
 * @since 2.0.0
 */
public abstract class NavigationTreeTableModelBuilder implements NavigationModelBuilder{

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(NavigationTreeModelBuilder.class);
    /**
     * The model dealed by the builder.
     *
     * <b>Note:</b> It is a good idea to keep only one instance of the model.
     * If reset is required, should empty the model but not reinstanciate it.
     */
    protected NavigationTreeTableModel model;
    /**
     * default ui class to use if node does not define an ui class
     */
    protected Class<? extends JAXXObject> defaultUIClass;
    /**
     * [optional] default action class
     */
    protected Class<? extends JAXXAction> defaultUIHandlerClass;

    public NavigationTreeTableModelBuilder(Class<? extends JAXXObject> defaultUIClass, Class<? extends JAXXAction> defaultUIHandlerClass, NavigationTreeTableModel model) {
        this.defaultUIClass = defaultUIClass;
        this.defaultUIHandlerClass = defaultUIHandlerClass;
        this.model = model;
    }

    public NavigationTreeTableModel getModel() {
        return model;
    }

    public NavigationTreeNode buildEmptyRoot(JAXXContextEntryDef<?> entryDef, String contextName) {
        NavigationTreeTableNode node = createNavigationTreeTableNode(model.pathSeparator, contextName, entryDef, null);
        addI18nNodeRenderer(node, "");
        return addChildNode(null, node);
    }

    public NavigationTreeNode build(NavigationTreeNode parentNode, String libelle,
            JAXXContextEntryDef<?> entryDef,
            String entryPath,
            String contextName,
            Class<? extends JAXXObject> uiClass,
            Class<? extends JAXXAction> actionClass) {
        NavigationTreeTableNode node = createNavigationTreeTableNode(model.pathSeparator, contextName, entryDef, entryPath);
        addI18nNodeRenderer(node, libelle);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    public NavigationTreeNode build(NavigationTreeNode parentNode, String libelle,
            JAXXContextEntryDef<?> entryDef,
            String contextName,
            Class<? extends JAXXObject> uiClass,
            Class<? extends JAXXAction> actionClass) {
        NavigationTreeTableNode node = createNavigationTreeTableNode(model.pathSeparator, contextName, entryDef, null);
        addI18nNodeRenderer(node, libelle);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    public NavigationTreeNode build(NavigationTreeNode parentNode, String libelle,
            String entryPath,
            String contextName,
            Class<? extends JAXXObject> uiClass,
            Class<? extends JAXXAction> actionClass) {
        NavigationTreeTableNode node = createNavigationTreeTableNode(model.pathSeparator, contextName, null, entryPath);
        addI18nNodeRenderer(node, libelle);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    public NavigationTreeNode build(NavigationTreeNode parentNode, Decorator<?> decorator,
            JAXXContextEntryDef<?> entryDef,
            String entryPath,
            String contextName,
            Class<? extends JAXXObject> uiClass,
            Class<? extends JAXXAction> actionClass) {
        NavigationTreeTableNode node = createNavigationTreeTableNode(model.pathSeparator, contextName, entryDef, entryPath);
        addDecoratorNodeRenderer(node, decorator);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    public NavigationTreeNode build(NavigationTreeNode parentNode, Decorator<?> decorator,
            JAXXContextEntryDef<?> entryDef,
            String contextName,
            Class<? extends JAXXObject> uiClass,
            Class<? extends JAXXAction> actionClass) {
        NavigationTreeTableNode node = createNavigationTreeTableNode(model.pathSeparator, contextName, entryDef, null);
        addDecoratorNodeRenderer(node, decorator);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    public NavigationTreeNode build(NavigationTreeNode parentNode, Decorator<?> decorator,
            String entryPath,
            String contextName,
            Class<? extends JAXXObject> uiClass,
            Class<? extends JAXXAction> actionClass) {
        NavigationTreeTableNode node = createNavigationTreeTableNode(model.pathSeparator, contextName, null, entryPath);
        addDecoratorNodeRenderer(node, decorator);
        addNodeJaxxClasses(node, uiClass, actionClass);
        return addChildNode(parentNode, node);
    }

    protected NavigationTreeNode addChildNode(NavigationTreeNode parentNode, NavigationTreeTableNode node) {

        if (node.getUIClass() == null) {
            // no ui is associated with this node, use the default one
            node.setUIClass(defaultUIClass);
        }

        if (node.getUIHandlerClass() == null) {
            // no ui handler associated with this node, use the default one
            node.setUIHandlerClass(defaultUIHandlerClass);
        }
        if (parentNode == null) {
            model.setRoot(node);
        } else {
            parentNode.add(node);
        }
        model.nodeStructureChanged(node);
        return node;
    }

    public NavigationTreeNode removeChildNode(NavigationTreeNode node) {
        NavigationTreeNode parentNode = node.getParent();
        model.removeNodeFromParent(node);
        return parentNode;
    }

    public void addI18nNodeRenderer(NavigationTreeNode node, String libelle) {
        node.setRenderer(new NavigationTreeNodeRendererI18nImpl(libelle));
    }

    public void addDecoratorNodeRenderer(NavigationTreeNode node, Decorator<?> decorator) {
        node.setRenderer(new NavigationTreeNodeRendererDecoratorImpl(decorator));
    }

    public void addNodeJaxxClasses(
            NavigationTreeNode node,
            Class<? extends JAXXObject> uIClass,
            Class<? extends JAXXAction> uIHandlerClass) {
        node.setUIClass(uIClass);
        node.setUIHandlerClass(uIHandlerClass);
    }

    public void printModel(NavigationTreeNode node) {
        if (node == null) {
            return;
        }
        log.info("node " + node.getFullPath() + ", jxpath: " + node.getJaxxContextEntryPath() + ", entryContextDef: " + node.getJaxxContextEntryDef());
        if (log.isDebugEnabled()) {
            log.debug("node userObject" + node.getUserObject());
            log.debug("value from node " + node.getBean(getModel().getContext()));
            log.debug("value from model " + getModel().getBean(node));
        }
        Enumeration<?> children = node.children();
        while (children.hasMoreElements()) {
            printModel((NavigationTreeNode) children.nextElement());
        }
    }

    // To create your own instance of NavigationTreeTableNode
    public abstract NavigationTreeTableNode createNavigationTreeTableNode(String pathSeparator, String contextName, JAXXContextEntryDef<?> jaxxContextEntryDef, String jaxxContextEntryPath);

    public static abstract class ChildBuilder<O> {

        protected NavigationTreeModelBuilder builder;

        protected ChildBuilder(NavigationTreeModelBuilder builder) {
            this.builder = builder;
        }

        protected abstract void init(Class<? extends O> klass);

        protected abstract Decorator<? extends O> getDecorator(O child);

        protected abstract String getJXPath(O child);

        protected abstract String getNavigationPath(O child);

        public void build(NavigationTreeNode parent, boolean cacheValues, Class<? extends O> klass, java.util.Collection<? extends O> beans, Class<? extends JAXXObject> ui, Class<? extends JAXXAction> actionClass) {

            if (beans == null || beans.isEmpty()) {
                // no bean to treate
                return;
            }

            init(klass);

            NavigationTreeNode node;

            for (O o : beans) {
                node = builder.build(parent, getDecorator(o), getJXPath(o), getNavigationPath(o), ui, actionClass);
                if (cacheValues) {
                    // cache the bean value to improve performance
                    node.setBean(o);
                }
            }
        }

        public void build(NavigationTreeNode parent, boolean cacheValues, Class<? extends O> klass, O[] beans, Class<? extends JAXXObject> ui, Class<? extends JAXXAction> actionClass) {

            if (beans == null || beans.length == 0) {
                // no bean to treate
                return;
            }

            init(klass);

            NavigationTreeNode node;

            for (O o : beans) {
                node = builder.build(parent, getDecorator(o), getJXPath(o), getNavigationPath(o), ui, actionClass);
                if (cacheValues) {
                    // cache the bean value to improve performance
                    node.setBean(o);
                }
            }
        }
    }
}
