package jaxx.runtime;

import java.util.Collections;
import java.util.List;

/**
 * To qualify an entry in a {@link JAXXContext}.
 * <p/>
 * Use the factory methods <code>newDef</code> and <code>newListDef</code< to obtain new instances.
 *
 * @author chemit
 */
public class JAXXContextEntryDef<O> implements java.io.Serializable {

    /** name of the entry, can be nuill for a unamed entry. */
    protected String name;

    /** class of the entry, can not be null */
    protected Class<O> klass;

    private static final long serialVersionUID = 1L;

    public static <O> JAXXContextEntryDef<O> newDef(Class<O> klass) {
        return newDef(null, klass);
    }

    public static <O> JAXXContextEntryDef<O> newDef(String name, Class<O> klass) {
        return new JAXXContextEntryDef<O>(name, klass);
    }

    public static <O> JAXXContextEntryDef<List<O>> newListDef() {
        return newListDef(null);
    }

    @SuppressWarnings({"unchecked"})
    public static <O> JAXXContextEntryDef<List<O>> newListDef(String name) {
        JAXXContextEntryDef contextEntryDef = new JAXXContextEntryDef<List<O>>(name, JAXXContextEntryDef.<O>castList());
        contextEntryDef.klass = List.class;
        return contextEntryDef;
    }


    public String getName() {
        return name;
    }

    public Class<O> getKlass() {
        return klass;
    }

    public O getContextValue(JAXXContext context) {
        return context.getContextValue(klass, name);
    }

    public void removeContextValue(JAXXContext context) {
        context.removeContextValue(klass, name);
    }

    public void setContextValue(JAXXContext context, O value) {
        context.setContextValue(value, name);
    }

    @Override
    public String toString() {
        return super.toString() + "<" + klass + ":" + name + ">";
    }

    protected JAXXContextEntryDef(Class<O> klass) {
        this(null, klass);
    }

    protected JAXXContextEntryDef(String name, Class<O> klass) {
        if (klass == null) {
            throw new IllegalArgumentException("class can not be null");
        }
        this.name = name;
        this.klass = klass;
    }

    @SuppressWarnings({"unchecked"})
    protected static <O> Class<List<O>> castList() {
        return (Class<List<O>>) Collections.emptyList().getClass();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof JAXXContextEntryDef)) {
            return false;
        }
        JAXXContextEntryDef that = (JAXXContextEntryDef) o;
        return klass.equals(that.klass) && !(name != null ? !name.equals(that.name) : that.name != null);

    }

    @Override
    public int hashCode() {
        int result = (name != null ? name.hashCode() : 0);
        return 31 * result + klass.hashCode();
    }

    public boolean accept(Class<?> klass, String name) {
        if (klass == Object.class && this.klass != Object.class) {
            // try on name only
            return (this.name != null && name != null && this.name.equals(name));
        }
        return klass.isAssignableFrom(this.klass) && (this.name == null && name == null
                || (this.name != null && name != null && this.name.equals(name)));
    }

    public boolean accept2(Class<?> klass, String name) {
        return !(klass == Object.class && this.klass != Object.class) &&
                this.klass.isAssignableFrom(klass) && (this.name == null &&
                name == null || (this.name != null && name != null && this.name.equals(name)));
    }
}
