/**
 * *##% 
 * JAXX Demo
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.demo;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.DefaultApplicationContext;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.context.JAXXInitialContext;
import jaxx.runtime.decorator.DecoratorProvider;
import jaxx.runtime.swing.AboutPanel;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.editor.config.ConfigUI;
import jaxx.runtime.swing.editor.config.ConfigUIBuilder;
import jaxx.runtime.swing.editor.config.model.ConfigUIModel;
import jaxx.runtime.swing.navigation.NavigationTreeNode;
import jaxx.runtime.swing.renderer.DecoratorProviderListCellRenderer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

import javax.swing.*;
import java.awt.*;
import java.net.URL;
import java.util.Locale;

/**
 * Le handler de l'ui principale.
 *
 * @author chemit
 * @see DemoUI
 */
public class DemoUIHandler { //implements JAXXHelp {

    /**
     * to use log facility, just put in your code: log.info(\"...\");
     */
    static private Log log = LogFactory.getLog(DemoUIHandler.class);
    static final JAXXContextEntryDef<DemoUI> MAIN_UI_ENTRY_DEF = jaxx.runtime.Util.newContextEntryDef("mainui", DemoUI.class);

    /**
     * Methode pour initialiser l'ui principale sans l'afficher.
     *
     * @param rootContext le context applicatif
     * @param fullscreen  flag pour indiquer si on doit ouvrir l'ui en model console (pleine ecran).
     * @return l'ui instancie et initialisee mais non visible encore
     */
    public DemoUI initUI(DefaultApplicationContext rootContext, boolean fullscreen) {

        if (log.isDebugEnabled()) {
            log.debug("fullscreen ? " + fullscreen);
        }

        // create restrict context for ui 
        JAXXInitialContext context = new JAXXInitialContext();

        // share handler
        context.add(this);

        // share tree helper
        context.add(rootContext.getContextValue(DemoTreeHelper.class));

        // share config
        context.add(rootContext.getContextValue(DemoConfig.class));

        // share a unique DecoratorProviderListCellRenderer
        DecoratorProvider decoratorProvider = rootContext.getContextValue(DecoratorProvider.class);

        context.add(decoratorProvider);
        
        context.add(new DecoratorProviderListCellRenderer(decoratorProvider));

        // instanciate ui
        DemoUI ui = new DemoUI(context);

        // keep it in root context
        MAIN_UI_ENTRY_DEF.setContextValue(rootContext, ui);

        // synch to error dialog
        ErrorDialogUI.init(ui);

        // set fullscreen propery on main ui
        ui.getGraphicsConfiguration().getDevice().setFullScreenWindow(fullscreen ? ui : null);

        return ui;
    }

    public void displayUI(final DemoUI ui, final String nodePath) {

        // expend all nodes of the demo tree
        jaxx.runtime.SwingUtil.expandTree(ui.getNavigation());

        SwingUtilities.invokeLater(new Runnable() {

            @Override
            public void run() {

                String path;
                if (nodePath == null) {
                    // take first node
                    path = "$root/jaxxdemo.tree.component.swing";
                } else {
                    // take selected node
                    path = nodePath;
                }
                log.info("node to re select " + path);
                // select node
                ui.getTreeHelper().selectNode(ui, path);

                // use best dimensions
                ui.getSplitPane().resetToPreferredSizes();
            }
        });

        // show ui after all (in another invocation, tu avoid layout adjustement
        // to be seen).

        SwingUtilities.invokeLater(new Runnable() {

            @Override
            public void run() {

                // show ui
                ui.setVisible(true);
            }
        });
    }

    /**
     * Permet de recharger l'ui principale et de changer de le mode d'affichage.
     *
     * @param rootContext le contexte applicatif
     * @param fullscreen  le type de fenetre à reouvrir
     */
    public void reloadUI(DefaultApplicationContext rootContext, boolean fullscreen) {

        // scan main ui
        DemoUI ui = getUI(rootContext);

        NavigationTreeNode node;
        String nodePath = null;

        if (ui != null) {

            ui.getConfig().removeJaxxPropertyChangeListener();

            node = ui.getTreeHelper().getSelectedNode(ui);

            if (node != null) {
                nodePath = node.getFullPath();
                log.info("selected node " + nodePath);
            }

            ErrorDialogUI.init(null);

            ui.dispose();

            ui.setVisible(false);

            MAIN_UI_ENTRY_DEF.removeContextValue(rootContext);

        }

        ui = initUI(rootContext, fullscreen);

        displayUI(ui, nodePath);
    }

    /**
     * Méthode pour changer de mode d'affichage.
     * <p/>
     * Si <code>fullscreen</code> est à <code>true</code> alors on passe en
     * mode console (c'est à dire en mode plein écran exclusif), sinon on
     * passe en mode fenetré normal.
     *
     * @param context    l'ui principale de l'application
     * @param fullscreen le nouvel état requis.
     */
    public void changeScreen(JAXXContext context, final boolean fullscreen) {

        DemoUI ui = getUI(context);

        // sauvegarde de l'état dans la configuration
        ui.getConfig().setFullscreen(fullscreen);

        // rechargement de l'ui
        reloadUI(RunDemo.get(), fullscreen);
    }

    public void changeLanguage(JAXXContext context, Locale newLocale) {

        DemoUI ui = getUI(context);

        DemoConfig config = ui.getConfig();

        // sauvegarde de la nouvelle locale
        config.setLocale(newLocale);

        // chargement de la nouvelle locale dans le système i18n
        I18n.init(newLocale);

        // rechargement de l'ui
        reloadUI(RunDemo.get(), config.isFullScreen());
    }

    /**
     * Ferme l'application.
     *
     * @param context l'ui principale de l'application
     */
    public void close(JAXXContext context) {
        log.info("JAXX Demo quitting...");
        try {

            DemoUI ui = getUI(context);
            ui.dispose();
        } finally {
            System.exit(0);
        }
    }

    public void showConfig(JAXXContext context) {
        DemoUI ui = getUI(context);
        DemoConfig config = ui.getConfig();
        ConfigUIModel model = new ConfigUIModel(config);

        // categorie repertoires
        model.addCategory(
                n_("jaxxdemo.config.category.directories"),
                n_("jaxxdemo.config.category.directories.description"),
                DemoConfig.Option.CONFIG_FILE);

        // others
        model.addCategory(
                n_("jaxxdemo.config.category.other"),
                n_("jaxxdemo.config.category.other.description"),
                DemoConfig.Option.FULL_SCREEN,
                DemoConfig.Option.LOCALE);

        ConfigUI configUI = ConfigUIBuilder.newConfigUI(context, model, "jaxxdemo.config.category.directories");

        ConfigUIBuilder.showConfigUI(configUI, ui, false);
    }

    //    @Override
    public void showHelp(JAXXContext context, String helpId) {
//        DemoUI mainUI = getUI(context);
//        ObserveHelpBroker helpBroker = context.getContextValue(ObserveHelpBroker.class);
//
//        if (helpId == null) {
//            helpId = helpBroker.getDefaultID();
//        }
//        log.debug("show help " + helpId);
//        mainUI.getHelp().setCurrentID(helpId);
    }

    public void gotoSite(JAXXContext context) {

        DemoUI ui = getUI(context);
        DemoConfig config = ui.getConfig();

        URL siteURL = config.getOptionAsURL("application.site.url");

        log.info(_("jaxxdemo.message.goto.site", siteURL));

        if (log.isDebugEnabled()) {
            log.debug("goto " + siteURL);
        }
        if (Desktop.isDesktopSupported() && Desktop.getDesktop().isSupported(Desktop.Action.BROWSE)) {
            try {
                Desktop.getDesktop().browse(siteURL.toURI());
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                ErrorDialogUI.showError(ex);
            }
        }
    }

    public void showAbout(DemoUI ui) {

        DemoConfig config = ui.getConfig();

        String iconPath = config.getOption("application.icon.path");
        String licensePath = config.getOption("application.license.path");
        String thirdPartyPath = config.getOption("application.third-party.path");

        AboutPanel about = new AboutPanel();
        about.setTitle(_("jaxxdemo.title.about"));
        about.setAboutText(_("jaxxdemo.about.message"));
        about.setBottomText(ui.getConfig().getCopyrightText());
        about.setIconPath(iconPath);
        about.setLicenseFile(licensePath);
        about.setThirdpartyFile(thirdPartyPath);
        about.buildTopPanel();
        about.init();
        about.showInDialog(ui, true);
    }

    DemoUI getUI(JAXXContext context) {
        if (context instanceof DemoUI) {
            return (DemoUI) context;
        }
        DemoUI ui = MAIN_UI_ENTRY_DEF.getContextValue(context);
        return ui;
    }
}
