/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.java;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;

/**
 * Represents a method in a Java source file being generated for output.  <code>JavaMethods</code> are created
 * and added to a {@link JavaFile}, which can then output Java source code.  In addition to normal methods, a
 * <code>JavaMethod</code> can represent a constructor -- constructors should be named after their containing
 * classes and have a return type of <code>null</code>.
 */
public class JavaMethod extends JavaElement implements Comparable<JavaMethod> {

    /**
     * Method comparator
     */
    static final JavaMethodComparator COMPARATOR = new JavaMethodComparator();
    /**
     * return type of the method (null for constructors)
     */
    private String returnType;
    /**
     * arguments of the method (can be empty)
     */
    private JavaArgument[] arguments;
    /**
     * exceptions thrown by the method( can be empty)
     */
    private String[] exceptions;
    /**
     * body of the mehotd (can be empty)
     */
    private String body;
    /**
     * flag to known if the method overrids a super-method
     */
    private boolean override;

    /**
     * Constructs a new <code>JavaMethod</code> containing the specified body code.  The <code>modifiers</code> parameter
     * is a bit mask of the  constants from {@link java.lang.reflect.Modifier}, and the <code>returnType</code> and
     * <code>exceptions</code> of the method should be represented as they would appear in Java source code (<code>null</code>
     * for a constructor).  The method body is initially empty.
     *
     * @param modifiers  the modifier keywords that should appear as part of the method's declaration
     * @param returnType the return type of the method as it would appear in Java source code
     * @param name       the method's name
     * @param arguments  the method's arguments
     * @param exceptions a list of exceptions the methods can throw, as they would be represented in Java source code
     * @param bodyCode   Java source code which should appear in the method body
     * @param override flag with {@code true} value when the method overrides (or implements) a super class method
     */
    public JavaMethod(int modifiers, String returnType, String name, JavaArgument[] arguments, String[] exceptions, String bodyCode, boolean override) {
        super(modifiers, name);
        this.returnType = returnType;
        this.override = override;
        this.arguments = arguments;
        this.exceptions = exceptions;
        this.body = bodyCode == null ? "" : bodyCode;
    }

    /**
     * Returns the method's return type, as it would be represented in Java source code.
     *
     * @return the method's return type
     */
    public String getReturnType() {
        return returnType;
    }

    /**
     * Returns a list of the method's arguments.
     *
     * @return the method's arguments
     */
    public JavaArgument[] getArguments() {
        return arguments;
    }

    /**
     * Returns a list of exceptions the method can throw.
     *
     * @return the method's exceptions
     */
    public String[] getExceptions() {
        return exceptions;
    }

    public boolean isOverride() {
        return override;
    }

    public void setOverride(boolean override) {
        this.override = override;
    }

    public String getBody() {
        return body;
    }

    @Override
    public int compareTo(JavaMethod o) {
        return COMPARATOR.compare(this, o);
    }

    public enum MethodOrder {

        statics(Modifier.STATIC, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- Statics methods --------------------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {
        },
        constructors(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- Constructors -----------------------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            @Override
            public boolean accept(JavaMethod method) {
                return method.returnType == null;
            }
        },
        JAXXObject(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- JAXXObject implementation ----------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            private List<String> methods = Arrays.asList("applyDataBinding", "firePropertyChange", "getObjectById", "get$objectMap", "processDataBinding", "removeDataBinding");

            @Override
            public boolean accept(JavaMethod method) {
                return methods.contains(method.getName());
            }
        },
        JAXXContext(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- JAXXContext implementation ---------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            private List<String> methods = Arrays.asList("getContextValue", "getDelegateContext", "getParentContainer", "removeContextValue", "setContextValue");

            @Override
            public boolean accept(JavaMethod method) {
                return methods.contains(method.getName());
            }
        },
        JAXXValidation(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- JAXXValidation implementation ------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            private List<String> methods = Arrays.asList("getValidator", "getValidatorIds");

            @Override
            public boolean accept(JavaMethod method) {
                return methods.contains(method.getName());
            }
        },
        events(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- Event methods ----------------------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            @Override
            public boolean accept(JavaMethod method) {
                return (method.getName().startsWith("do") && method.getName().indexOf("__") > -1);
            }
        },
        publicGetters(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- public acessor methods -------------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            @Override
            public boolean accept(JavaMethod method) {
                return (method.getName().startsWith("get") || method.getName().startsWith("is"));
            }
        },
        publicSetters(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- public mutator methods -------------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            @Override
            public boolean accept(JavaMethod method) {
                return (method.getName().startsWith("set"));
            }
        },
        otherPublic(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- public mutator methods -------------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            @Override
            public boolean accept(int mod) {
                return super.accept(mod) && !Modifier.isStatic(mod);
            }
        },
        protectedGetters(Modifier.PROTECTED, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- protected acessors methods ---------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            @Override
            public boolean accept(JavaMethod method) {
                return (method.getName().startsWith("get") || method.getName().startsWith("is"));
            }
        },
        createMethod(Modifier.PROTECTED | Modifier.PRIVATE, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- ui creation methods ----------------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            @Override
            public boolean accept(JavaMethod method) {
                return method.getName().startsWith("create") || method.getName().startsWith("add") ||
                        method.getName().equals("$completeSetup") ||
                        method.getName().equals("$initialize");
            }
        },
        protecteds(Modifier.PROTECTED, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- Other protected methods ------------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {
        },
        packageLocal(0, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- Package methods --------------------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {

            @Override
            public boolean accept(int mod) {
                return !Modifier.isStatic(mod) && !Modifier.isPublic(mod) && !Modifier.isProtected(mod);
            }
        },
        privates(Modifier.PRIVATE, "/*---------------------------------------------------------------------------------*/\n" +
        "/*-- Private methods --------------------------------------------------------------*/\n" +
        "/*---------------------------------------------------------------------------------*/") {
        };
        private final String header;
        private int modifier;

        MethodOrder(int modifier, String header) {
            this.header = header;
            this.modifier = modifier;
        }

        public String getHeader() {
            return header;
        }

        public boolean accept(JavaMethod method) {
            return true;
        }

        public boolean accept(int mod) {
            return (mod & modifier) != 0;
        }

        public boolean accept(int mod, JavaMethod method) {
            return accept(mod) && accept(method);
        }

        public static MethodOrder valueOf(JavaMethod method, int scope) {
            for (MethodOrder o : values()) {
                if (o.accept(scope, method)) {
                    return o;
                }
            }
            throw new IllegalArgumentException("could not find a " + MethodOrder.class + " for method " + method);
        }
    }

    public static EnumMap<MethodOrder, List<JavaMethod>> getSortedMethods(List<JavaMethod> methods) {

        EnumMap<MethodOrder, List<JavaMethod>> result = new EnumMap<MethodOrder, List<JavaMethod>>(MethodOrder.class);
        for (MethodOrder methodOrder : MethodOrder.values()) {
            result.put(methodOrder, new ArrayList<JavaMethod>());
        }

        EnumSet<MethodOrder> allConstants = EnumSet.allOf(MethodOrder.class);
        List<JavaMethod> allMethods = new ArrayList<JavaMethod>(methods);
        int[] scopes = new int[]{Modifier.STATIC, Modifier.PUBLIC, Modifier.PROTECTED, Modifier.PRIVATE};
        for (int scope : scopes) {
            EnumSet<MethodOrder> constants = getMethodOrderScope(allConstants, scope);

            Iterator<JavaMethod> itMethods = allMethods.iterator();
            while (itMethods.hasNext()) {
                JavaMethod method = itMethods.next();
                for (MethodOrder constant : constants) {
                    if (constant.accept(method.getModifiers(), method)) {
                        result.get(constant).add(method);
                        itMethods.remove();
                        break;
                    }
                }
            }
            constants.clear();
        }

        if (!allMethods.isEmpty()) {
            throw new IllegalArgumentException("could not find a " + MethodOrder.class + " for method " + allMethods);
        }

        for (MethodOrder methodOrder : MethodOrder.values()) {
            // sort methods
            java.util.Collections.sort(result.get(methodOrder));
        }
        return result;
    }

    public static EnumSet<MethodOrder> getMethodOrderScope(EnumSet<MethodOrder> allConstants, int scope) {
        EnumSet<MethodOrder> constants = EnumSet.noneOf(MethodOrder.class);
        for (MethodOrder order : allConstants) {
            if (order.accept(scope)) {
                constants.add(order);
            }
        }
        return constants;
    }

    static class JavaMethodComparator implements Comparator<JavaMethod> {

        @Override
        public int compare(JavaMethod o1, JavaMethod o2) {

            /*int result;
            if ((result = compareStatic(o1, o2)) != 0) {
            return result;
            }
            if ((result = compareConstructor(o1, o2)) != 0) {
            return result;
            }
            if ((result = compareVisibility(o1, o2)) != 0) {
            return result;
            }*/
            return o1.getName().compareTo(o2.getName());
        }

        public int compareStatic(JavaMethod o1, JavaMethod o2) {
            if (Modifier.isStatic(o1.getModifiers()) && !Modifier.isStatic(o2.getModifiers())) {
                return -1;
            }
            if (!Modifier.isStatic(o1.getModifiers()) && Modifier.isStatic(o2.getModifiers())) {
                return 1;
            }
            return 0;
        }

        public int compareConstructor(JavaMethod o1, JavaMethod o2) {
            if (o1.getReturnType() == null && o2.getReturnType() != null) {
                return -1;
            }
            if (o1.getReturnType() != null && o2.getReturnType() == null) {
                return 1;
            }
            return 0;
        }

        public int compareVisibility(JavaMethod o1, JavaMethod o2) {
            if (Modifier.isPublic(o1.getModifiers()) && !Modifier.isPublic(o2.getModifiers())) {
                return -1;
            }
            if (!Modifier.isPublic(o1.getModifiers()) && Modifier.isPublic(o2.getModifiers())) {
                return 1;
            }
            if (Modifier.isProtected(o1.getModifiers()) && !Modifier.isProtected(o2.getModifiers())) {
                return -1;
            }
            if (!Modifier.isProtected(o1.getModifiers()) && Modifier.isProtected(o2.getModifiers())) {
                return 1;
            }
            if (Modifier.isPrivate(o1.getModifiers()) && !Modifier.isPrivate(o2.getModifiers())) {
                return -1;
            }
            if (!Modifier.isPrivate(o1.getModifiers()) && Modifier.isPrivate(o2.getModifiers())) {
                return 1;
            }
            return 0;
        }
    }
}
