package org.nuiton.eugene.plugin;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.nuiton.eugene.ModelFileWriter;
import org.nuiton.eugene.ModelFileWriterConfiguration;
import org.nuiton.eugene.ModelFileWriterUtil;
import org.nuiton.eugene.models.Model;
import org.nuiton.eugene.plugin.writer.BaseModelFileWriter;
import org.nuiton.eugene.plugin.writer.BaseXmiToModelFileWriter;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.*;

/**
 * User: chemit
 * Date: 24 nov. 2009
 * Time: 00:22:37
 *
 * @goal generate-model-files
 * @projectRequired true
 * @requiresDependencyResolution compile
 */
public class GeneratateModelFilesMojo extends AbstractEugeneMojo implements ModelFileWriterConfiguration {

    /**
     * Inputs files to used to generate the required model files.
     * </p>
     * An include has the following pattern :
     * <pre>
     *  writer:
     * </pre>
     * when you want to use a specific writer with his default io values.
     * <p/>
     * Can also write :
     * <pre>
     *  [writer:]directory:includes
     * </pre>
     * where {@code includes} is the pattern to find files from the directory given and must be terminated by the extension
     * of files.
     * <p/>
     * Specifying the {@code writer} can be usefull when you want to use a writer for an unknown extension
     * by any writer.
     * <p/>
     * Example :
     * <pre>
     * &lt;includes&gt;
     *     &lt;include&gt;zargo:&lt;include&gt;
     *     &lt;include&gt;src/main/xmi2:**\/*.zargo&lt;include&gt;
     *     &lt;include&gt;zargo:src/main/xmi:**\/*.zargo2&lt;include&gt;
     * &lt;/includes&gt;
     * </pre>
     *
     * @parameter expression="${eugene.includes}"
     * @required
     * @since 2.0.0
     */
    protected String[] includes;
    /**
     * Where to generate files.
     *
     * @parameter expression="${eugene.outputDirectory}" default-value="target/generated-sources"
     * @required
     * @since 2.0.0
     */
    protected File outputDirectory;
    /**
     * Properties to pass to writer.
     *
     * @parameter
     * @since 2.0.0
     */
    protected Map<String, Object> properties;
    /**
     * Ne génère rien, analyse juste la configuration.
     *
     * @parameter expression="${eugene.dryRun}" default-value="false"
     * @since 2.0.0
     */
    protected boolean dryRun;
    /**
     * Nom du paquetage pour les fichiers générés
     *
     * @parameter expression="${generator.fullPackagePath}" default-value="${project.groupId}.${project.artifactId}"
     * @since 2.0.0
     */
    protected String fullPackagePath;
    /**
     * Nom du paquetage à générer
     *
     * @parameter expression="${generator.extractedPackages}" default-value="${project.groupId}.${project.artifactId}"
     * @since 2.0.0
     */
    protected String extractedPackages;
    /**
     * Nom du resolver a utiliser
     *
     * @parameter expression="${generator.resolver}" default-value="org.nuiton.util.ResourceResolver"
     * @since 2.0.0
     */
    protected String resolver;
    /**
     * An extra directory to be added to the classpath.
     *
     * @parameter expression="${eugene.extraClassPathDirectory}"
     * @since 2.0.0
     */
    protected File extraClassPathDirectory;

    /**
     * All available writers
     *
     * @component role="org.nuiton.eugene.ModelFileWriter"
     */
    protected Map<String, ModelFileWriter> writers;

    /**
     * All available writers for the given {@link #getModelClass()}
     */
    protected Set<ModelFileWriter> availableWriters;

    /**
     * Writers to process
     */
    protected List<ModelFileWriter> plan;

    @Override
    protected boolean init() throws Exception {
        boolean b = super.init();
        if (b) {
            if (includes.length == 0) {
                throw new MojoExecutionException("Must specify something to include using the includes property");
            }

            availableWriters = ModelFileWriterUtil.filterWriterForModelType(writers, getModelClass());

            if (availableWriters.isEmpty()) {
                throw new MojoExecutionException("Could not find any writer in class-path.");
            }

            for (ModelFileWriter writer : availableWriters) {
                if (writer instanceof BaseModelFileWriter) {
                    ((BaseModelFileWriter) writer).setLog(getLog());
                }
            }

            plan = new ArrayList<ModelFileWriter>();

            // first pass to detect top level writers
            for (String include : includes) {

                getLog().info("Register include : " + include);
                ModelFileWriterUtil.registerInclude(this, include, plan);
            }

            if (properties == null) {
                properties = new TreeMap<String, Object>();
            }
        }
        return b;
    }

    @Override
    protected void doAction() throws Exception {
        if (dryRun) {
            getLog().warn("dryRun property is set, no file will be generated.");
        }
        try {
            // transfert to properties

            ClassLoader loader = fixClassLoader();

            properties.put(BaseModelFileWriter.PROP_CLASS_LOADER, loader);
            properties.put(BaseXmiToModelFileWriter.PROP_FULL_PACKAGE_PATH, fullPackagePath);
            //properties.put("extractedPackages", extractedPackages);
            properties.put(BaseXmiToModelFileWriter.PROP_RESOLVER, resolver);

            // launch writers in incoming order of dicovering of them
            for (ModelFileWriter writer : plan) {
                int size = writer.getEntries().size();
                if (size == 1) {
                    getLog().info("Process phase [" + writer.getInputProtocol() + "] for one entry.");
                } else {
                    getLog().info("Process phase [" + writer.getInputProtocol() + "] for " + size + " entries.");
                }
                if (dryRun || isVerbose()) {
                    for (ModelFileWriterUtil.ModelFileWriterEntry entry : writer.getEntries()) {
                        getLog().info(" entry : " + entry.getInputDirectory() + " - " + entry.getIncludePattern());
                    }
                    if (dryRun) {
                        continue;
                    }
                }
                if (getLog().isDebugEnabled()) {
                    getLog().debug("Generating files and copying resources...");
                }

                writer.generate(this);

            }
        } finally {
            // always clear everything to avoid side-effects in goal is invoked more than once
            properties.clear();
            for (ModelFileWriter writer : plan) {
                writer.clear();
            }
            plan.clear();
        }
    }

    @Override
    public File getOutputDirectory() {
        return outputDirectory;
    }

    @Override
    public Map<String, Object> getProperties() {
        return properties;
    }

    @Override
    public Set<ModelFileWriter> getAvailableWriters() {
        return availableWriters;
    }

    @Override
    public Class<? extends Model> getModelClass() {
        return modelClass;
    }

    @Override
    public File getBasedir() {
        return getProject().getBasedir();
    }

    /**
     * Prepare le classLoader a utiliser dans le generateur.
     * <p/>
     * Si un {@link #extraClassPathDirectory} a été renseigné, il est rajouté.
     * <p/>
     * Si des références à des sibling modules, ils seront rajoutés aussi.
     *
     * @return le class loader modifie
     * @throws MojoExecutionException if any pb
     */
    protected ClassLoader fixClassLoader() throws MojoExecutionException {
        Set<String> urlsAsString = new HashSet<String>();
        List<URL> urls = new ArrayList<URL>();
        try {
            ClassLoader loader;
            if (extraClassPathDirectory != null) {
                if (verbose) {
                    getLog().info("Add extra directory in generator's classLoader : " + extraClassPathDirectory);
                }
                addDirectoryToUrlsList(extraClassPathDirectory, urls, urlsAsString);
            }
            if (project.getProjectReferences() != null) {
                // this case is for multi-module when calling from a parent module
                for (Object o : project.getProjectReferences().entrySet()) {
                    Map.Entry<?, ?> entry = (Map.Entry<?, ?>) o;
                    MavenProject relatedProject = (MavenProject) entry.getValue();
                    if (verbose) {
                        getLog().info("Add project reference in generator's classLoader : '" + relatedProject.getArtifact() + "'");
                    }
                    //TODO il faudrait peut-etre aussi ajouter les dependances ?
                    addDirectoryToUrlsList(relatedProject.getArtifact().getFile(), urls, urlsAsString);
                }
            }
            if (!project.getArtifacts().isEmpty()) {
                // this is a special case when artifacts were resolved (for example in site phase)
                if (verbose) {
                    getLog().info("Use resolved artifacts to build class-path");
                }
                for (Object o : project.getArtifacts()) {
                    Artifact a = (Artifact) o;
                    if (!a.getScope().equals("provided")) {
                        addDirectoryToUrlsList(a.getFile(), urls, urlsAsString);
                    }
                }
            }
            // we ask to add the directory in classloader
            loader = getClass().getClassLoader();
            if (getLog().isDebugEnabled()) {
                getLog().info("original classloader " + loader);
            }
            if (loader instanceof URLClassLoader) {
                // on reinjecte les urls de loader de base
                // car sinon on risque de ne pas retrouver les resources...
                for (URL u : ((URLClassLoader) loader).getURLs()) {
                    addUrlToUrlsList(u, urls, urlsAsString);
                    if (getLog().isDebugEnabled()) {
                        getLog().debug("original cp entry: " + u);
                    }
                }
                // et on force l'utilisation du classloader parent
                // s'il existe
                if (loader.getParent() != null) {
                    loader = loader.getParent();
                }
            }
            if (!urls.isEmpty()) {
                loader = new URLClassLoader(urls.toArray(new URL[urls.size()]),
                        loader);
            }
            if (getLog().isDebugEnabled()) {
                for (URL u : urls) {
                    getLog().debug("cp entry: " + u);
                }
            }
            return loader;
        } catch (MalformedURLException e) {
            throw new MojoExecutionException(e.getMessage());
        } finally {
            urls.clear();
            urlsAsString.clear();
        }

    }
}
