/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: MailingManager.java 21 2011-05-09 16:43:58Z sletellier $
 * $HeadURL: http://svn.chorem.org/svn/vradi/tags/vradi-0.6/vradi-services/src/main/java/org/chorem/vradi/services/managers/MailingManager.java $
 * %%
 * Copyright (C) 2009 - 2010 Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package org.chorem.vradi.services.managers;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.tika.exception.TikaException;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.parser.AutoDetectParser;
import org.apache.tika.sax.BodyContentHandler;
import org.chorem.vradi.VradiConstants;
import org.chorem.vradi.VradiServiceConfigurationHelper;
import org.chorem.vradi.beans.SendingHelper;
import org.chorem.vradi.entities.Form;
import org.chorem.vradi.entities.Infogene;
import org.chorem.vradi.entities.Sending;
import org.chorem.vradi.entities.User;
import org.chorem.vradi.services.FileService;
import org.chorem.vradi.services.FileServiceImpl;
import org.chorem.vradi.services.VradiException;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.DateUtil;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.search.Search;
import org.nuiton.wikitty.search.operators.Element;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.mail.Authenticator;
import javax.mail.BodyPart;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.PasswordAuthentication;
import javax.mail.SendFailedException;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.search.FlagTerm;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import static org.chorem.vradi.VradiConstants.SendingStatus;
import static org.nuiton.i18n.I18n._;

/**
 * Mailing manager.
 * <p/>
 * Handle:
 * <ul>
 * <li>mail imap sending/receiving</li>
 * </ul>
 *
 * @author Yolpo, kmorin, chatellier
 */
public class MailingManager {

    private static final Log log = LogFactory.getLog(MailingManager.class);

    protected static final String SSL_FACTORY = "javax.net.ssl.SSLSocketFactory";

    protected ApplicationConfig config;

    protected WikittyProxy wikittyProxy;

    protected FileService fileService;

    protected FormTypeManager formTypeManager;

    public MailingManager(ApplicationConfig config, WikittyProxy wikittyProxy, FormTypeManager formTypeManager) {
        this.config = config;
        this.wikittyProxy = wikittyProxy;
        this.fileService = new FileServiceImpl(config);
        this.formTypeManager = formTypeManager;
    }

    /**
     * Sends an email to the specified recipient with the specified subject
     * and the specified message and the specified forms PDF.
     *
     * @param recipient      the email address of the recipient
     * @param subject        mail subject
     * @param message        mail content
     * @param forms          the list of forms to add to the mail
     * @param receptionProof if true, ask for a reception proof
     * @param attachedFiles  files to attach
     * @return the message id of the email sent
     * @throws VradiException
     */
    public String postMail(String recipient, String subject,
                           String message, List<Form> forms, boolean receptionProof,
                           Collection<File> attachedFiles)
            throws VradiException {

        if (StringUtils.isBlank(recipient)
            || message == null && (forms == null || forms.isEmpty())) {
            if (log.isWarnEnabled()) {
                log.warn(_("Nothing to send : %s", recipient));
            }
            return null;
        }
        String result;

        // TODO EC-20100505 use Session.getDefaultInstance()
        // and common Properties form receiving/sending mails !
        Properties props = new Properties();
        props.put("mail.smtp.host", VradiServiceConfigurationHelper.getSmtpHost(config));
        props.put("mail.smtp.port", VradiServiceConfigurationHelper.getSmtpPort(config));
        //props.put("mail.smtp.auth", "true");
        //props.put("mail.smtp.socketFactory.port", config.getSmtpPort());
        //props.put("mail.smtp.socketFactory.class", SSL_FACTORY);
        //props.put("mail.smtp.socketFactory.fallback", "false");
        //props.put("mail.debug", config.isMailDebug());

        Session session = Session.getInstance(props,
                                              new Authenticator() {
                                                  @Override
                                                  protected PasswordAuthentication getPasswordAuthentication() {
                                                      return new PasswordAuthentication(
                                                              VradiServiceConfigurationHelper.getMailUser(config),
                                                              VradiServiceConfigurationHelper.getMailPassword(config));
                                                  }
                                              });

        // EC-20100427 : attention, ca affiche le contenu des pdf joint
        // c'est absolument illisible
        //session.setDebug(log.isDebugEnabled());

        // create a message
        MimeMessage msg = new MimeMessage(session);

        try {
            // set the from and to address
            InternetAddress addressFrom = new InternetAddress(
                    VradiServiceConfigurationHelper.getMailFrom(config),
                    VradiServiceConfigurationHelper.getMailFromName(config));

            msg.setFrom(addressFrom);

            InternetAddress addressTo = new InternetAddress(recipient);
            msg.addRecipient(Message.RecipientType.TO, addressTo);

            if (receptionProof) {
                // Optional : You can also set your custom headers in the Email if you Want
                msg.addHeader("Return-Receipt-To", VradiServiceConfigurationHelper.getMailFrom(config));
                msg.addHeader("Disposition-Notification-To", VradiServiceConfigurationHelper.getMailFrom(config));
            }

            Multipart multiparts = new MimeMultipart();

            // create the message part
            MimeBodyPart msgBodyPart = new MimeBodyPart();
            msgBodyPart.setText(message);
            multiparts.addBodyPart(msgBodyPart);

            //counter to count the number of attachments
            int count = 0;

            // manage pdf attachments
            if (forms != null) {

                for (Form form : forms) {
                    File formFile = getGeneratedPdfFile(form);

                    if (formFile.exists()) {
                        // Part two is attachment
                        MimeBodyPart msgCurrentPdfPart = new MimeBodyPart();
                        // default is set to application/octet-stream
                        msgCurrentPdfPart.setHeader("Content-Type", "application/pdf");
                        DataSource source = new FileDataSource(formFile);
                        msgCurrentPdfPart.setDataHandler(new DataHandler(source));
                        msgCurrentPdfPart.setFileName(formFile.getName());
                        multiparts.addBodyPart(msgCurrentPdfPart);
                        //add an attachment
                        count++;
                    } else {
                        if (log.isWarnEnabled()) {
                            log.warn("No pdf attachment found for form : " + form.getObjet());
                        }
                    }
                }
            }

            if (count == 0) {
                throw new VradiException(_("vradi.error.sending.noAttachedPDF"));
            }

            // Manage session attached files
            if (attachedFiles != null && !attachedFiles.isEmpty()) {
                for (File file : attachedFiles) {
                    String mineType;
                    try {
                        mineType = getMineType(file);
                    } catch (Exception eee) {
                        log.error("Failled to get mine type of file : " + file.getName(), eee);
                        // default is set to application/octet-stream
                        mineType = "application/octet-stream";
                    }
                    MimeBodyPart msgCurrentPdfPart = new MimeBodyPart();
                    msgCurrentPdfPart.setHeader("Content-Type", mineType);
                    DataSource source = new FileDataSource(file);
                    msgCurrentPdfPart.setDataHandler(new DataHandler(source));
                    msgCurrentPdfPart.setFileName(file.getName());
                    multiparts.addBodyPart(msgCurrentPdfPart);

                    if (log.isInfoEnabled()) {
                        log.info("Sending with attached file : " + file.getName());
                    }
                }
            } else {
                if (log.isInfoEnabled()) {
                    log.info("Sending with no attached file");
                }
            }

            // Put parts in message
            msg.setContent(multiparts);

            // Setting the Subject and Content Type
            msg.setSubject(subject);

            try {
                Transport.send(msg);
            } catch (MessagingException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Cant send mail", eee);
                }

                // match SMTPAddressFailedException && SMTPSendFailedException
                if (eee.getCause() instanceof SendFailedException) {
                    SendFailedException exception = (SendFailedException) eee.getCause();
                    throw new VradiException(_("SMTP fail to send mail :\n(%s)", exception.getLocalizedMessage()), eee);
                }

                throw new VradiException("Can't send message", eee);
            }

            if (log.isDebugEnabled()) {
                log.debug("MessageID: " + msg.getMessageID());
            }
            result = msg.getMessageID();

        } catch (MessagingException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't generate message", eee);
            }
            throw new VradiException(_("vradi.error.sending.send", eee.getMessage()));
        } catch (UnsupportedEncodingException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't generate message", eee);
            }
            throw new VradiException(_("vradi.error.sending.generation", eee.getMessage()));
        }
        return result;
    }

    /**
     * Receives the emails and check if some are error emails or reception proofs
     * and treat them in consequence.
     *
     * @throws VradiException
     */
    public void receiveMails() throws VradiException {
        try {
            // mail from mailer daemon are supposed to be
            // delivery error
            //String mailerDeamonAddress = config.getMailerDaemonEmail();

            // TODO EC-20100505 use Session.getDefaultInstance()
            // and common Properties form receiving/sending mails !
            Properties props = new Properties();
            String imapHost = VradiServiceConfigurationHelper.getImapHost(config);
            if (imapHost == null) {
                log.warn("imap is not configured, abording");
                return;
            }

            final String mailUser = VradiServiceConfigurationHelper.getMailUser(config);
            int imapPort = VradiServiceConfigurationHelper.getImapPort(config);
            props.put("mail.imap.host", imapHost);
            props.put("mail.imap.auth", "true");
            props.put("mail.imap.user", mailUser);
            props.put("mail.imap.port", imapPort);

            //props.put("mail.imap.starttls.enable", "true");
            //props.put("mail.imap.socketFactory.port", config.getImapPort());
            //props.put("mail.imap.socketFactory.class", DummySSLSocketFactory.class.getName());
            //props.put("mail.imap.socketFactory.fallback", "false");

            //props.put("mail.imap.ssl.enable", "true");
            //props.put("mail.imap.ssl.port", config.getImapPort());
            //props.put("mail.imap.ssl.trust", "true");
            //props.put("mail.imap.ssl.socketFactory.class", DummySSLSocketFactory.class.getName());
            //props.put("mail.imap.ssl.socketFactory.fallback", "false");

            props.put("mail.store.protocol", "imap");

            Session session = Session.getInstance(props,
                                                  new Authenticator() {
                                                      @Override
                                                      protected PasswordAuthentication getPasswordAuthentication() {
                                                          return new PasswordAuthentication(
                                                                  mailUser,
                                                                  VradiServiceConfigurationHelper.getMailPassword(config));
                                                      }
                                                  });

            // Récupère la "messagerie" et se connecte
            Store store = session.getStore("imap");
            store.connect(
                    imapHost,
                    mailUser,
                    VradiServiceConfigurationHelper.getMailPassword(config));

            // Récupère le fichier "Boite de réception" et l'ouvre
            Folder folder = store.getFolder("INBOX");
            folder.open(Folder.READ_WRITE);

            FlagTerm flagTerm = new FlagTerm(new Flags(Flags.Flag.SEEN), false);
            Message messages[] = folder.search(flagTerm);

            for (Message message : messages) {
                // pour chaque message, on regarde les messages de type
                // multipart, et plus particulierement leur pieces jointe.
                // different cas apres le parcours de toutes les "part"
                if (message.getContentType().startsWith("multipart/")) {

                    // FIXME EC-20100506 , normalement tous les mails
                    // doivent être marqué comme lu
                    // mais dans le cas ou plusieurs vradi test les mails
                    // qu'il n'ont potentiellement pas envoyé
                    // on ne marque que ceux qu'il ont pu traiter
                    boolean markAsRead = false;

                    if (log.isDebugEnabled()) {
                        log.debug("Testing message " + Arrays.toString(message.getHeader("Message-ID")) + " as multipart");
                    }

                    Multipart multiPart = (Multipart) message.getContent();

                    String errorMessage = null;
                    String errorMessageId = null;
                    String originalMessageID = null;
                    for (int index = 0; index < multiPart.getCount(); ++index) {
                        BodyPart bodyPart = multiPart.getBodyPart(index);
                        if (log.isDebugEnabled()) {
                            log.debug("Part " + index + " : " + bodyPart.getContentType());
                        }

                        // first delivery failure part
                        if ("message/delivery-status".equals(bodyPart.getContentType())) {
                            // Diagnostic-Code is part of the content
                            String content = getStreamContent(bodyPart.getInputStream());
                            errorMessage = getHeaderValueInContent("Diagnostic-Code", content);
                        }

                        // second delivery failure part
                        if ("text/rfc822-headers".equals(bodyPart.getContentType())) {
                            String content = getStreamContent(bodyPart.getInputStream());
                            errorMessageId = getHeaderValueInContent("Message-ID", content);
                        }

                        // response for reception proof
                        if (bodyPart.getContentType().startsWith("message/disposition-notification")) {
                            String content = getStreamContent(bodyPart.getInputStream());
                            originalMessageID = getHeaderValueInContent("Original-Message-ID", content);
                        }
                    }

                    // si on a eu un un message message/delivery-status et un
                    // message/rfc822
                    if (StringUtils.isNotEmpty(errorMessage) && StringUtils.isNotEmpty(errorMessageId)) {
                        // put this message into file
                        if (log.isInfoEnabled()) {
                            log.info("Error message received for message " + errorMessageId);
                            log.info(" error description : " + errorMessage);
                        }

                        Sending sending = getSendingByMessageID(errorMessageId, wikittyProxy);
                        if (sending != null) {
                            sending.setStatus(SendingStatus.ERROR
                                                      .getValue());
                            wikittyProxy.store(sending);

                            // FIXME EC20100506 mark all mails as read
                            markAsRead = true;
                        } else {
                            if (log.isWarnEnabled()) {
                                log.warn("Can't find sending for message id : " + errorMessageId);
                            }
                        }
                    }

                    // cas de l'accusé de reception
                    else if (StringUtils.isNotEmpty(originalMessageID)) {
                        // put this message into file
                        if (log.isInfoEnabled()) {
                            log.info("Proof notification received for message " + originalMessageID);
                        }

                        Sending sending = getSendingByMessageID(originalMessageID, wikittyProxy);
                        if (sending != null) {
                            if (SendingStatus.isWaitingReceptionProof(sending)) {
                                sending.setStatus(SendingStatus.RECEIVED.getValue());
                                sending.setReceptionDate(new Date());
                                wikittyProxy.store(sending);
                                setValidEmailForUser(sending);

                                // FIXME EC20100506 mark all mails as read
                                markAsRead = true;
                            } else {
                                if (log.isWarnEnabled()) {
                                    log.warn("Received a reception proof for a sending with non reception proof waiting status");
                                }
                            }
                        } else {
                            if (log.isWarnEnabled()) {
                                log.warn("Can't find sending for message id : " + originalMessageID);
                            }
                        }
                    }

                    // mark message as READ
                    // FIXME EC20100506 mark all mails as read
                    message.setFlag(Flags.Flag.SEEN, markAsRead);
                } else {
                    if (log.isWarnEnabled()) {
                        log.warn("Mail inbox contains non multipart mails !");
                    }
                }
            }

            folder.close(true);
            store.close();

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't recieve messages", e);
            }
            throw new VradiException("Can't recieve messages", e);
        }

    }

    /**
     * Set {@link User#FQ_FIELD_USER_VALIDEMAIL} attribute for User associated with sending.
     *
     * @param sending
     */
    protected void setValidEmailForUser(Sending sending) {

        String userId = sending.getUser();
        User user = wikittyProxy.restore(User.class, userId);

        // update valid email field only is necessary
        if (user != null && !user.getValidEmail()) {
            if (log.isInfoEnabled()) {
                log.info("User email " + user.getEmail() + " has been validated");
            }
            user.setValidEmail(true);
            wikittyProxy.store(user);
        }
    }

    /**
     * Convert an input stream to String.
     *
     * @param inputStream
     * @return
     * @throws IOException
     */
    protected String getStreamContent(InputStream inputStream) throws IOException {
        StringBuilder sb = new StringBuilder();
        String line;
        try {
            BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, "UTF-8"));
            while ((line = reader.readLine()) != null) {
                sb.append(line).append("\n");
            }
        } finally {
            inputStream.close();
        }
        return sb.toString();
    }

    /**
     * Find a header definition in mail content.
     * <p/>
     * Exemple:
     * <pre>
     * Original-Message-ID: <1833582695.1.1272991628810.JavaMail.chatellier@genux>
     * Diagnostic-Code: smtp; 550 5.1.1 <toto@codelutin.com>: Recipient address
     *      rejected: User unknown in local recipient table
     * Message-ID: <804992706.5.1272991629788.JavaMail.chatellier@genux>
     * </pre>
     *
     * @param headerName
     * @param content
     * @return
     */
    protected String getHeaderValueInContent(String headerName, String content) {
        String result = null;

        String headerText = headerName + ":";
        int firstIndex = content.indexOf(headerText) + headerText.length();
        int lastIndex = content.indexOf("\n", firstIndex);
        if (lastIndex == -1) {
            // case EOF
            lastIndex = content.length();
        }

        result = content.substring(firstIndex, lastIndex);
        result = result.trim();

        // TODO EC-20100505 multilines header

        return result;
    }

    /**
     * Find first sending entity with given messageID.
     *
     * @param messageID message id to find sending
     * @param proxy     wikitty proxy
     * @return {@link Sending} or {@code null} if none found
     */
    protected static Sending getSendingByMessageID(String messageID, WikittyProxy proxy) {

        Sending result = null;

        // find sending with wikitty criteria
        Search search = Search.query();
        search.eq(Element.ELT_EXTENSION, Sending.EXT_SENDING).eq(
                Sending.FQ_FIELD_SENDING_MESSAGEID, messageID);
        Criteria criteria = search.criteria();

        PagedResult<Sending> sendingsResult = proxy.findAllByCriteria(
                Sending.class, criteria);

        List<Sending> sendingList = sendingsResult.getAll();
        if (sendingList.size() == 1) {
            result = sendingList.get(0);
        }

        return result;
    }

    /**
     * Return generated pdf file path containing form wikitty id and form version.
     *
     * @param form form
     * @return generated pdf files
     */
    public File getGeneratedPdfFile(Form form) {
        String filename = form.getWikittyId() + "-" + form.getWikittyVersion() + ".pdf";
        File pdfDir = VradiServiceConfigurationHelper.getPdfDir(config);
        File result = new File(pdfDir, filename);
        return result;
    }

    protected String getMineType(File file) throws IOException, SAXException, TikaException {
        FileInputStream is = new FileInputStream(file);
        ContentHandler contenthandler = new BodyContentHandler();
        Metadata metadata = new Metadata();
        metadata.set(Metadata.RESOURCE_NAME_KEY, file.getName());
        AutoDetectParser parser = new AutoDetectParser();
        parser.parse(is, contenthandler, metadata);
        String result = metadata.get(Metadata.CONTENT_TYPE);
        log.info("Mine type of " + file.getName() + " is : " + result);
        return result;
    }

    /**
     * Generate all the PDFs needed to send a session.
     *
     * @param sessionId the session wikitty id
     */
    public void generatePDFForSession(String sessionId) throws VradiException {
        List<String> formsIdsToGenerate = new ArrayList<String>();

        org.chorem.vradi.entities.Session session = wikittyProxy.restore(org.chorem.vradi.entities.Session.class, sessionId);

        Set<String> sendingIds = session.getSending();

        if (sendingIds == null) {
            return;
        }

        for (String id : sendingIds) {
            Sending sending = wikittyProxy.restore(Sending.class, id);

            if (sending != null) {
                Set<String> formsIds = SendingHelper.extractForms(wikittyProxy, sending.getGroupForms());

                if (formsIds != null) {
                    for (String formId : formsIds) {

                        if (!formsIdsToGenerate.contains(formId)) {
                            formsIdsToGenerate.add(formId);
                        }
                    }
                }
            }
        }

        for (String formId : formsIdsToGenerate) {
            try {
                generatePDF(formId, true);
            } catch (VradiException eee) {
                throw new VradiException("An error occurred while generating PDF file for " +
                                         "form : " + formId, eee);
            }
        }
    }

    public String generatePDF(String formId, boolean force) throws VradiException {

        String uri = null;

        String extensionName = null;
        Form form = wikittyProxy.restore(Form.class, formId);
        for (String ext : form.getExtensionNames()) {
            if (!ext.equals(Infogene.EXT_INFOGENE)
                && !ext.equals(Form.EXT_FORM)) {
                extensionName = ext;
                break;
            }

        }
        WikittyExtension extension = formTypeManager.getFormType(extensionName);
        // VradiHelper.getFormTypeTemplate(extension);
        String template = extension.getTagValue(VradiConstants.FormTypeTemplateEnum.FIELD_TEMPLATE_ATTRIBUTE.getValue());
        File templateFile = formTypeManager.getTemplate(extensionName, template);
        Map<String, String> fieldMap = formTypeManager.getAssociatedFields(extension.getName(), template);

        if (log.isDebugEnabled()) {
            log.debug("Generating PDF file for form " + form.getWikittyId() +
                      " (template = " + template + ")");
        }

        try {
            // but only those associated with current templates !
            List<Form> currentForms = Collections.singletonList(form);
            List<File> filesToAdd = generateFilledDocumentInPDF(
                    templateFile, currentForms, fieldMap, force);

            if (filesToAdd != null && !filesToAdd.isEmpty()) {
                uri = filesToAdd.get(0).getName();
            }
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't generate pdf", ex);
            }
            throw new VradiException("Can't generate pdf", ex);
        }

        return uri;
    }


    /**
     * Generate offer PDF form given forms with specified template.
     *
     * @param template      template to use to generate PDF
     * @param forms         to to generate
     * @param fieldBindings map between template field name and form fields names
     * @param replace       replace already existing generated forms
     * @return generated pdf files
     * @throws VradiException for various possible errors
     */
    protected List<File> generateFilledDocumentInPDF(File template, List<Form> forms,
                                                     Map<String, String> fieldBindings,
                                                     boolean replace)
            throws VradiException {

        List<File> generatedPDFList = null;

        if (template == null) {
            if (log.isWarnEnabled()) {
                log.warn("Try to generate form pdf without template");
            }
        } else if (fieldBindings == null) {
            if (log.isWarnEnabled()) {
                log.warn("No association field found, abording");
            }
        } else {
            TemplateManager templateManager = new TemplateManager(config, template);

            // split fqn form field name in an array
            // composed of extension name, and extension field
            // TODO EC20100510 set in commun with same function in
            // {@link generateFilledDocumentInPDF(TemplateManager, Form form, Map<String, String[]>, boolean)
            Map<String, String[]> map = new HashMap<String, String[]>();
            for (Map.Entry<String, String> binding : fieldBindings.entrySet()) {
                String fieldFQName = binding.getValue();
                if (fieldFQName != null && fieldFQName.indexOf(".") > 0) {
                    int dot = fieldFQName.indexOf(".");
                    String[] fqField = new String[2];
                    fqField[0] = fieldFQName.substring(0, dot);
                    fqField[1] = fieldFQName.substring(dot + 1);
                    map.put(binding.getKey(), fqField);
                }
            }

            generatedPDFList = new ArrayList<File>();
            for (Form form : forms) {
                File f = generateFilledDocumentInPDF(templateManager, form, map, replace);
                generatedPDFList.add(f);
            }
        }

        return generatedPDFList;
    }

    /**
     * Generate a single pdf file for specified form.
     *
     * @param templateManager pdf template manager
     * @param form            form to get pdf
     * @param fieldBindings   association between field name and form fields (couple extension name, extension field)
     * @param replace         replace already existing pdf
     * @return the pdf file associated with pdf
     * @throws VradiException for various possible errors
     */
    protected File generateFilledDocumentInPDF(TemplateManager templateManager,
                                               Form form, Map<String, String[]> fieldBindings,
                                               boolean replace) throws VradiException {

        // generated pdf are cached
        File result = getGeneratedPdfFile(form);
        if (!replace && result.exists()) {
            return result;
        }

        Map<String, Object> fieldValues = new HashMap<String, Object>();
        for (Map.Entry<String, String[]> binding : fieldBindings.entrySet()) {
            String[] fqField = binding.getValue();
            Object value = form.getField(fqField[0], fqField[1]);
            fieldValues.put(binding.getKey(), value);
        }

        // Extract files
        List<String> formUrls = new ArrayList<String>();
        if (form.getFiles() != null) {
            for (String file : form.getFiles()) {
                String formUrl = fileService.getFormEmbeddedUrl(file, form.getWikittyId());
                formUrls.add(formUrl);
            }
        }

        templateManager.generateDoc(result.getPath(), fieldValues,
                                    formUrls.toArray(new String[formUrls.size()]));

        return result;
    }

    public String sendMessages(String sessionId) throws VradiException {

        org.chorem.vradi.entities.Session session = wikittyProxy.restore(org.chorem.vradi.entities.Session.class, sessionId);

        if (VradiConstants.SessionStatus.isSent(session)) {
            return StringUtils.EMPTY;
        }

        // Pass to active for session in error or stopped
        if (VradiConstants.SessionStatus.isToRestart(session)) {

            log.info("Session was stopped or in error, resume sending");

            session.setStatus(VradiConstants.SessionStatus.ACTIVE.getValue());
            session.setSessionLogs(StringUtils.EMPTY);
            session = wikittyProxy.store(session);
        }

        // Get all files to attach
        Set<String> filesNames = session.getFiles();
        List<File> files = new ArrayList<File>();
        if (filesNames != null) {
            for (String fileName : filesNames) {
                File file = fileService.downloadSessionAttachment(fileName, sessionId);
                files.add(file);
                log.info("Files founds for session : " + session.getSessionDate() + " : " + (file == null ? "not found" : file.getName()));
            }
        }

        // Init logger to keep exception
        List<String> sessionLogger = new ArrayList<String>();

        Set<String> sendingIds = session.getSending();
        if (sendingIds != null) {

            for (String sendingId : sendingIds) {

                // Flag to detect if it's canceled by user
                session = wikittyProxy.restore(org.chorem.vradi.entities.Session.class, session.getWikittyId());
                if (VradiConstants.SessionStatus.isStopped(session)) {
                    log.warn("Sending stopped by user");
                    sessionLogger.add(_("vradi.error.session.stopped", session.getNum(),
                                        DateUtil.formatDate(session.getSessionDate(), DateUtil.DEFAULT_PATTERN)));
                }

                Sending sending = wikittyProxy.restore(Sending.class, sendingId);
                if (sending != null) {
                    if (SendingStatus.isSent(sending)) {
                        if (log.isDebugEnabled()) {
                            log.debug("Sending " + sendingId + " already sent, skip");
                        }
                    } else if (SendingStatus.isToSend(sending)) {

                        // Send
                        String log = sendMessage(sendingId, session.getParagraph(), null, false, files);
                        if (log != null) {
                            sessionLogger.add(log);
                        }
                    }
                }
            }
        }

        // Get logs
        String formatedLogs = formatSessionLog(sessionLogger);

        // Set session to sent or error
        if (sessionLogger.isEmpty()) {
            session.setStatus(VradiConstants.SessionStatus.SENT.getValue());
            session.setSessionLogs(StringUtils.EMPTY);
        } else {
            session.setStatus(VradiConstants.SessionStatus.ERROR.getValue());
            session.setSessionLogs(formatedLogs);
        }
        session.setSessionDate(new Date());
        wikittyProxy.store(session);

        return formatedLogs;
    }

    protected String formatSessionLog(List<String> logs) {
        return StringUtils.join(logs, ",\n");
    }

    public String sendMessage(String sendingId, String sessionParagraph, String email, boolean reSend, Collection<File> filesToAttach) throws VradiException {

        Sending sending = wikittyProxy.restore(Sending.class, sendingId);
        User user = wikittyProxy.restore(User.class, sending.getUser());

        // Will resend sending already sent
        if (!reSend) {
            // Do nothing
            if (!SendingStatus.isToSend(sending)) {
                return null;
            }
        }

        // If its client or group sending : do nothing
        // Never append
//        if (sending.getUser() == null) {
//            sending.setStatus(VradiConstants.SendingStatus.SENT.getValue());
//            return wikittyProxy.store(sending);
//        }

        Set<String> formsIds = SendingHelper.extractForms(wikittyProxy, sending.getGroupForms());

        // If no forms founds
        if (formsIds.isEmpty()) {

            // Set session in error
            sending.setStatus(SendingStatus.DELETED.getValue());
            wikittyProxy.store(sending);

            // Do nothing
            return null;
        }

        // Get email address
        // si aucune n'est specifiée, on utilise celle du client
        if (StringUtils.isBlank(email)) {
            email = user.getEmail();

            // il peut arriver qu'un user n'est pas d'email et que l'admin
            // souhaite quand meme faire l'envoie d'une session.
            // dans ce cas on ne traite pas l'envoi de mail courant.
            if (StringUtils.isBlank(email)) {

                if (log.isWarnEnabled()) {
                    log.warn("User " + user.getName() + " has not valid email (skip current mail sending)");
                }
                // Set session in error
                sending.setStatus(SendingStatus.ERROR.getValue());
                wikittyProxy.store(sending);

                return _("vradi.error.sending.emailIsBlank", user.getName());
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Sending message to " + email);
        }

        String subject = VradiServiceConfigurationHelper.getOfferMailSubject(config);

        // Set message
        String message = sessionParagraph + "\n\n" + sending.getParagraph();

        List<Form> forms = wikittyProxy.restore(Form.class, new ArrayList<String>(formsIds));

        boolean receptionProof = sending.getReceptionProof();

        // Post mail and save msg id
        String messageId;
        try {
            messageId = postMail(
                    email, subject, message, forms, receptionProof, filesToAttach);

        } catch (VradiException eee) {

            log.error("Failed to send sending for email : " + email, eee);

            // Set sending in error
            sending.setStatus(SendingStatus.ERROR.getValue());
            wikittyProxy.store(sending);

            return _("vradi.error.sending.failedToSend", user.getName(), eee.getMessage());
        }
        sending.setMessageId(messageId);

        // Set sending status
        sending.setStatus(receptionProof ?
                          SendingStatus.WAITING_RECEPTION_PROOF.getValue() :
                          SendingStatus.SENT.getValue());

        // Update sending
        wikittyProxy.store(sending);

        return null;
    }

}
