package org.chorem.pollen.ui.actions.user;
/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id: CreateFavoriteList.java 3595 2012-08-12 11:45:39Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.1/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/user/CreateFavoriteList.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.chorem.pollen.business.persistence.PersonList;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.services.exceptions.FavoriteListAlreadyExistException;
import org.chorem.pollen.services.exceptions.FavoriteListImportException;
import org.chorem.pollen.services.exceptions.FavoriteListNotFoundException;
import org.chorem.pollen.services.exceptions.ParticipantAlreadyFoundInListException;
import org.chorem.pollen.services.impl.FavoriteService;
import org.chorem.pollen.ui.actions.FileUploadAware;
import org.chorem.pollen.ui.actions.PollenActionSupportForEdition;

import java.io.File;
import java.util.List;

/**
 * To create a new favorite list.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.5
 */
public class CreateFavoriteList extends PollenActionSupportForEdition implements FileUploadAware {

    private static final long serialVersionUID = 1L;

    protected PersonList createFavoriteList;

    protected File csvImport;

    protected String csvImportContentType;

    protected String csvImportFileName;

    protected String ldapImport;

    @Override
    public void addFile(int index, File file) {
        csvImport = file;
    }

    @Override
    public void addFileContentType(int index, String contentType) {
        csvImportContentType = contentType;
    }

    @Override
    public void addFileName(int index, String fileName) {
        csvImportFileName = fileName;
    }

    public void setLdapImport(String ldapImport) {
        this.ldapImport = ldapImport;
    }

    public PersonList getCreateFavoriteList() {
        if (createFavoriteList == null) {
            createFavoriteList = getFavoriteService().newFavoriteList();
        }
        return createFavoriteList;
    }

    public String getLdapImport() {
        return ldapImport;
    }

    public String getAction() {
        return "create";
    }

    @Override
    public void validate() {

        if (StringUtils.isBlank(getCreateFavoriteList().getName())) {
            addFieldError("createFavoriteList.name",
                          _("pollen.error.favoriteListName.required"));
        }

    }

    @Override
    public String execute() throws Exception {

        Preconditions.checkNotNull(createFavoriteList);
        Preconditions.checkNotNull(createFavoriteList.getName());

        int nbImports = 0;
        try {
            PersonList personList =
                    getFavoriteService().createFavoriteList(getPollenUserAccount(),
                                                            createFavoriteList.getName());

            if (csvImportFileName != null) {

                nbImports = addImportFromCsv(personList);
            }

            if (StringUtils.isNotBlank(ldapImport)) {

                nbImports = addImportFromLDAP(personList);
            }

        } catch (FavoriteListAlreadyExistException ex) {
            addFieldError("createFavoriteList.name",
                          _("pollen.error.favoriteList.already.used"));
        }

        String result;
        if (hasAnyErrors()) {
            result = INPUT;

        } else {
            getTransaction().commitTransaction();

            if (nbImports > 0) {
                addFlashMessage(_("pollen.information.favoriteList.imported",
                                  createFavoriteList.getName(), nbImports));
            } else {

                addFlashMessage(_("pollen.information.favoriteList.created",
                                  createFavoriteList.getName()));
            }
            createFavoriteList = null;
            result = SUCCESS;
        }
        return result;
    }

    protected int addImportFromCsv(PersonList personList) throws FavoriteListNotFoundException {

        int nbImports = 0;

        try {
            List<PollAccount> importedAccounts = getFavoriteService().importFromCsvfile(
                    csvImportFileName, csvImport);

            nbImports = addImport(importedAccounts, personList);

        } catch (FavoriteListImportException ex) {
            String message = ex.getLocalizedMessage(getLocale());
            addFlashError(message);
        }
        return nbImports;
    }

    protected int addImportFromLDAP(PersonList personList) throws FavoriteListNotFoundException {

        int nbImports = 0;

        try {
            List<PollAccount> importedAccounts = getFavoriteService().importFromLDAP(
                    ldapImport);

            nbImports = addImport(importedAccounts, personList);

        } catch (FavoriteListImportException ex) {
            String message = ex.getLocalizedMessage(getLocale());
            addFlashError(message);
        }
        return nbImports;
    }

    protected int addImport(List<PollAccount> importedAccounts,
                            PersonList list)
            throws FavoriteListNotFoundException {

        FavoriteService favoriteService = getFavoriteService();

        for (PollAccount importedAccount : importedAccounts) {
            try {
                favoriteService.addPollAccountToFavoriteList(list, importedAccount);

            } catch (ParticipantAlreadyFoundInListException ex) {
                // WARNING ?
                addFlashError(
                        _("pollen.error.favoriteList.import.participantExists"
                                , importedAccount.getEmail())
                );
            }
        }

        return importedAccounts.size();
    }
}