/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id: AbstractVoteAction.java 3528 2012-06-19 13:22:18Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/AbstractVoteAction.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.interceptor.ParameterAware;
import org.apache.struts2.interceptor.ServletRequestAware;
import org.chorem.pollen.bean.PollResult;
import org.chorem.pollen.bean.PollResultList;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.Comment;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteToChoice;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.services.exceptions.PollAccountNotFound;
import org.chorem.pollen.services.exceptions.PollNotFoundException;
import org.chorem.pollen.services.impl.SecurityService;
import org.chorem.pollen.ui.PollenUIUtils;
import org.chorem.pollen.ui.actions.PageSkin;

import javax.servlet.http.HttpServletRequest;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Abstract action for actions on the vote poll page.
 *
 * @author tchemit <chemit@codelutin.com>
 * @author fdesbois <fdesbois@codelutin.com>
 * @since 1.3
 */
public abstract class AbstractVoteAction extends AbstractPollUriIdAction implements ParameterAware, ServletRequestAware {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(AbstractVoteAction.class);

    public static final String PREPARE_VOTE_PAGE = "prepareVotePage";

    /**
     * Loaded poll.
     *
     * @since 1.3
     */
    private Poll poll;

    /**
     * Is feed exists for this poll ?
     *
     * @since 1.3
     */
    private boolean feedFileExisting;

    /**
     * Is user the poll's creator ?
     *
     * @since 1.3
     */
    private boolean creatorUser;

    /**
     * Loaded poll account (from the pollUri).
     *
     * @since 1.3
     */
    private PollAccount pollAccount;

    /**
     * Is vote allowed for current user?
     *
     * @since 1.3
     */
    private boolean voteAllowed;

    /**
     * Is current user can go to results.
     *
     * @since 1.4
     */
    private boolean resultAllowed;

    /**
     * All votes for the poll.
     *
     * @since 1.3
     */
    private List<Vote> votes;

    /**
     * Current vote to treat (for create/updatevote action).
     *
     * @since 1.3
     */
    private Vote vote;

    /**
     * Comment author to display in UI.
     *
     * @since 1.3
     */
    private String commentAuthor;

    /**
     * Results of the poll.
     *
     * @since 1.3
     */
    private List<PollResult> results;

    /**
     * List of comments to display.
     *
     * @since 1.3
     */
    private List<Comment> comments;

    /**
     * Injected parameters from request.
     *
     * @since 1.3
     */
    private Map<String, String[]> parameters;

    /**
     * The accountId role on this page.
     *
     * @since 1.4
     */
    private SecurityService.AccountIdRole accountIdRole;

    /**
     * The incoming request (some stuff are store in it from security filters).
     *
     * @since 1.4
     */
    private transient HttpServletRequest request;

    /**
     * @return {@code true} if moderation is possible, {@code false} otherwise
     * @since 1.4
     */
    public abstract boolean isModerate();

    @Override
    public PageSkin getSkin() {
        return PageSkin.VOTE;
    }

    @Override
    public void setParameters(Map<String, String[]> parameters) {
        this.parameters = parameters;
    }

    @Override
    public void setServletRequest(HttpServletRequest request) {
        this.request = request;
    }

    public Poll getPoll() {
        return poll;
    }

    public PollAccount getPollAccount() {
        return pollAccount;
    }

    public List<Vote> getVotes() {
        return votes;
    }

    public Vote getVote() {
        return vote;
    }

    public List<PollResult> getResults() {
        return results;
    }

    public List<Comment> getComments() {
        return comments;
    }

    public String getCreatorName() {
        return poll.getCreator().getVotingId();
    }

    public String getVoteSizeMessage() {
        return _("pollen.common.voteNbVotes", getPoll().sizeVote());
    }

    public String getCommentAuthor() {
        if (commentAuthor == null) {

            UserAccount user = getPollenUserAccount();
            if (user == null) {

                // Use current pollAccount name for comment
                commentAuthor = getPollAccount().getVotingId();

            } else {

                // Connecter user name
                commentAuthor = user.getDisplayName();
            }
        }
        return commentAuthor;
    }

    public String getSummaryUrl() {
        PollUrl url = getPollUrlService().getPollSummaryUrl(poll);
        url.getPollUri().setAccountId(getAccountId());
        return url.getUrl();
    }

    public String getResultUrl() {
        PollUrl url = getPollUrlService().getPollResultUrl(poll);
        url.getPollUri().setAccountId(getAccountId());
        getSecurityService().removeAccountIdWhenConnected(url, getPollenUserAccount());
        return url.getUrl();
    }

    public String getVoteMessages() {
        return _("pollen.common.voteNbVotes", poll.sizeVote());
    }

    public boolean isFeedFileExisting() {
        return feedFileExisting;
    }

    public boolean isCreatorOrAdmin() {
        return creatorUser || isUserAdmin();
    }

    public boolean isAccountFieldDisplayed() {
        return !poll.isAnonymous() || isRestrictedPoll() || isGroupPoll();
    }

    public boolean isPollChoiceOrVoteStarted() {
        Date now = serviceContext.getCurrentTime();
        return poll.isAddChoiceStarted(now) || poll.isStarted(now);
    }

    public boolean isAnonymousVote() {
        return poll.isAnonymous();
    }

    public boolean isPollChoiceRunning() {
        Date now = serviceContext.getCurrentTime();
        return poll.isAddChoiceRunning(now);
    }

    public boolean isPollStarted() {
        Date now = serviceContext.getCurrentTime();
        return poll.getBeginDate() == null || poll.getBeginDate().before(now);
    }

    public boolean isPollFinished() {
        Date now = serviceContext.getCurrentTime();
        return poll.getEndDate() != null && poll.getEndDate().before(now);
    }

    public boolean isDescNull(Choice choice) {
        return StringUtils.isEmpty(choice.getDescription());
    }

    public boolean isFreePoll() {
        return poll.getPollType() == PollType.FREE;
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextType() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateType() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImageType() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    public boolean isNormalVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.NORMAL;
    }

    public boolean isPercentageVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.PERCENTAGE;
    }

    public boolean isCondorcetVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.CONDORCET;
    }

    public boolean isNumberVoteCounting() {
        return poll.getVoteCountingType() == VoteCountingType.NUMBER;
    }

    public boolean isVoteAllowed() {
        return voteAllowed;
    }

    public boolean isResultAllowed() {
        return resultAllowed;
    }

    public boolean isModifyVoteAllowed(Vote vote) {
        return getSecurityService().isCanModifyVote(
                getPoll(),
                vote.getTopiaId(),
                getAccountId(),
                getPollenUserAccount());
    }

    public boolean isDeleteCommentAllowed(Comment comment) {
        return getSecurityService().isCanDeleteComment(
                comment,
                getUriId().getAccountId(),
                accountIdRole,
                getPollenUserAccount());
    }

    public boolean isDeleteVoteAllowed(Vote vote) {
        return getSecurityService().isCanDeleteVote(
                getPoll(),
                vote.getTopiaId(),
                getUriId().getAccountId(),
                accountIdRole,
                getPollenUserAccount());
    }

    public String getResultValue(Choice choice) {

        String val = getPollResultsService().getResultValue(choice, results);
        return val;
    }

    /**
     * Retourne si le choix fait partie du vote (s'il a été renseigné par le
     * votant).
     *
     * @param choice le choix concerné
     * @return true si le choix est dans le vote
     */
    public boolean isChoiceInVote(VoteToChoice choice) {
        boolean result = false;
        if (choice != null) {
            switch (poll.getVoteCountingType()) {
                case NORMAL:
                    result = choice.getVoteValue() > 0;
                    break;
                case PERCENTAGE:
                    result = true;
                    break;
                case CONDORCET:
                    result = choice.getVoteValue() < 100;
                    break;
                case NUMBER:
                    result = choice.getVoteValue() >= 0;
            }
        }
        return result;
    }

    public Date getChoiceAsDate(Choice choice) {
        return new Date(Long.valueOf(choice.getName()));
    }

    public void setCommentAuthor(String commentAuthor) {
        this.commentAuthor = commentAuthor;
    }

    public String prepareVotePage() throws Exception {

        boolean moderate = isModerate();

        loadPoll();

        // Current poll account
        loadPollAccount();

        // All votes
        // TODO no pagination for the moment, need to retrieve the correct page depends on current pollAccount
        votes = getVoteService().getAllVotes(poll);

        accountIdRole = PollenUIUtils.getAccountIdRole(request);

        // is vote allowed ?
        if (moderate) {
            voteAllowed = false;
        } else {
            String accountId = getAccountId();
            if (accountIdRole == SecurityService.AccountIdRole.CREATOR) {

                // remove accountId (can vote even if creator ?)
                accountId = null;
            }
            voteAllowed = getSecurityService().isCanVote(poll,
                                                         accountId,
                                                         accountIdRole);
        }

        // is can display result link ?
        resultAllowed =
                getSecurityService().isCanAccessResult(poll,
                                                       getAccountId(),
                                                       accountIdRole,
                                                       getPollenUserAccount());

        if (voteAllowed) {

            // load modifiable vote
            vote = getVoteService().getVoteEditable(poll, pollAccount);
        }

        // load poll results
        PollResultList pollResultList =
                getPollResultsService().getResults(poll);

        results = pollResultList.getPollResults();

        if (log.isDebugEnabled()) {
            for (PollResult res : results) {
                log.debug(res.getName() + ": " + res.getValue()
                          + ", (voteCounting=" + res.getVoteCountingType()
                          + ", byGroup=" + res.isByGroup() + ")");
            }
        }

        // load comments
        comments = getPollCommentService().getAllComments(poll.getPollId());

        feedFileExisting = getPollFeedService().isFeedExists(poll);

        creatorUser = getSecurityService().isPollCreator(poll,
                                                         getAccountId(),
                                                         getPollenUserAccount());

        if (log.isInfoEnabled()) {
            Date now = serviceContext.getCurrentTime();
            log.info("pollChoiceOrVoteStarted = " + isPollChoiceOrVoteStarted());
            log.info("pollChoiceRunning       = " + isPollChoiceRunning());
            log.info("pollRunning             = " + poll.isRunning(now));
            log.info("accountFieldDisplayed   = " + isAccountFieldDisplayed());
            log.info("creatorUser             = " + creatorUser);
        }
        return INPUT;
    }

    public String escapeLineBreak(String text) {
        return text;
    }

    protected void loadPollAccount() throws PollAccountNotFound {

        // Current poll account
        pollAccount = getPollService().getPollAccountEditable(
                getAccountId(), getPollenUserAccount(), poll);
    }

    protected void loadPoll() throws PollNotFoundException {

        // Ensure uri for poll and pollAccount loading
        preparePollUri(parameters);

        String pollId = getPollId();
        if (StringUtils.isNotEmpty(pollId)) {
            poll = getPollService().getExistingPollByPollId(pollId);
        }
        Preconditions.checkNotNull(poll,
                                   "Can't load poll with id = [" + pollId + "]");

        if (log.isDebugEnabled()) {
            log.debug("Poll TopiaId: " + poll.getTopiaId());
        }
    }

}
