/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.converters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.VoteDTO;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.ChoiceDAO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteDAO;
import org.chorem.pollen.business.persistence.VoteToChoice;
import org.chorem.pollen.business.persistence.VoteToChoiceDAO;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.business.persistence.VotingListDAO;
import org.nuiton.topia.TopiaException;

/**
 * Gestion de la conversion des dtos Vote en entité et vice-versa.
 *
 * @version $Id: DataVoteConverter.java 2615 2009-07-01 13:47:45Z nrannou $
 */
public class DataVoteConverter extends DataConverter {

    /**
     * Création d'une entité vote à partir d'un dto vote.
     *
     * @param voteDTO Le dto vote.
     * @param eVote L'entité vote.
     */
    public void populateVoteEntity(VoteDTO voteDTO, Vote eVote)
            throws TopiaException {
        eVote.setWeight(voteDTO.getWeight());
        eVote.setComment(voteDTO.getComment());

        if (voteDTO.getPollId() != null) {
            PollDAO pollDAO = PollenModelDAOHelper.getPollDAO(transaction);
            Poll pollEntity = pollDAO.findByTopiaId(voteDTO.getPollId());
            pollEntity.addVote(eVote); // Rajout du vote au sondage
            eVote.setPoll(pollEntity);
        }
        if (voteDTO.getPollAccountId() != null) {
            PollAccountDAO pollAccountDAO = PollenModelDAOHelper
                    .getPollAccountDAO(super.transaction);
            PollAccount pollAccountEntity = pollAccountDAO
                    .findByTopiaId(voteDTO.getPollAccountId());
            eVote.setPollAccount(pollAccountEntity);
        }
        if (voteDTO.getVotingListId() != null) {
            VotingListDAO votingListDAO = PollenModelDAOHelper
                    .getVotingListDAO(super.transaction);
            VotingList votingListEntity = votingListDAO.findByTopiaId(voteDTO
                    .getVotingListId());
            eVote.setVotingList(votingListEntity);
        }
    }

    /**
     * Création d'un dto vote à partir d'une entité vote.
     *
     * @param eVote L'entité vote
     * @return Le dto vote
     */
    public VoteDTO createVoteDTO(Vote eVote) {
        VoteDTO voteDTO = new VoteDTO();
        voteDTO.setId(eVote.getTopiaId());
        voteDTO.setWeight(eVote.getWeight());
        voteDTO.setComment(eVote.getComment());

        if (eVote.getPoll() != null) {
            voteDTO.setPollId(eVote.getPoll().getTopiaId());
        }
        if (eVote.getPollAccount() != null) {
            voteDTO.setPollAccountId(eVote.getPollAccount().getTopiaId());
        }
        if (eVote.getVotingList() != null) {
            voteDTO.setVotingListId(eVote.getVotingList().getTopiaId());
        }

        // Conversion des VoteToChoice
        if (eVote.getChoiceVoteToChoice() != null) {
            DataChoiceConverter choiceConverter = new DataChoiceConverter();
            for (VoteToChoice voteToChoice : eVote.getChoiceVoteToChoice()) {
                ChoiceDTO choiceDTO = choiceConverter
                        .createChoiceDTO(voteToChoice.getChoice());
                choiceDTO.setValue(voteToChoice.getVoteValue());
                voteDTO.getChoiceDTOs().add(choiceDTO);
            }
        }

        return voteDTO;
    }

    /**
     * Retourne une liste de dtos vote à partir d'une collection d'entités.
     *
     * @param cVotes collection des entités vote
     * @return la liste des dtos vote
     */
    public List<VoteDTO> createVoteDTOs(Collection<Vote> cVotes) {
        List<VoteDTO> results = new ArrayList<VoteDTO>();
        VoteDTO dto;
        for (Vote v : cVotes) {
            dto = this.createVoteDTO(v);
            results.add(dto);
        }
        return results;
    }

    /**
     * Mise à jour des VoteToChoice d'une entité vote à partir d'un dto vote.
     *
     * @param voteDTO Le dto vote.
     * @param eVote L'entité vote.
     */
    public void populateChoiceVote(VoteDTO voteDTO, Vote eVote)
            throws TopiaException {

        for (ChoiceDTO choiceDTO : voteDTO.getChoiceDTOs()) {
            choiceDTO.setVoteId(eVote.getTopiaId());
        }

        eVote.setChoiceVoteToChoice(this.getVoteToChoices(voteDTO
                .getChoiceDTOs()));
    }

    /**
     * Retourne la liste des VoteToChoice à partir d'une liste de dtos choix.
     *
     * @param choiceDTOs La liste des dtos choix
     * @return la liste d'entités VoteToChoice
     */
    private List<VoteToChoice> getVoteToChoices(List<ChoiceDTO> choiceDTOs)
            throws TopiaException {
        List<VoteToChoice> liste = new ArrayList<VoteToChoice>();

        for (ChoiceDTO choice : choiceDTOs) {
            VoteToChoiceDAO voteToChoiceDAO = PollenModelDAOHelper
                    .getVoteToChoiceDAO(super.transaction);
            VoteToChoice voteToChoice = voteToChoiceDAO.create();
            voteToChoice.setVoteValue(choice.getValue());

            // Ajout du vote
            VoteDAO voteDAO = PollenModelDAOHelper
                    .getVoteDAO(super.transaction);
            Vote voteEntity = voteDAO.findByTopiaId(choice.getVoteId());
            voteToChoice.setVote(voteEntity);

            // Ajout choix
            ChoiceDAO choiceDAO = PollenModelDAOHelper
                    .getChoiceDAO(super.transaction);
            Choice choiceEntity = choiceDAO.findByTopiaId(choice.getId());
            voteToChoice.setChoice(choiceEntity);

            liste.add(voteToChoice);
        }

        return liste;
    }
}
