package fr.ifremer.tutti.ui.swing.content.operation;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditFishingOperationUIHandler.java 1097 2013-06-11 13:21:03Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.3.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/EditFishingOperationUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.referential.Gear;
import fr.ifremer.tutti.persistence.entities.referential.Person;
import fr.ifremer.tutti.persistence.entities.referential.TuttiLocation;
import fr.ifremer.tutti.persistence.entities.referential.Vessel;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.AbstractCaracteristicTabUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.GearUseFeatureTabUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.fishing.VesselUseFeatureTabUIModel;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiTabContainerUIHandler;
import fr.ifremer.tutti.ui.swing.util.TabHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionHelper;
import fr.ifremer.tutti.ui.swing.util.editor.CoordinateEditorType;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXDatePicker;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * Handler for UI {@link EditFishingOperationUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EditFishingOperationUIHandler extends AbstractTuttiTabContainerUIHandler<EditFishingOperationUIModel, EditFishingOperationUI>
        implements TabHandler {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EditFishingOperationUIHandler.class);

    private final FishingOperationsUI parentUi;

    private final PersistenceService persistenceService;

    /**
     * listener to set the same end date as the start date
     * when the user changes it only (remove this listener before selecting the
     * fishing operation)
     */
    private final PropertyChangeListener startDateListener = new PropertyChangeListener() {

        public void propertyChange(PropertyChangeEvent evt) {
            //check if the user changed the date and not only the time)
            Date oldDate = (Date) evt.getOldValue();
            Date newDate = (Date) evt.getNewValue();
            if (newDate != null && getModel().getGearShootingEndDate() == null &&
                (oldDate == null || !DateUtils.isSameDay(oldDate, newDate))) {
                getModel().setGearShootingEndDate(newDate);
            }
        }
    };

    private final PropertyChangeListener coordinatePropertiesListener = new PropertyChangeListener() {

        private List<String> properties = Lists.newArrayList(
                EditFishingOperationUIModel.PROPERTY_FISHING_OPERATION_RECTILIGNE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE_DECIMAL_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE_DEGREE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE_SECOND,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE_DECIMAL_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE_DEGREE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE_SECOND,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE_DECIMAL_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE_DEGREE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE_SECOND,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE_DECIMAL_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE_DEGREE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE_MINUTE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE_SECOND
        );

        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            if (properties.contains(evt.getPropertyName())) {
                EditFishingOperationUIModel source = (EditFishingOperationUIModel) evt.getSource();
                if (source.isFishingOperationRectiligne()) {
                    source.computeDistance();
                }
            }
        }
    };

    /**
     * To monitor changes on the incoming fishing operation.
     *
     * @since 0.3
     */
    private final TuttiBeanMonitor<EditFishingOperationUIModel> fishingOperationMonitor;

    public EditFishingOperationUIHandler(FishingOperationsUI parentUi,
                                         EditFishingOperationUI ui) {
        super(parentUi.getHandler().getContext(), ui);
        this.parentUi = parentUi;
        this.persistenceService = context.getPersistenceService();
        this.fishingOperationMonitor = new TuttiBeanMonitor<EditFishingOperationUIModel>(
                EditFishingOperationUIModel.PROPERTY_STATION_NUMBER,
                EditFishingOperationUIModel.PROPERTY_FISHING_OPERATION_NUMBER,
                EditFishingOperationUIModel.PROPERTY_STRATA,
                EditFishingOperationUIModel.PROPERTY_SUB_STRATA,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LATITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_LONGITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_DATE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LATITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_LONGITUDE,
                EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_END_DATE,
                EditFishingOperationUIModel.PROPERTY_FISHING_OPERATION_RECTILIGNE,
                EditFishingOperationUIModel.PROPERTY_TRAWL_DISTANCE,
                EditFishingOperationUIModel.PROPERTY_FISHING_OPERATION_VALID,
                EditFishingOperationUIModel.PROPERTY_LOCATION,
                EditFishingOperationUIModel.PROPERTY_RECORDER_PERSON,
                EditFishingOperationUIModel.PROPERTY_VESSEL,
                EditFishingOperationUIModel.PROPERTY_GEAR,
                EditFishingOperationUIModel.PROPERTY_MULTIRIG_AGGREGATION,
                EditFishingOperationUIModel.PROPERTY_COMMENT);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInitUI() {

        EditFishingOperationUIModel model = new EditFishingOperationUIModel();
        model.setCoordinateEditorType(getConfig().getCoordinateEditorType());
        model.setValidationContext(getContext().getValidationContext());

        model.addPropertyChangeListener(EditFishingOperationUIModel.PROPERTY_STRATA, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                EditFishingOperationUIModel source = (EditFishingOperationUIModel) evt.getSource();
                TuttiLocation newStrata = (TuttiLocation) evt.getNewValue();
                onSelectedStrata(source, newStrata);
            }
        });

        model.addPropertyChangeListener(EditFishingOperationUIModel.PROPERTY_SUB_STRATA, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                EditFishingOperationUIModel source = (EditFishingOperationUIModel) evt.getSource();
                TuttiLocation newStrata = (TuttiLocation) evt.getNewValue();
                onSelectedSubStrata(source, newStrata);
            }
        });

        getContext().addPropertyChangeListener(TuttiUIContext.PROPERTY_VALIDATION_CONTEXT, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                getModel().setValidationContext((String) evt.getNewValue());
            }
        });

        model.addPropertyChangeListener(EditFishingOperationUIModel.PROPERTY_FISHING_OPERATION_VALID, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Boolean valid = (Boolean) evt.getNewValue();
                Color color = null;
                Color fontColor = Color.BLACK;
                if (valid == Boolean.TRUE) {
                    color = Color.GREEN;

                } else if (valid == Boolean.FALSE) {
                    color = Color.RED;
                    fontColor = Color.WHITE;
                }
                Component tab = getTabPanel().getTabComponentAt(0);
                tab.setForeground(fontColor);
                tab.setBackground(color);
            }
        });

        ui.setContextValue(model);

        fishingOperationMonitor.setBean(model);
    }

    @Override
    public void afterInitUI() {

        ui.getFishingOperationValidPanel().remove(
                ui.getFishingOperationResetRadio());

        final EditFishingOperationUIModel model = getModel();

        initUI(ui);

        List<Person> people = getDataContext().getPersons();
        initBeanList(ui.getRecorderPersonList(),
                     people,
                     model.getRecorderPerson());

        initBeanFilterableComboBox(ui.getGearComboBox(),
                                   Lists.<Gear>newArrayList(),
                                   null);

        Cruise cruise = getDataContext().getCruise();
        Preconditions.checkNotNull(cruise,
                                   "Could not find cruise in ui context");
        ui.getMultirigAggregationField().setEnabled(cruise.getMultirigNumber() != 1);

        Program program = getDataContext().getProgram();
        Preconditions.checkNotNull(program,
                                   "Could not find program in ui context");

        TuttiLocation zone = program.getZone();

        TuttiLocation strata = model.getStrata();

        String strataId = strata == null ? null : strata.getId();

        TuttiLocation subStrata = model.getSubStrata();
        String substrataId = subStrata == null ? null : subStrata.getId();

        TuttiLocation location = model.getLocation();

        String zoneId = zone.getId();
        List<TuttiLocation> stratas =
                Lists.newArrayList(persistenceService.getAllFishingOperationStrata(zoneId));

        List<TuttiLocation> subStratas =
                Lists.newArrayList(persistenceService.getAllFishingOperationSubStrata(zoneId, strataId));

        List<TuttiLocation> locations =
                Lists.newArrayList(persistenceService.getAllFishingOperationLocation(zoneId, strataId, substrataId));

        initBeanFilterableComboBox(ui.getStrataComboBox(), stratas, strata);

        initBeanFilterableComboBox(ui.getSubStrataComboBox(), subStratas, subStrata);

        initBeanFilterableComboBox(ui.getLocationComboBox(), locations, location);

        changeValidatorContext(model.getValidationContext(), ui.getValidator());
        listenValidationTableHasNoFatalError(ui.getValidator(), model);

        setCustomTab(0, model);
        listModelIsModify(model);

        //init gear use features
        GearUseFeatureTabUIModel gearUseFeatureModel =
                ui.getGearUseFeatureTabContent().getModel();
        gearUseFeatureModel.setAvailableCaracteristics(getDataContext().getCaracteristics());
        setCustomTab(1, gearUseFeatureModel);

        //init vessel use features

        VesselUseFeatureTabUIModel vesselUseFeatureModel =
                ui.getVesselUseFeatureTabContent().getModel();
        vesselUseFeatureModel.setAvailableCaracteristics(getDataContext().getCaracteristics());
        setCustomTab(2, vesselUseFeatureModel);

        // listen when id becones empty or not toupdate cancel action and button
        getModel().addPropertyChangeListener(
                EditFishingOperationUIModel.PROPERTY_ID,
                new PropertyChangeListener() {

                    public void propertyChange(PropertyChangeEvent evt) {
                        JButton button = ui.getCancelButton();
                        CancelEditFishingOperationAction action =
                                TuttiActionHelper.getLogicAction(button);
                        if (evt.getNewValue() == null) {
                            button.setText(_("tutti.editFishingOperation.action.cancelEditFishingOperation"));
                            button.setToolTipText(_("tutti.editFishingOperation.action.cancelEditFishingOperation.tip"));
                            button.setMnemonic(_("tutti.editFishingOperation.action.cancelEditFishingOperation.mnemonic").charAt(0));
                            action.setActionDescription(_("tutti.editFishingOperation.action.cancelEditFishingOperation.tip"));

                        } else {
                            button.setText(_("tutti.editFishingOperation.action.resetEditFishingOperation"));
                            button.setToolTipText(_("tutti.editFishingOperation.action.resetEditFishingOperation.tip"));
                            button.setMnemonic(_("tutti.editFishingOperation.action.resetEditFishingOperation.mnemonic").charAt(0));
                            action.setActionDescription(_("tutti.editFishingOperation.action.resetEditFishingOperation.tip"));
                        }
                    }
                });
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getStationNumberField();
    }

    @Override
    protected Set<String> getPropertiesToIgnore() {
        Set<String> result = super.getPropertiesToIgnore();
        result.add(EditFishingOperationUIModel.PROPERTY_ATTACHMENT);
        return result;
    }

    @Override
    public void onCloseUI() {

        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        clearValidators();

        ui.getFishingOperationAttachmentsButton().onCloseUI();
        closeUI(ui.getGearUseFeatureTabContent());
        closeUI(ui.getVesselUseFeatureTabContent());
    }

    @Override
    public SwingValidator<EditFishingOperationUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void setDate(ActionEvent event, String property) {
        JXDatePicker field = (JXDatePicker) event.getSource();
        Date value = field.getDate();
        Date date = (Date) TuttiUIUtil.getProperty(getModel(), property);
        if (value != null && date != null) {
            Calendar cal = DateUtils.toCalendar(date);
            value = DateUtils.setHours(value, cal.get(Calendar.HOUR_OF_DAY));
            value = DateUtils.setMinutes(value, cal.get(Calendar.MINUTE));
        }
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTabContainerUIHandler methods                         --//
    //------------------------------------------------------------------------//

    @Override
    protected boolean onTabChanged(int currentIndex, int newIndex) {
        ui.getFishingOperationAttachmentsButton().onCloseUI();
        return super.onTabChanged(currentIndex, newIndex);
    }

    @Override
    protected JTabbedPane getTabPanel() {
        return ui.getFishingOperationTabPane();
    }

    //------------------------------------------------------------------------//
    //-- TabHandler methods                                                 --//
    //------------------------------------------------------------------------//

    @Override
    public boolean onHideTab(int currentIndex, int newIndex) {
        boolean result;
        ui.getFishingOperationAttachmentsButton().onCloseUI();
        if (isAModelModified()) {

            // something was modified

            EditFishingOperationUIModel model = getModel();

            if (model.isValid()) {

                // ask if user want to save, do not save or cancel action

                String message;

                if (TuttiEntities.isNew(model.getFishingOperation())) {
                    message = _("tutti.editFishingOperation.askSaveBeforeLeaving.createFishingOperation");
                } else {
                    message = _("tutti.editFishingOperation.askSaveBeforeLeaving.saveFishingOperation");
                }

                int answer = askSaveBeforeLeaving(message);
//                ActionEvent event = new ActionEvent(this, newIndex, null);
                switch (answer) {
                    case JOptionPane.OK_OPTION:
                        TuttiActionHelper.runAction(ui.getSaveButton());
                        result = true;
                        break;

                    case JOptionPane.NO_OPTION:
                        TuttiActionHelper.runAction(ui.getCancelButton());
                        result = true;
                        break;
                    default:

                        // other case, use cancel action
                        result = false;
                }
            } else {

                // model is not valid, ask user to continue or not

                result = askCancelEditBeforeLeaving(
                        _("tutti.editFishingOperation.askCancelEditBeforeLeaving.cancelEditFishingOperation"));

                if (result) {
                    // ok will revert any modification
                    TuttiActionHelper.runAction(ui.getCancelButton());
                }
            }

        } else {

            // model not modify, can change tab
            result = true;
        }
        return result;
    }

    @Override
    public void onShowTab(int currentIndex, int newIndex) {
        registerValidators(ui.getValidator());
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void registerValidator() {
        registerValidators(ui.getValidator());
    }

    public AbstractCaracteristicTabUIModel[] getSubModels() {
        return new AbstractCaracteristicTabUIModel[]{
                ui.getGearUseFeatureTabContent().getModel(),
                ui.getVesselUseFeatureTabContent().getModel()
        };
    }

    public void uninstallStartDateListener() {
        getModel().removePropertyChangeListener(EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_DATE, startDateListener);
    }

    public void installStartDateListener() {
        getModel().addPropertyChangeListener(EditFishingOperationUIModel.PROPERTY_GEAR_SHOOTING_START_DATE, startDateListener);
    }

    public void uninstallCoordinatesListener() {
        getModel().removePropertyChangeListener(coordinatePropertiesListener);
    }

    public void installCoordinatesListener() {
        getModel().addPropertyChangeListener(coordinatePropertiesListener);
    }

    public FishingOperationsUI getParentUi() {
        return parentUi;
    }

    public TuttiBeanMonitor<EditFishingOperationUIModel> getFishingOperationMonitor() {
        return fishingOperationMonitor;
    }

    public String getGearLatitudeLabelText(CoordinateEditorType coordinateEditorType) {
        String result = _("tutti.editFishingOperation.field.gearLatitude", coordinateEditorType.toString());
        return result;
    }

    public String getGearLongitudeLabelText(CoordinateEditorType coordinateEditorType) {
        String result = _("tutti.editFishingOperation.field.gearLongitude", coordinateEditorType.toString());
        return result;
    }

    public String getTrawlDistanceTooltipText(Float trawlDistance) {
        String result = trawlDistance == null ?
                        null :
                        _("tutti.editFishingOperation.field.trawlDistance.inMilles.tip", TuttiUIUtil.getDistanceInMilles(trawlDistance));
        return result;
    }

    public String getDuration(Date startDate, Date endDate) {
        String duration = TuttiEntities.getDuration(
                startDate,
                endDate,
                _("tutti.editFishingOperation.duration.format")
        );
        return duration;
    }

    public String decorateVessel(Vessel vessel) {
        return decorate(vessel);
    }

    //------------------------------------------------------------------------//
    //-- Protected methods                                                  --//
    //------------------------------------------------------------------------//

    protected void onSelectedStrata(EditFishingOperationUIModel model,
                                    TuttiLocation newStrata) {

        // reset substrata value
        model.setSubStrata(null);

        // reset sub strata combo
        ui.getSubStrataComboBox().setData(null);

        TuttiLocation zone = getDataContext().getProgram().getZone();

        String zoneId = zone.getId();
        String strataId = newStrata == null ? null : newStrata.getId();

        List<TuttiLocation> subStrata =
                persistenceService.getAllFishingOperationSubStrata(zoneId, strataId);

        List<TuttiLocation> location =
                persistenceService.getAllFishingOperationLocation(zoneId, strataId, null);

        ui.getSubStrataComboBox().setData(subStrata);

        ui.getLocationComboBox().setData(location);

        if (newStrata == null) {

            // reset strata, keep focus on it
            ui.getStrataComboBox().grabFocus();

        } else {

            // try to load substrata


            if (CollectionUtils.isEmpty(subStrata)) {

                // try to load localite
                ui.getLocationComboBox().grabFocus();
            }
        }
    }

    protected void onSelectedSubStrata(EditFishingOperationUIModel model,
                                       TuttiLocation newSubStrata) {
        // reset localite value
        model.setLocation(null);

        // reset localite combo
        ui.getLocationComboBox().setData(null);

        TuttiLocation zone = getDataContext().getProgram().getZone();
        String zoneId = zone.getId();

        TuttiLocation strata = model.getStrata();
        String subStrataId = newSubStrata == null ? null : newSubStrata.getId();

        String strataId = strata == null ? null : strata.getId();

        List<TuttiLocation> location =
                persistenceService.getAllFishingOperationLocation(zoneId, strataId, subStrataId);

        ui.getLocationComboBox().setData(location);

        if (newSubStrata == null) {

            // reset substrata, keep focus on it
            ui.getSubStrataComboBox().grabFocus();
        }
    }

    protected boolean areAllModelsValid() {
        boolean result = getModel().isValid();
        AbstractCaracteristicTabUIModel[] subModels = getSubModels();
        for (AbstractCaracteristicTabUIModel subModel : subModels) {
            result &= subModel.isValid();
        }
        return result;
    }

    protected boolean isAModelModified() {
        boolean result = getModel().isModify();
        AbstractCaracteristicTabUIModel[] subModels = getSubModels();
        for (AbstractCaracteristicTabUIModel subModel : subModels) {
            result |= subModel.isModify();
        }
        return result;
    }

    protected void resetAllModels() {

        for (AbstractCaracteristicTabUIModel subModel : getSubModels()) {
            subModel.setModify(false);
        }
        getModel().setModify(false);
    }

    public void reset(FishingOperation bean) {

        List<Attachment> attachments =
                getContext().getPersistenceService().getAllAttachments(
                        getModel().getObjectType(),
                        bean.getIdAsInt());

        getModel().removeAllAttachment(getModel().getAttachment());
        getModel().addAllAttachment(attachments);
    }
}
