/*
  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

  Licensed under the Apache License, Version 2.0 (the "License").
  You may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/
package software.amazon.msk.auth.iam.internals;

import com.amazonaws.util.ClassLoaderHelper;
import com.amazonaws.util.VersionInfoUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.util.Properties;
import java.util.StringJoiner;

import static com.amazonaws.util.IOUtils.closeQuietly;

/**
 * This class is used to generate the user agent for the authentication request.
 */
final class UserAgentUtils {
    private static final Logger log = LoggerFactory.getLogger(UserAgentUtils.class);

    private static final String USER_AGENT_SEP = "/";
    private static final String USER_AGENT_NAME = "aws-msk-iam-auth";
    private static final String VERSION_INFO_FILE = "version.properties";
    //TODO: UPDATE WHEN NEW VERSIONS ARE PUBLISHED.
    //TODO: Find a way to read the library version from a resource generated by the build system.

    private static final String[] AGENT_COMPONENTS = new String[] {
            USER_AGENT_NAME,
            getLibraryVersion(),
            VersionInfoUtils.getUserAgent()
    };

    private static final String USER_AGENT_STRING = generateUserAgentString(AGENT_COMPONENTS);

    private static final String generateUserAgentString(String[] components) {
        StringJoiner joiner = new StringJoiner(USER_AGENT_SEP);
        for (String component : components) {
            joiner.add(component);
        }
        return joiner.toString();
    }

    private static String getLibraryVersion() {
        String version = "unknown-version";

        InputStream inputStream = ClassLoaderHelper.getResourceAsStream(
                VERSION_INFO_FILE, true, UserAgentUtils.class);
        Properties versionProperties = new Properties();
        try {
            if (inputStream == null) {
                log.info("Unable to load version information for msk iam auth plugin");
            } else {
                versionProperties.load(inputStream);
                version = versionProperties.getProperty("version");
            }
        } catch (Exception e) {
            log.info("Unable to load version information for the running SDK: " + e.getMessage());
        } finally {
            closeQuietly(inputStream, null);
        }
        return version;
    }

    public static String getUserAgentValue() {
        return USER_AGENT_STRING;
    }
}
