/*
 * Copyright 2012-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.boot.actuate.autoconfigure.metrics.export.signalfx;

import java.time.Duration;

import org.springframework.boot.actuate.autoconfigure.metrics.export.properties.StepRegistryProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;

/**
 * {@link ConfigurationProperties @ConfigurationProperties} for configuring metrics export
 * to SignalFX.
 *
 * @author Jon Schneider
 * @author Andy Wilkinson
 * @author Stephane Nicoll
 * @since 2.0.0
 */
@ConfigurationProperties(prefix = "management.signalfx.metrics.export")
public class SignalFxProperties extends StepRegistryProperties {

	/**
	 * Step size (i.e. reporting frequency) to use.
	 */
	private Duration step = Duration.ofSeconds(10);

	/**
	 * SignalFX access token.
	 */
	private String accessToken;

	/**
	 * URI to ship metrics to.
	 */
	private String uri = "https://ingest.signalfx.com";

	/**
	 * Uniquely identifies the app instance that is publishing metrics to SignalFx.
	 * Defaults to the local host name.
	 */
	private String source;

	/**
	 * Type of histogram to publish.
	 */
	private HistogramType publishedHistogramType = HistogramType.DEFAULT;

	@Override
	public Duration getStep() {
		return this.step;
	}

	@Override
	public void setStep(Duration step) {
		this.step = step;
	}

	public String getAccessToken() {
		return this.accessToken;
	}

	public void setAccessToken(String accessToken) {
		this.accessToken = accessToken;
	}

	public String getUri() {
		return this.uri;
	}

	public void setUri(String uri) {
		this.uri = uri;
	}

	public String getSource() {
		return this.source;
	}

	public void setSource(String source) {
		this.source = source;
	}

	public HistogramType getPublishedHistogramType() {
		return this.publishedHistogramType;
	}

	public void setPublishedHistogramType(HistogramType publishedHistogramType) {
		this.publishedHistogramType = publishedHistogramType;
	}

	public enum HistogramType {

		/**
		 * Default, time-based histogram.
		 */
		DEFAULT,

		/**
		 * Cumulative histogram.
		 */
		CUMULATIVE,

		/**
		 * Delta histogram.
		 */
		DELTA

	}

}
