/*
 * Decompiled with CFR 0.152.
 */
package org.owasp.dependencycheck.utils;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileLock;
import java.security.SecureRandom;
import java.sql.Timestamp;
import java.util.Date;
import javax.annotation.concurrent.NotThreadSafe;
import org.owasp.dependencycheck.exception.H2DBLockException;
import org.owasp.dependencycheck.utils.Checksum;
import org.owasp.dependencycheck.utils.Settings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class H2DBLock {
    private static final Logger LOGGER = LoggerFactory.getLogger(H2DBLock.class);
    public static final int SLEEP_DURATION = 10000;
    public static final int MAX_SLEEP_COUNT = 120;
    private FileLock lock = null;
    private RandomAccessFile file = null;
    private File lockFile = null;
    private final Settings settings;
    private final String magic;

    public H2DBLock(Settings settings) {
        this.settings = settings;
        byte[] random = new byte[16];
        SecureRandom gen = new SecureRandom();
        gen.nextBytes(random);
        this.magic = Checksum.getHex((byte[])random);
    }

    public boolean isLocked() {
        return this.lock != null && this.lock.isValid();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void lock() throws H2DBLockException {
        try {
            File dir = this.settings.getDataDirectory();
            this.lockFile = new File(dir, "dc.update.lock");
            if (!this.lockFile.getParentFile().isDirectory() && !this.lockFile.mkdir()) {
                throw new H2DBLockException("Unable to create path to data directory.");
            }
            if (this.lockFile.isFile() && this.getFileAge(this.lockFile) > 30.0) {
                LOGGER.debug("An old db update lock file was found: {}", (Object)this.lockFile.getAbsolutePath());
                if (!this.lockFile.delete()) {
                    LOGGER.warn("An old db update lock file was found but the system was unable to delete the file. Consider manually deleting {}", (Object)this.lockFile.getAbsolutePath());
                }
            }
            int ctr = 0;
            do {
                try {
                    if (!this.lockFile.exists() && this.lockFile.createNewFile()) {
                        this.file = new RandomAccessFile(this.lockFile, "rw");
                        this.lock = this.file.getChannel().lock();
                        this.file.writeBytes(this.magic);
                        this.file.getChannel().force(true);
                        Thread.sleep(20L);
                        this.file.seek(0L);
                        String current = this.file.readLine();
                        if (current != null && !current.equals(this.magic)) {
                            this.lock.close();
                            this.lock = null;
                            LOGGER.debug("Another process obtained a lock first ({})", (Object)Thread.currentThread().getName());
                        } else {
                            Timestamp timestamp = new Timestamp(System.currentTimeMillis());
                            LOGGER.debug("Lock file created ({}) {} @ {}", new Object[]{Thread.currentThread().getName(), this.magic, timestamp.toString()});
                        }
                    }
                }
                catch (IOException | InterruptedException ex) {
                    LOGGER.trace("Expected error as another thread has likely locked the file", (Throwable)ex);
                }
                finally {
                    if (this.lock == null && this.file != null) {
                        try {
                            this.file.close();
                            this.file = null;
                        }
                        catch (IOException ex) {
                            LOGGER.trace("Unable to close the lock file", (Throwable)ex);
                        }
                    }
                }
                if (this.lock != null && this.lock.isValid()) continue;
                try {
                    Timestamp timestamp = new Timestamp(System.currentTimeMillis());
                    LOGGER.debug("Sleeping thread {} ({}) for 10 seconds because an exclusive lock on the database could not be obtained ({})", new Object[]{Thread.currentThread().getName(), this.magic, timestamp.toString()});
                    Thread.sleep(10000L);
                }
                catch (InterruptedException ex) {
                    LOGGER.debug("sleep was interrupted.", (Throwable)ex);
                    Thread.currentThread().interrupt();
                }
            } while (++ctr < 120 && (this.lock == null || !this.lock.isValid()));
            if (this.lock == null || !this.lock.isValid()) {
                throw new H2DBLockException("Unable to obtain the update lock, skipping the database update. Skippinig the database update.");
            }
        }
        catch (IOException ex) {
            throw new H2DBLockException(ex.getMessage(), ex);
        }
    }

    public void release() {
        if (this.lock != null) {
            try {
                this.lock.release();
                this.lock = null;
            }
            catch (IOException ex) {
                LOGGER.debug("Failed to release lock", (Throwable)ex);
            }
        }
        if (this.file != null) {
            try {
                this.file.close();
                this.file = null;
            }
            catch (IOException ex) {
                LOGGER.debug("Unable to delete lock file", (Throwable)ex);
            }
        }
        if (this.lockFile != null && this.lockFile.isFile()) {
            try (RandomAccessFile f = new RandomAccessFile(this.lockFile, "rw");){
                String m = f.readLine();
                f.close();
                if (m != null && m.equals(this.magic) && !this.lockFile.delete()) {
                    LOGGER.error("Lock file '{}' was unable to be deleted. Please manually delete this file.", (Object)this.lockFile.toString());
                    this.lockFile.deleteOnExit();
                }
            }
            catch (IOException ex) {
                LOGGER.debug("Error deleting lock file", (Throwable)ex);
            }
        }
        this.lockFile = null;
        Timestamp timestamp = new Timestamp(System.currentTimeMillis());
        LOGGER.debug("Lock released ({}) {} @ {}", new Object[]{Thread.currentThread().getName(), this.magic, timestamp.toString()});
    }

    private double getFileAge(File file) {
        Date d = new Date();
        long modified = file.lastModified();
        double time = (double)(d.getTime() - modified) / 1000.0 / 60.0;
        LOGGER.debug("Lock file age is {} minutes", (Object)time);
        return time;
    }
}

