/*
 * #%L
 * ToPIA :: Service Replication
 * 
 * $Id: ReplicationEngine.java 1894 2010-04-15 15:44:51Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.3.3/topia-service-replication/src/main/java/org/nuiton/topia/replication/ReplicationEngine.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.topia.replication;

import org.nuiton.topia.replication.model.ReplicationOperationDef;
import org.nuiton.topia.replication.model.ReplicationModel;
import org.nuiton.topia.replication.model.ReplicationNode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityEnum;
import org.nuiton.topia.persistence.util.TopiaEntityHelper;
import org.nuiton.topia.replication.model.ReplicationOperationPhase;
import static org.nuiton.i18n.I18n._;

/**
 * Implantation du service de replication.
 * 
 * @author chemit
 */
public class ReplicationEngine implements TopiaReplicationImplementor {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(ReplicationEngine.class);
    /**
     * le contexte sur la base source de la replication
     */
    protected TopiaContextImplementor context;
    /**
     * la liste des operations disponibles (chargee automatiquement via un 
     * ServiceLoader sur le contract {@link TopiaReplicationOperation})
     *
     * @see #preInit(TopiaContextImplementor) 
     */
    protected static TopiaReplicationOperation[] operations;

    //--------------------------------------------------------------------------
    //-- TopiaService implementation -------------------------------------------
    //--------------------------------------------------------------------------
    @Override
    public String getServiceName() {
        return SERVICE_NAME;
    }

    @Override
    public Class<?>[] getPersistenceClasses() {
        // pas de classes persistentes pour ce service
        return null;
    }

    @Override
    public boolean preInit(TopiaContextImplementor context) {
        if (operations == null) {

            // on effectue cette rechercher une seule fois par jvm
            synchronized (this) {
                // chargement des operations disponibles une seule fois

                ServiceLoader<TopiaReplicationOperation> loader = ServiceLoader.load(TopiaReplicationOperation.class);

                List<TopiaReplicationOperation> discoveredOperations = new ArrayList<TopiaReplicationOperation>();

                for (TopiaReplicationOperation op : loader) {
                    if (log.isDebugEnabled()) {
                        log.debug("detected operation " + op);
                    }
                    discoveredOperations.add(op);

                }
                operations = discoveredOperations.toArray(new TopiaReplicationOperation[discoveredOperations.size()]);
            }
        }
        return true;
    }

    @Override
    public boolean postInit(TopiaContextImplementor context) {
        this.context = context;
        //TODO avoir un objet pour lire les contrainte de resolution de cycle
        //TODO sur les dependances (par exemple, une dependance marquee comme
        // non nulle : ne peut jamais etre dettache, alors que dans le cas
        // contraire on peut dettacher la dependance
        // Cela permet de traiter plus de cas...

        //Properties prop = context.getConfig();
        return true;
    }

    //--------------------------------------------------------------------------
    //-- TopiaReplicationService implementation --------------------------------
    //--------------------------------------------------------------------------
    @Override
    public ReplicationModel prepare(TopiaEntityEnum[] contracts, String... entities) throws TopiaException {
        ReplicationModel model = createModel(contracts, entities);

        initModel(model, true);

        return model;
    }

    @Override
    public ReplicationModel prepareForAll(TopiaEntityEnum[] contracts) throws TopiaException {
        ReplicationModel model = createModelForAll(contracts);

        initModel(model, true);

        return model;
    }

    @Override
    public ReplicationModel prepareWithComputedOrder(TopiaEntityEnum[] contracts, String... topiaIds) throws TopiaException {
        ReplicationModel model = createModelWithComputedOrder(contracts, topiaIds);

        initModel(model, false);

        return model;
    }

    @Override
    public void addBeforeOperation(ReplicationModel model, TopiaEntityEnum type, Class<? extends TopiaReplicationOperation> operationClass, Object... parameters) {
        createOperation(model, type, ReplicationOperationPhase.before, operationClass, parameters);
    }

    @Override
    public void addAfterOperation(ReplicationModel model, TopiaEntityEnum type, Class<? extends TopiaReplicationOperation> operationClass, Object... parameters) {
        createOperation(model, type, ReplicationOperationPhase.after, operationClass, parameters);
    }

    @Override
    public void doReplicate(ReplicationModel model, TopiaContext dstCtxt) throws Exception {

        checkNotNull("doReplicate", "model", model);
        checkNotNull("doReplicate", "dstCtxt", dstCtxt);

        TopiaContextImplementor srcCtxt = null;

        Map<Class<? extends TopiaEntity>, List<String>> data = null;
        Set<ReplicationNode> treated = new HashSet<ReplicationNode>();
        try {

            srcCtxt = (TopiaContextImplementor) context.beginTransaction();

            data = getIds(model, srcCtxt);

            srcCtxt.closeContext();

            model.adjustOperations(data);

            for (ReplicationNode node : model.getOrder()) {
                srcCtxt = (TopiaContextImplementor) context.beginTransaction();
                TopiaContext tx = dstCtxt.beginTransaction();
                try {
                    doReplicateNode(node, srcCtxt, tx, data, treated);
                } catch (Exception e) {
                    tx.rollbackTransaction();
                    throw e;
                } finally {
                    srcCtxt.closeContext();
                    tx.closeContext();
                }
            }
//        } catch (Exception e) {
//            dstCtxt.rollbackTransaction();
//            throw e;
        } finally {
            if (data != null) {
                data.clear();
            }
            if (treated != null) {
                treated.clear();
            }
            // on ne doit jamais commiter sur la base source
            if (srcCtxt != null && !srcCtxt.isClosed()) {
                srcCtxt.rollbackTransaction();
                srcCtxt.closeContext();
            }
//            dstCtxt.closeContext();
        }
    }

    //--------------------------------------------------------------------------
    //-- TopiaReplicationImplementor implementation ----------------------------
    //--------------------------------------------------------------------------
    @Override
    public ReplicationModel createModel(TopiaEntityEnum[] contracts, String... topiaIds) throws TopiaException {
        Set<Class<? extends TopiaEntity>> detectTypes = detectTypes(contracts, topiaIds);
        ReplicationModel model = new ReplicationModel(contracts, detectTypes, topiaIds);
        return model;
    }

    @Override
    public ReplicationModel createModelWithComputedOrder(TopiaEntityEnum[] contracts, String... topiaIds) throws TopiaException {
        ReplicationModel model = new ReplicationModel(contracts, false, topiaIds);
        return model;
    }

    @Override
    public ReplicationModel createModelForAll(TopiaEntityEnum[] contracts) throws TopiaException {
        ReplicationModel model = new ReplicationModel(contracts, true);
        return model;
    }

    @Override
    public ReplicationModel initModel(ReplicationModel model, boolean computeOrder) throws TopiaException {
        checkNotNull("initModel", "model", model);

        model.detectAssociations();
        model.detectDirectDependencies();
        if (computeOrder) {
            model.detectShell();
            model.detectDependencies();
        }
        model.detectObjectsToDettach();
        model.detectOperations();
        return model;
    }

    @Override
    public TopiaReplicationOperation getOperation(Class<? extends TopiaReplicationOperation> operationClass) {
        checkNotNull("getOperation", "operationClass", operationClass);

        if (operations == null) {
            throw new IllegalStateException("service was not init!");
        }
        TopiaReplicationOperation result = null;

        for (TopiaReplicationOperation op : operations) {
            if (operationClass.isAssignableFrom(op.getClass())) {
                result = op;
                break;
            }
        }
        return result;
    }

    @Override
    public void createOperation(ReplicationModel model, TopiaEntityEnum type, ReplicationOperationPhase phase, Class<? extends TopiaReplicationOperation> operationClass, Object... parameters) {

        checkNotNull("createOperation", "model", model);
        checkNotNull("createOperation", "type", type);
        checkNotNull("createOperation", "phase", phase);
        checkNotNull("createOperation", "operationClass", operationClass);

        TopiaReplicationOperation operation = getOperation(operationClass);

        if (operation == null) {
            throw new IllegalArgumentException(_("topia.replication.engine.error.unkown.operation", operationClass.getSimpleName(), Arrays.toString(operations)));
        }

        ReplicationNode node = model.getNode(type);
        if (node == null) {
            throw new IllegalArgumentException(_("topia.replication.engine.error.unkown.owner.node", type, operationClass.getSimpleName(), model.getNodes()));
        }
        operation.register(model, node, phase, parameters);
    }

    @Override
    public Set<Class<? extends TopiaEntity>> detectTypes(TopiaEntityEnum[] contracts, String... ids) throws TopiaException {
        TopiaContext ctxt = context.beginTransaction();
        try {
            TopiaEntity[] entities = getEntities(ctxt, ids);

            // on detecte tous les types connus pour les entites données
            Set<Class<? extends TopiaEntity>> types = TopiaEntityHelper.detectTypes(contracts, entities);

            if (log.isDebugEnabled()) {
                log.debug("for type : " + entities.getClass());
                for (Class<? extends TopiaEntity> k : types) {
                    log.debug(k);
                }
            }
            return types;
        } finally {
            ctxt.closeContext();
        }
    }

    @Override
    public Map<Class<? extends TopiaEntity>, List<String>> getIds(ReplicationModel model, TopiaContextImplementor srcCtxt) throws TopiaException {
        Map<Class<? extends TopiaEntity>, List<String>> data;
        // on recupere les objets a repliquer par type
        if (model.isReplicateAll()) {
            // on recupere pour chaque type tous les ids des entites a repliquer
            data = new HashMap<Class<? extends TopiaEntity>, List<String>>();
            for (TopiaEntityEnum e : model.getContracts()) {
                List<String> ids = srcCtxt.getDAO(e.getContract()).findAllIds();
                data.put(e.getContract(), ids);
            }
        } else {
            // on recupere les entites specifies a repliquer
            TopiaEntity[] entities = getEntities(srcCtxt, model.getTopiaIds());
            // on calcule toutes les ids des entites a repliquer
            data = TopiaEntityHelper.detectEntityIds(model.getContracts(), model.getTypes(), entities);
        }
        return data;
    }

    @Override
    public <E extends TopiaEntity> void doReplicateNode(
            ReplicationNode node,
            TopiaContext srcCtxt,
            TopiaContext dstCtxt,
            Map<Class<? extends TopiaEntity>, List<String>> data,
            Set<ReplicationNode> treated) throws Exception {

        // on trie toujours les operations a realiser selon leur phase (avant ou apres la duplication)
        node.sortOperations();

        List<ReplicationOperationDef> operationDefs = node.getOperations();

        if (operationDefs.isEmpty()) {
            log.info("skip node " + node + " - no operation detected.");
        } else {

            log.info("start for " + node + " : " + operationDefs.size() + " operation(s)");

            List<String> nodeEntityIds = data.get(node.getContract().getContract());
            if (log.isInfoEnabled()) {
                log.info("will replicate on " + nodeEntityIds.size() + " entity(ies)");
            }
            if (log.isDebugEnabled()) {
                for (String id : nodeEntityIds) {
                    log.debug(id);
                }
            }

            List<? extends TopiaEntity> nodeEntities = getEntitiesList(srcCtxt, nodeEntityIds.toArray(new String[nodeEntityIds.size()]));

            for (ReplicationOperationDef operationDef : operationDefs) {

                log.info("start " + operationDef);

                TopiaReplicationOperation operation = getOperation(operationDef.getOperationClass());

                operation.run(operationDef, (TopiaContextImplementor) srcCtxt, (TopiaContextImplementor) dstCtxt, nodeEntities, data);

            }
            // on rollback le context source (car on a peut-etre modifie
            // des associations ou des dependances mais on ne veut rien
            // retenir au niveau d'hibernate, sinon on s'expose a des erreurs
            // lorsque l'on veut recharger des objets dans le context...)
            srcCtxt.rollbackTransaction();
        }

        treated.add(node);
    }

    public static <E extends TopiaEntity> List<E> getEntities(TopiaContextImplementor srcCtxt, List<E> entityList, boolean canBeNull) throws TopiaException {
        List<E> srcList = new ArrayList<E>(entityList.size());
        for (E e : entityList) {
            E e2 = (E) srcCtxt.findByTopiaId(e.getTopiaId());
            if (e2 == null && !canBeNull) {
                if (!canBeNull) {
                    throw new IllegalStateException("topia.replication.engine.error.entity.must.exists");
                }
                continue;
            }
            srcList.add(e2);
        }
        return srcList;
    }

    public static TopiaEntity[] getEntities(TopiaContext srcCtxt, String... entityList) throws TopiaException {
        TopiaEntity[] srcList = new TopiaEntity[entityList.length];
        int index = 0;
        for (String id : entityList) {
            TopiaEntity e2 = srcCtxt.findByTopiaId(id);
            srcList[index++] = e2;
        }
        return srcList;
    }

    public static List<? extends TopiaEntity> getEntitiesList(TopiaContext srcCtxt, String... entityList) throws TopiaException {
        List<TopiaEntity> srcList = new ArrayList<TopiaEntity>(entityList.length);
        for (String id : entityList) {
            TopiaEntity e2 = srcCtxt.findByTopiaId(id);
            srcList.add(e2);
        }
        return srcList;
    }

    public static void checkNotNull(String methodName, String parameterName, Object value) {
        if (value == null) {
            throw new NullPointerException(_("topia.replication.engine.error.null.param", parameterName, methodName));
        }
    }

    public static void checkParameters(Class<?>[] paramsType, Object... params) {
        checkSize(paramsType.length, params);
        for (int i = 0, j = paramsType.length; i < j; i++) {
            checkType(paramsType, i, params);
        }
    }

    public static void checkSize(int size, Object[] params) {
        if (params.length != size) {
            throw new IllegalArgumentException("l'operation requiere " + size + " parametres mais en a " + params.length);
        }
    }

    public static void checkType(Class<?>[] paramsType, int index, Object[] params) {
        Class<?> requiredType = paramsType[index];
        Object value = params[index];
        if (value == null) {
            throw new IllegalArgumentException("le parametre de positiion" + index + " est null!");
        }
        Class<?> foundType = value.getClass();

        if (!(requiredType.isAssignableFrom(foundType))) {
            throw new IllegalArgumentException("le paremetre de position " + index + "  requiere un parametre de type " + requiredType + " mais est de type " + foundType);
        }
    }
}
