package org.nuiton.topia.persistence;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: HqlAndParametersBuilder.java 2983 2014-01-18 18:21:51Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-9/topia-persistence/src/main/java/org/nuiton/topia/persistence/HqlAndParametersBuilder.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.util.Arrays;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.text.WordUtils;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * A builder to create syntactically correct HQL and associated parameters given properties or after various constraint
 * adds.
 * <p/>
 * It may be used in a Dao to ease dynamic construction of queries.
 *
 * @since 3.0
 */
public class HqlAndParametersBuilder<E extends TopiaEntity> {

    protected Class<E> entityClass;

    protected Set<String> whereClauses = Sets.newLinkedHashSet();

    protected Map<String, Object> parameters = Maps.newLinkedHashMap();

    protected Set<String> orderByArguments;

    protected String alias = "topiaEntity_";

    public void setAlias(String alias) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(alias));
        this.alias = alias;
    }

    public HqlAndParametersBuilder(Class<E> entityClass) {
        this.entityClass = entityClass;
    }

    public void addNull(String property) {
        Preconditions.checkArgument(StringUtils.isNotBlank(property));
        whereClauses.add(alias + "." + property + " is null");
    }

    public void addNotNull(String property) {
        Preconditions.checkArgument(StringUtils.isNotBlank(property));
        whereClauses.add(alias + "." + property + " is not null");
    }

    public void addEquals(String property, Object value) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(property));
        // TODO brendan 02/10/13 do not use HQL parameters of Object are primitive types
        // TODO brendan 02/10/13 if value is intanceof TopiaEntity, we can check type
        if (value == null) {
            addNull(property);
        } else {
            String hqlParameterName = putHqlParameterWithAvailableName(property, value);
            whereClauses.add(alias + "." + property + " = :" + hqlParameterName);
        }
    }

    public void addNotEquals(String property, Object value) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(property));
        // TODO brendan 02/10/13 do not use HQL parameters of Object are primitive types
        // TODO brendan 02/10/13 if value is intanceof TopiaEntity, we can check type
        if (value == null) {
            addNotNull(property);
        } else {
            String hqlParameterName = putHqlParameterWithAvailableName(property, value);
            whereClauses.add(alias + "." + property + " != :" + hqlParameterName);
        }
    }

    public void addIn(String property, Iterable<?> values) {
        addInOrNotIn(property, values, true);
    }

    public void addNotIn(String property, Iterable<?> values) {
        addInOrNotIn(property, values, false);
    }

    /**
     * @param in true if property value must be in given collection, false if value must not be in given collection
     */
    protected void addInOrNotIn(String property, Iterable<?> values, boolean in) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(property));
        Preconditions.checkNotNull(values);
        // TODO brendan 02/10/13 if value is intanceof TopiaEntity, we can check type
        // TODO brendan 02/10/13 do not use HQL parameters of Object are primitive
        String aliasedProperty = alias + "." + property;
        if (Iterables.size(values) == 1) {
            // if there is only one possible value, replace "in" clause by a "=" clause
            Object onlyElement = Iterables.getOnlyElement(values);
            if (in) {
                addEquals(property, onlyElement);
            } else {
                addNotEquals(property, onlyElement);
            }
        } else {
            boolean propertyMayBeNull = false; // true if values contains null
            Set<String> hqlParameterNames = Sets.newLinkedHashSet();
            for (Object value : values) {
                if (value == null) {
                    propertyMayBeNull = true;
                } else {
                    String hqlParameterName = putHqlParameterWithAvailableName(property, value);
                    hqlParameterNames.add(hqlParameterName);
                }
            }
            String whereClause;
            String inClauseValues;
            if (hqlParameterNames.isEmpty()) {
                inClauseValues = "";
            } else {
                inClauseValues = ":" + StringUtils.join(hqlParameterNames, ", :");
            }
            if (in) {
                String inClause = aliasedProperty + " in (" + inClauseValues + ")";
                if (propertyMayBeNull) {
                    whereClause = aliasedProperty + " is null or " + inClause;
                } else {
                    whereClause = inClause;
                }
            } else {
                String inClause = aliasedProperty + " not in (" + inClauseValues + ")";
                if (propertyMayBeNull) {
                    whereClause = aliasedProperty + " is not null and " + inClause;
                } else {
                    whereClause = inClause;
                }
            }
            whereClauses.add(whereClause);
        }
    }

    public void addTopiaIdEquals(String property, String topiaId) {
        Preconditions.checkNotNull(topiaId);
        addEquals(property + "." + TopiaEntity.PROPERTY_TOPIA_ID, topiaId);
    }

    public void addTopiaIdIn(String property, Iterable<String> topiaIds) {
        addIn(property + "." + TopiaEntity.PROPERTY_TOPIA_ID, topiaIds);
    }

    public void addTopiaIdNotEquals(String property, String topiaId) {
        Preconditions.checkNotNull(topiaId);
        addNotEquals(property + "." + TopiaEntity.PROPERTY_TOPIA_ID, topiaId);
    }

    public void addTopiaIdNotIn(String property, Iterable<String> topiaIds) {
        addNotIn(property + "." + TopiaEntity.PROPERTY_TOPIA_ID, (Iterable) topiaIds);
    }

    public void addContains(String property, Object value) {
        String hqlParameterName = putHqlParameterWithAvailableName(property, value);
        whereClauses.add(":" + hqlParameterName + " in elements(" + alias + "." + property + ")");
    }

    public void addNotContains(String property, Object value) {
        String hqlParameterName = putHqlParameterWithAvailableName(property, value);
        whereClauses.add(":" + hqlParameterName + " not in elements(" + alias + "." + property + ")");
    }

    public void addWhereClause(String whereClause) {
        Preconditions.checkArgument(StringUtils.isNotBlank(whereClause));
        whereClauses.add(whereClause);
    }

    public void addWhereClause(String whereClause, Map<String, Object> hqlParameters) {
        Preconditions.checkNotNull(hqlParameters);
        boolean noCollision = Sets.intersection(parameters.keySet(), hqlParameters.keySet()).isEmpty();
        Preconditions.checkArgument(noCollision, "given arguments " + hqlParameters.keySet() + " contains a conflicting name with the already defined parameters of the query (" + parameters.keySet() + "). You should changes the names of given arguments.");
        addWhereClause(whereClause);
        parameters.putAll(hqlParameters);
    }

    public void setWhereClauses(Set<String> whereClauses) {
        Preconditions.checkNotNull(whereClauses);
        this.whereClauses = whereClauses;
    }

    public Set<String> getWhereClauses() {
        return whereClauses;
    }

    public void setParameters(Map<String, Object> parameters) {
        Preconditions.checkNotNull(parameters);
        this.parameters = parameters;
    }

    public void setOrderByArguments(Set<String> orderByArguments) {
        Preconditions.checkNotNull(orderByArguments);
        this.orderByArguments = orderByArguments;
    }

    public void setOrderByArguments(String... orderByArguments) {
        Set<String> orderByArgumentsAsSet = Sets.newLinkedHashSet();
        orderByArgumentsAsSet.addAll(Arrays.asList(orderByArguments));
        this.orderByArguments = orderByArgumentsAsSet;
    }

    public String getHql() {
        StringBuilder hqlStringBuilder = new StringBuilder();
        hqlStringBuilder.append("from ").append(entityClass.getCanonicalName()).append(" ").append(alias);
        if (!whereClauses.isEmpty()) {
            hqlStringBuilder.append(" where (").append(StringUtils.join(whereClauses, ") and (")).append(")");
        }
        if (CollectionUtils.isNotEmpty(orderByArguments)) {
            hqlStringBuilder.append(" order by ").append(alias).append(".").append(StringUtils.join(orderByArguments, ", " + alias + "."));
        }
        String hql = hqlStringBuilder.toString();
        return hql;
    }

    /**
     * Converts a (nested) property name to an HQL argument name.
     * <p/>
     * For example getParameterName("yearlyDeclaration.survey.topiaId") → "yearlyDeclarationSurveyTopiaId"
     *
     * @param propertyName the name of a property, can be a path to a nested property
     * @return a string that can syntactically be used as an HQL parameter name, not prefixed by ':'
     */
    protected String getParameterName(String propertyName) {
        Preconditions.checkArgument(StringUtils.isNotBlank(propertyName));
        String capitalize = WordUtils.capitalize(propertyName, '.');
        String withoutDots = capitalize.replaceAll("\\.", "");
        String parameterName = StringUtils.uncapitalize(withoutDots);
        return parameterName;
    }

    /**
     * Add a parameter in the parameters map searching with the suitable parameter name in order to prevent conflicts.
     *
     * @return the found key where the parameter has been added, suitable to use in the where clause
     */
    protected String putHqlParameterWithAvailableName(String propertyName, Object value) {
        String parameterNamePrefix = getParameterName(propertyName);
        int suffix = 0;
        String parameterName = parameterNamePrefix;
        while (parameters.containsKey(parameterName)) {
            suffix += 1;
            parameterName = parameterNamePrefix + suffix;
        }
        parameters.put(parameterName, value);
        return parameterName;
    }

    public Map<String, Object> getHqlParameters() {
        return parameters;
    }

    public boolean isOrderByClausePresent() {
        return CollectionUtils.isNotEmpty(orderByArguments);
    }

}
