/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: TopiaDAO.java 2788 2013-08-05 08:20:12Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-2/topia-persistence/src/main/java/org/nuiton/topia/persistence/TopiaDAO.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

/* *
 * TopiaDAO.java
 *
 * Created: 30 déc. 2005 03:00:57
 *
 * @author poussin <poussin@codelutin.com>
 * @version $Revision: 2788 $
 *
 * Last update: $Date: 2013-08-05 10:20:12 +0200 (Mon, 05 Aug 2013) $
 * by : $Author: tchemit $
 */
package org.nuiton.topia.persistence;

import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.event.TopiaEntityListener;
import org.nuiton.topia.event.TopiaEntityVetoable;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.generator.EntityDAOTransformer;
import org.nuiton.topia.persistence.pager.TopiaPagerBean;

import java.security.Permission;
import java.util.List;
import java.util.Map;

/**
 * TopiaDAO is used to manipulate entities corresponding to {@code E} type :
 * create, delete, update or find entities.
 * <p/>
 * This interface is implemented by {@link TopiaDAOImpl} overridden by generation
 * from {@link EntityDAOTransformer}.
 * <p/>
 *
 * @param <E> the entity type managed by the dao
 * @author bpoussin <poussin@codelutin.com>
 * @author fdesbois <fdesbois@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 * @version $Id: TopiaDAO.java 2788 2013-08-05 08:20:12Z tchemit $
 */
public interface TopiaDAO<E extends TopiaEntity> extends Iterable<E> {

    //------------------------------------------------------------------------//
    //-- Create - update - delete methods ------------------------------------//
    //------------------------------------------------------------------------//

    /**
     * Create a new instance of managed entity <strong>not persisted</strong>.
     *
     * @return new entity instance
     * @throws TopiaException if any pb while creating the entity
     * @since 2.3.1
     */
    E newInstance() throws TopiaException;

    /**
     * Creates a new instance of the entity managed by the DAO
     *
     * @param propertyNamesAndValues the list of properties that the created entity will have. Arguments are key-value
     *                               paired : [propertyName;value;propertyName;value;...]
     * @return the newly created entity
     * @throws TopiaException           if any problem during instantiation
     * @throws IllegalArgumentException if the arguments count is not correct or
     *                                  if some property type is not the
     *                                  expected one
     * @see #create(Map)
     */
    E create(Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Creates a new instance of the entity managed by the DAO
     *
     * @param properties the key-value list of properties that the created entity will have.
     * @return the newly created entity
     * @throws TopiaException           if any problem during instantiation
     * @throws IllegalArgumentException if some property type is not the
     *                                  expected one
     */
    E create(Map<String, Object> properties) throws TopiaException;

    /**
     * Creates an entity not created without the DAO. Generally used when DAO
     * is layered after a service layer where the {@link #create(Object...)}
     * would be to onerous.
     *
     * @param e the instance to persist
     * @return the persisted entity (with its topiaId valued)
     * @throws TopiaException if any problem while creating data
     * @since 2.3.1
     */
    E create(E e) throws TopiaException;

    /**
     * Update an entity. May be used for an entity coming from another context.
     *
     * @param e the entity to create or update
     * @return the given entity
     * @throws TopiaException if any problem while updating datas
     */
    E update(E e) throws TopiaException;

    /**
     * Removes the given entity from the storage
     *
     * @param e the entity to remove
     * @throws TopiaException if any problem while deleting datas
     */
    void delete(E e) throws TopiaException;

    /**
     * Permet de supprimer des entités.
     *
     * @param entities les entités à supprimer
     * @throws TopiaException if any pb while deleting datas
     */
    void deleteAll(Iterable<E> entities) throws TopiaException;

    //------------------------------------------------------------------------//
    //-- findByXXX methods ---------------------------------------------------//
    //------------------------------------------------------------------------//

    /**
     * Find an entity corresponding to the {@code id}. If the {@code id} is
     * null, nothing will be searched.
     *
     * @param id topiaId of the entity to found
     * @return the entity found or null if not
     * @throws TopiaException for Topia errors on query
     */
    E findByTopiaId(String id) throws TopiaException;

    /**
     * Find an entity matching {@code value} for the given {@code propertyName}.
     *
     * @param propertyName property name to filter
     * @param value        value of the property to match
     * @return the first entity matching the request
     * @throws TopiaException if any pb while getting datas
     */
    E findByProperty(String propertyName,
                     Object value) throws TopiaException;

    /**
     * Find an entity matching a succession of propertyName + value arguments.
     *
     * @param propertyName           the first property name to filter
     * @param value                  the first value of the property to match
     * @param propertyNamesAndValues other property names and values. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return the first entity matching the request
     * @throws TopiaException if any pb while getting datas
     */
    E findByProperties(String propertyName,
                       Object value,
                       Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Find an entity matching {@code properties}.
     *
     * @param properties the properties key + value to match
     * @return the first entity matching the request
     * @throws TopiaException if any pb while getting datas
     */
    E findByProperties(Map<String, Object> properties) throws TopiaException;

    /**
     * Executes and returns the result (entity E) of the given HQL query string.
     *
     * @param hql                    the HQL query
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return the entity E found or null
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.12
     */
    E findByQuery(String hql,
                  Object... propertyNamesAndValues) throws TopiaException;

    // TODO AThimel 20/07/13 Add this method : E findByQuery(String hql, Map<String, Object> propertyNamesAndValues) throws TopiaException;

    /**
     * Executes and returns the result (entity R) of the given HQL query string.
     *
     * @param type                   the expected result type
     * @param hql                    the HQL query
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return the entity R found or null
     * @throws TopiaException if any pb while getting datas
     * @throws ClassCastException if the found type is not the expected one
     * @since 2.6.12
     */
    <R> R findByQuery(Class<R> type,
                      String hql,
                      Object... propertyNamesAndValues) throws TopiaException;

    // TODO AThimel 20/07/13 Add this method : <R> R findByQuery(Class<R> type, String hql, Map<String, Object> propertyNamesAndValues) throws TopiaException;

    /**
     * Find an entity using the natural ids. Each field of the natural id has to be present in the given Map.
     *
     * @param keys Map with the natural id property name as Map.key, and value as Map.value
     * @return the entity E found or null
     * @throws TopiaException if any pb while getting datas
     */
    E findByPrimaryKey(Map<String, Object> keys) throws TopiaException;

    /**
     * Find an entity using the natural ids. Each field of the natural id has to
     * be present in the given Map.
     *
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return the entity E found or null
     * @throws TopiaException if any pb while getting datas
     */
    E findByPrimaryKey(Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Find the first entity which given collection (propertyName) contains the
     * given value
     *
     * @param propertyName the name of the property (must be a collection)
     * @param value        the value to use for find
     * @return the entity E found or null
     * @throws TopiaException if any pb while getting datas
     * @since 2.5.4
     */
    E findContains(String propertyName,
                   Object value) throws TopiaException;

    //------------------------------------------------------------------------//
    //-- findAllXXX methods --------------------------------------------------//
    //------------------------------------------------------------------------//

    /**
     * Finds all the entities managed by this DAO.
     *
     * @return the full list of entities
     * @throws TopiaException if any pb while getting datas
     */
    List<E> findAll() throws TopiaException;

    /**
     * Finds all entites E managed by this DAO. The returned list will be ordered according to the given
     * {@code propertyNames}.
     * <p/>
     * You can add on each {@code property} {@code ASC} or {@code DESC} to force the result order
     * (by default is {@code ASC}).
     *
     * @param propertyNames property names of order to apply
     * @return all entities E of the dao entity type with given order
     * @throws TopiaException if any pb while getting datas
     */
    List<E> findAllWithOrder(String... propertyNames) throws TopiaException;

    /**
     * Find all the ids for the E entity type
     *
     * @return the full list of ids
     * @throws TopiaException if any pb while getting datas
     */
    // TODO AThimel 20/07/13 This method should return a Set ?
    List<String> findAllIds() throws TopiaException;

    /**
     * Finds all entities E which value for the given {@code propertyName} is
     * {@code value}
     *
     * @param propertyName property name to use
     * @param value        value to expect
     * @return the list of entities E having the given value
     * @throws TopiaException if any pb while getting datas
     */
    List<E> findAllByProperty(String propertyName,
                              Object value) throws TopiaException;

    /**
     * Finds all entities E matching the given {@code propertyName}, {@code value}
     * AND all other properties
     *
     * @param propertyName           property name to use
     * @param value                  value to expect
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return the list of entities E having the given value
     * @throws TopiaException if any pb while getting datas
     */
    List<E> findAllByProperties(String propertyName,
                                Object value,
                                Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Finds all entities E matching all the {@code properties} values.
     *
     * @param properties properties to match
     * @return the list of entities E having the given values
     * @throws TopiaException if any pb while getting datas
     */
    List<E> findAllByProperties(Map<String, Object> properties) throws TopiaException;

    /**
     * Finds all entities E when executing the given HQL query.
     *
     * @param hql                    the HQL query
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return the list of entities E found by the query and parameters
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.12
     */
    List<E> findAllByQuery(String hql,
                           Object... propertyNamesAndValues) throws TopiaException;

    // TODO AThimel 20/07/13 Add this method : List<E> findAllByQuery(String hql, Map<String, Object> propertyNamesAndValues) throws TopiaException;

    /**
     * Gets all entities when executing the given select query for the given
     * {@code type} which may not be a entity type (int, long, map,...).
     *
     * @param type                   the expected result type
     * @param hql                    the HQL query
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return entites of the query result
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.12
     */
    <R> List<R> findAllByQuery(Class<R> type,
                               String hql,
                               Object... propertyNamesAndValues) throws TopiaException;

    // TODO AThimel 20/07/13 Add this method : <R> List<R> findAllByQuery(Class<R> type, String hql, Map<String, Object> propertyNamesAndValues) throws TopiaException;

    /**
     * Finds all entities E in lazy mode when executing the given select query
     * for the dao entity type.
     * <p/>
     * <strong>Important note:</strong> // TODO AThimel 20/07/13 Write the important note...
     *
     * @param hql                    the HQL query
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return entites E of the query result
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.14
     */
    Iterable<E> findAllLazyByQuery(String hql,
                                   Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Finds all entities R in lazy mode when executing the given select query
     * for the given {@code type} which may not be a entity type (int, long, map,...).
     * <p/>
     * <strong>Important note:</strong> // TODO AThimel 20/07/13 Write the important note...
     *
     * @param type                   the expected result type
     * @param hql                    the HQL query
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return entites R of the query result
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.14
     */
    <R> Iterable<R> findAllLazyByQuery(Class<R> type,
                                       String hql,
                                       Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Finds all entities E in lazy mode when executing the given select query
     * for the dao entity type.
     * <p/>
     * <strong>Important note:</strong> // TODO AThimel 20/07/13 Write the important note...
     *
     * @param batchSize              batch size
     * @param hql                    the HQL query
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return entites E of the query result
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.14
     */
    Iterable<E> findAllLazyByQuery(int batchSize,
                                   String hql,
                                   Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Finds all entities R in lazy mode when executing the given select query
     * for the given {@code type} which may not be a entity type (int, long, map,...).
     * <p/>
     * <strong>Important note:</strong> // TODO AThimel 20/07/13 Write the important note...
     *
     * @param type                   the expected result type
     * @param batchSize              batch size
     * @param hql                    the HQL query
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return entites R of the query result
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.14
     */
    <R> Iterable<R> findAllLazyByQuery(Class<R> type,
                                       int batchSize,
                                       String hql,
                                       Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Finds a page of entities E when executing the given select query for the dao
     * entity type (will only return the window of {@code startIndex -
     * endIndex} entities).
     * // TODO AThimel 20/07/13 Reformulate(?) the "window thing"
     *
     * @param hql                    the HQL query
     * @param startIndex             first index of entity to return
     * @param endIndex               last index of entity to return
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return entites E of the paginated query result
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.12
     */
    List<E> findAllByQueryWithBound(String hql,
                                    int startIndex,
                                    int endIndex,
                                    Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Finds a page of entities R when executing the given select query for the dao
     * entity type (will only return the window of {@code startIndex -
     * endIndex} entities).
     * // TODO AThimel 20/07/13 Reformulate(?) the "window thing"
     *
     * @param type                   the expected result type
     * @param hql                    the HQL query
     * @param startIndex             first index of entity to return
     * @param endIndex               last index of entity to return
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return entites R of the paginated query result
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.12
     */
    <R> List<R> findAllByQueryWithBound(Class<R> type,
                                        String hql,
                                        int startIndex,
                                        int endIndex,
                                        Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Gets a page of entities E of the given select {@code hql} query using the
     * {@code pager} to obtain the window of entities to return.
     *
     * @param hql                    the HQL query
     * @param pager                  pager to obtain the correct window of data
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return entities E of the paginated query result
     * @throws TopiaException if any pb while getting datas
     * @see TopiaPagerBean
     * @since 2.6.12
     */
    List<E> findAllByQueryAndPager(String hql,
                                   TopiaPagerBean pager,
                                   Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Finds a page of entities R of the given select {@code hql} query using the
     * {@code pager} to obtain the window of entities to return.
     *
     * @param type                   the expected result type
     * @param hql                    the HQL query
     * @param pager                  pager to obtain the correct window of data
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return entities R of the paginated query result
     * @throws TopiaException if any pb while getting datas
     * @see TopiaPagerBean
     * @since 2.6.12
     */
    <R> List<R> findAllByQueryAndPager(Class<R> type,
                                       String hql,
                                       TopiaPagerBean pager,
                                       Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Find all the entities E which given collection (propertyName) contains the
     * given value
     *
     * @param propertyName the name of the property (must be a collection)
     * @param value        the value to use for find
     * @return all the entities E found
     * @throws TopiaException if any pb while getting datas
     * @since 2.5.4
     */
    List<E> findAllContains(String propertyName,
                            Object value) throws TopiaException;

    //------------------------------------------------------------------------//
    //-- existsByXXX methods -------------------------------------------------//
    //------------------------------------------------------------------------//

    /**
     * Check the existence of an entity with technical {@code id}.
     *
     * @param id unique id of the entity to test existence.
     * @return true if entity exists, false otherwise
     * @throws TopiaException for any error
     * @since 2.3.4
     */
    boolean existByTopiaId(String id) throws TopiaException;

    /**
     * Check the existence of an entity with {@code propertyName} with {@code
     * propertyValue}. {@code others} properties can be added to test
     * existence.
     *
     * @param propertyName           the first property name to test existence
     * @param propertyValue          the first property value to test existence
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return true if entity exists, false otherwise
     * @throws TopiaException for Topia errors
     * @since 2.3.4
     */
    boolean existByProperties(String propertyName,
                              Object propertyValue,
                              Object... propertyNamesAndValues) throws TopiaException;

    /**
     * Check the existence of an entity using the given HQL query.
     *
     * @param hql                    the HQL query
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return true if entity exists, false otherwise
     * @throws TopiaException
     * @since 2.6.12
     */
    boolean existsByQuery(String hql,
                          Object... propertyNamesAndValues) throws TopiaException;

    //------------------------------------------------------------------------//
    //-- countXXX methods ----------------------------------------------------//
    //------------------------------------------------------------------------//

    /**
     * Count the number of existing entities.
     *
     * @return number of total entities
     * @throws TopiaException if any pb while getting datas
     * @since 2.3.4
     */
    long count() throws TopiaException;

    /**
     * Count the number of entities based on a the given HQL query.
     *
     * @param hql                    the HQL query to use
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @return number of entities filtered by the query
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.12
     */
    long countByQuery(String hql,
                      Object... propertyNamesAndValues) throws TopiaException;

    //------------------------------------------------------------------------//
    //-- other request methods -----------------------------------------------//
    //------------------------------------------------------------------------//

    /**
     * Find usages of the given {@code entity} in the entities of the given
     * {@code type}.
     *
     * @param type   the type of entity to search
     * @param entity the entity on which search is done
     * @param <R>    type of entity to search
     * @return the list of entities R which uses the given entity
     * @throws TopiaException if any problem while getting data
     * @since 2.3.0
     */
    <R extends TopiaEntity> List<R> findUsages(Class<R> type,
                                               E entity) throws TopiaException;

    /**
     * Find all usages of the given {@code entity}.
     *
     * @param entity the entity
     * @return the dictionnary of usages of the given entities (keys are entity
     *         usage container, values are the list of this type of entity to
     *         use the given entity).
     * @throws TopiaException if any pb while getting data
     * @since 2.3.0
     */

    Map<Class<? extends TopiaEntity>, List<? extends TopiaEntity>> findAllUsages(E entity) throws TopiaException;

    /**
     * Execute the count {@code hql} query and then synch the pager to this
     * result (says fill the
     * {@link TopiaPagerBean#records} field and then adapt
     * the number of pages available and the current number page).
     *
     * @param hql                    the HQL query to use
     * @param pager                  pager to obtain the correct window of data
     * @param propertyNamesAndValues the query parameters. Arguments are key-value paired :
     *                               [propertyName;value;propertyName;value;...]
     * @throws TopiaException if any pb while getting datas
     * @see TopiaPagerBean
     * @since 2.6.12
     */
    void computeAndAddRecordsToPager(String hql,
                                     TopiaPagerBean pager,
                                     Object... propertyNamesAndValues) throws TopiaException;

    //------------------------------------------------------------------------//
    //-- Misc methods --------------------------------------------------------//
    //------------------------------------------------------------------------//

    /**
     * Return the class of the entity managed by this DAO.
     *
     * @return this DAO's managed entity's class
     */
    Class<E> getEntityClass();

    /**
     * Returns the context used by this DAO.
     *
     * @return Returns the context.
     * @deprecated use {@link #getTopiaContext()}
     */
    @Deprecated
    TopiaContextImplementor getContext();

    /**
     * Returns the context used by this DAO.
     *
     * @return Returns the context.
     */
    // TODO AThimel 02/08/13 Use only TopiaPersistenceContext and TopiaHibernateSupport
    TopiaContext getTopiaContext();

    /**
     * Get the entityEnum of the type of entity managed by this DAO.
     *
     * @return entity type enum managed by this DAO
     */
    TopiaEntityEnum getTopiaEntityEnum();

    /**
     * Obtains the batch size used to load data.
     * <p/>
     * Default value if 1000.
     *
     * @return the batch size.
     * @since 2.6.14
     */
    int getBatchSize();

    /**
     * Set a new default batch size.
     *
     * @param batchSize new batch size to use when iterating.
     * @since 2.6.14
     */
    void setBatchSize(int batchSize);

    /**
     * Create the simple HQL query for the entity managed by the dao.
     * <p/>
     * A optional alias can be passed:
     * <p/>
     * <pre>FROM MyEntityImpl myAlias</pre>
     *
     * @param alias optional alias to use in query
     * @return the hql query
     * @since 2.6.14
     */
    String createSimpleQuery(String alias);

    //------------------------------------------------------------------------//
    //-- Listener methods ----------------------------------------------------//
    //------------------------------------------------------------------------//

    // TODO AThimel 20/07/13 Javadoc
    void addTopiaEntityListener(TopiaEntityListener listener);

    // TODO AThimel 20/07/13 Javadoc
    void addTopiaEntityVetoable(TopiaEntityVetoable vetoable);

    // TODO AThimel 20/07/13 Javadoc
    void removeTopiaEntityListener(TopiaEntityListener listener);

    // TODO AThimel 20/07/13 Javadoc
    void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable);

    /**
     * Retourne les permissions a verifier pour l'acces a l'entite pour le
     * service Taas.
     *
     * @param topiaId topiaId d'une entite
     * @param actions encoded actions
     * @return la liste des permissions
     * @throws TopiaException if any pb while getting datas
     * @since 2.6.14
     * @deprecated topia-service-security will be removed in 3.0
     */
    @Deprecated
    List<Permission> getRequestPermission(String topiaId,
                                          int actions) throws TopiaException;

} //TopiaDAO

