/*
 * #%L
 * Maven License Plugin
 * 
 * $Id: UpdateFileHeaderFilter.java 1832 2010-09-29 13:49:21Z ymartel $
 * $HeadURL: http://svn.nuiton.org/svn/maven-license-plugin/tags/maven-license-plugin-2.4/src/main/java/org/nuiton/license/plugin/header/UpdateFileHeaderFilter.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.license.plugin.header;

import org.nuiton.license.plugin.header.transformer.FileHeaderTransformer;

/**
 * Implementation of {@link FileHeaderFilter} to update an incoming header.
 *
 * @author tchemit <chemit@codelutin.com>
 * @plexus.component role="org.nuiton.license.plugin.header.FileHeaderFilter" role-hint="update-file-header"
 * @since 2.1
 */
public class UpdateFileHeaderFilter extends FileHeaderFilter {

    /**
     * Flag sets to {@code true} if description can be updated.
     *
     * @since 2.3.2
     */
    protected boolean updateDescription;

    /**
     * Flag set to {@code true} if license can be updated.
     *
     * @since 2.3.2
     */
    protected boolean updateLicense;

    /**
     * Flag sets to {@code true} if copyright can be updated.
     *
     * @since 2.1
     */
    protected boolean updateCopyright;

    public UpdateFileHeaderFilter() {
    }

    @Override
    protected FileHeader getNewHeader(FileHeader oldHeader) {

        FileHeader result = new FileHeader();

        FileHeader newHeader = getFileHeader();

        FileHeaderTransformer transformer = getTransformer();

        boolean modified = false;

        // by default, reuse the old header
        result.setDescription(oldHeader.getDescription());
        result.setCopyrightFirstYear(oldHeader.getCopyrightFirstYear());
        result.setCopyrightLastYear(oldHeader.getCopyrightLastYear());
        result.setCopyrightHolder(oldHeader.getCopyrightHolder());
        result.setLicense(oldHeader.getLicense());

        if (isUpdateDescription() &&
            !transformer.isDescriptionEquals(oldHeader, newHeader)) {

            // can update description and it has changed

            if (log.isDebugEnabled()) {
                log.debug("description has changed from [" + oldHeader.getDescription() + "] to [" + newHeader.getDescription() + "]");
            }

            // description has changed, mark header to be updated
            modified = true;

            // use the new description
            result.setDescription(newHeader.getDescription());
        }

        if (isUpdateCopyright() &&
            !transformer.isCopyrightEquals(oldHeader, newHeader)) {

            // can update copyright and it has changed

            if (log.isDebugEnabled()) {
                log.debug("copyright has changed from [" + oldHeader.getCopyright() + "] to [" + newHeader.getCopyright() + "]");
            }

            // description has changed, mark header to be updated
            modified = true;

            // use the new copyright
            result.setCopyrightFirstYear(result.getCopyrightFirstYear());
            result.setCopyrightLastYear(result.getCopyrightLastYear());
            result.setCopyrightHolder(result.getCopyrightHolder());
        }

        if (isUpdateLicense() &&
            !transformer.isLicenseEquals(oldHeader, newHeader)) {

            // can update license and it has changed

            if (log.isDebugEnabled()) {
                log.debug("license has changed from [" + oldHeader.getLicense() + "] to [" + newHeader.getLicense() + "]");
            }

            // description has changed, mark header to be updated
            modified = true;

            // use the new license
            result.setLicense(newHeader.getLicense());
        }
        
        if (!modified) {

            // nothing has to be updated, so return a {@code null} result
            result = null;
        }

        return result;
    }

    @Override
    @Deprecated
    protected String treateHeader(String ch) {

        if (getLog().isDebugEnabled()) {
            getLog().debug("header\n" + ch);
        }

        FileHeaderTransformer transformer = getTransformer();

        // unbox comment
        String unbox = transformer.unboxComent(ch);

        if (getLog().isDebugEnabled()) {
            getLog().info("unboxed comment header\n" + unbox);
        }

        // obtain current license of file
        FileHeader fileHeader = transformer.toFileHeader(unbox);

        // updates license
        fileHeader.setLicense(getFileHeader().getLicense());

        if (isUpdateCopyright()) {
            // 1) obtain the svn last comit on this file

            // 2) compute the last year of copyright

            // 3) if necessary update last year
            //fileHeader.setCopyrightLastYear(lastYear);
        }
        // build new brut header (with no boxing)
        String newHeader = transformer.toString(fileHeader);

        // box with process tag
        newHeader = transformer.boxProcessTag(newHeader);

        // box header with comment prefix
        newHeader = transformer.boxComment(newHeader, false);

        // remove all before process start tag
        // remove all after process end tag
        // this is a requirement for processor to respect involution.
        int index = newHeader.indexOf(transformer.getProcessStartTag());
        int lastIndex =
                newHeader.lastIndexOf(transformer.getProcessEndTag()) +
                transformer.getProcessEndTag().length();

        String result = newHeader.substring(index, lastIndex);

        if (getLog().isDebugEnabled()) {
            getLog().debug("updated Header =\n" + result);
        }
        return result;
    }

    public boolean isUpdateCopyright() {
        return updateCopyright;
    }

    public void setUpdateCopyright(boolean updateCopyright) {
        this.updateCopyright = updateCopyright;
    }

    public boolean isUpdateDescription() {
        return updateDescription;
    }

    public void setUpdateDescription(boolean updateDescription) {
        this.updateDescription = updateDescription;
    }

    public boolean isUpdateLicense() {
        return updateLicense;
    }

    public void setUpdateLicense(boolean updateLicense) {
        this.updateLicense = updateLicense;
    }
}
