/*
 * #%L
 * NuitonMatrix
 * 
 * $Id: MatrixFactory.java 471 2014-04-10 12:44:04Z bpoussin $
 * $HeadURL: https://svn.nuiton.org/nuiton-matrix/tags/nuiton-matrix-2.4.1/nuiton-matrix/src/main/java/org/nuiton/math/matrix/MatrixFactory.java $
 * %%
 * Copyright (C) 2004 - 2011 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.math.matrix;

import java.lang.reflect.Constructor;
import java.util.List;

/**
 * Cette classe permet de creer des matrices, toutes les creations de matrice
 * doivent etre faite a travers cette classe. Cette classe permet de modifier la
 * representation interne des matrices de facon simple.
 * <p>
 * Created: 11 octobre 2005 20:15:20 CEST
 * 
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 * @version $Revision: 471 $
 * 
 * Last update: $Date: 2014-04-10 14:44:04 +0200 (Thu, 10 Apr 2014) $ by :
 * $Author: bpoussin $
 */
public class MatrixFactory { // MatrixFactory

    /**
     * If true, createVector return all time LazyVector to prevent memory
     * allocation when not necessary. LazyVector init real vector only when
     * necessary
     */
    protected static boolean useLazyVector = true;

    /** Valeur par defaut si aucun type de Vector n'est donné */
    protected static Class<?> defaultVectorClass = DoubleBigVector.class;

    protected Class<?> vectorClass = null;

    protected static SemanticMapper defaultSemanticMapper = new SemanticMapper();

    private static ThreadLocal<MatrixFactory> matrixFactoryThreadLocal =
            new ThreadLocal<MatrixFactory>();


    protected MatrixFactory(Class<?> vectorClass) {
        this.vectorClass = vectorClass;
    }

    public static void setDefaultVectorClass(Class<?> vectorClass) {
        defaultVectorClass = vectorClass;
    }

    public static Class<?> getDefaultVectorClass() {
        return defaultVectorClass;
    }

    public static void setSemanticMapper(SemanticMapper semanticMapper) {
        defaultSemanticMapper = semanticMapper;
    }

    public static SemanticMapper getSemanticMapper() {
        return defaultSemanticMapper;
    }

    /**
     * Retourne une factory utilisant vectorClass comme classe de base a
     * l'implantation des matrices.
     * 
     * @param vectorClass vector class implementation
     * @return factory
     */
    public static MatrixFactory getInstance(Class<?> vectorClass) {
        return new MatrixFactory(vectorClass);
    }

    /**
     * Utilise par defaut {@link #defaultVectorClass}. Si une factory a ete
     * initialisee pour le thread local, alors celle-ci sera reutilise, sinon
     * une nouvelle est cree avec le backend par defaut
     * 
     * @return factory
     */
    public static MatrixFactory getInstance() {
        MatrixFactory result = matrixFactoryThreadLocal.get();
        if (result == null) {
            result = getInstance(defaultVectorClass);
        }
        return result;
    }

    /**
     * Initialise une factory pour le thread courant avec le vectorClass passe
     * en parametre.
     * @param vectorClass
     */
    public static void initMatrixFactoryThreadLocal(Class<?> vectorClass) {
        matrixFactoryThreadLocal.set(getInstance(vectorClass));
    }

    /**
     * Retire la factory pour le thread courant
     */
    public static void removeMatrixFactoryThreadLocal() {
        matrixFactoryThreadLocal.remove();
    }

    public MatrixND create(int[] dim) {
        return new MatrixNDImpl(this, dim);
    }

    /**
     * Convert a double array into matrix.
     * 
     * @param values The values to fill the matrix
     * @param dim An array representing the dimensions of the matrix
     * @return a 2D matrix filled with the values, null if the dimension is more
     *         than 2
     */
    public MatrixND create(double[] values, int[] dim) {

        if (dim.length > 2) {
            return null;
        }
        MatrixNDImpl matrix = new MatrixNDImpl(this, dim);

        if (dim.length == 2) {
            for (int i = 0; i < dim[0]; i++) {
                for (int j = 0; j < dim[1]; j++) {
                    int[] coordinates = { i, j };
                    matrix.setValue(coordinates, values[i * dim[1] + j]);
                }
            }
        }
        if (dim.length == 1) {
            for (int i = 0; i < dim[0]; i++) {
                int[] coordinates = { i };
                matrix.setValue(coordinates, values[i]);
            }
        }

        return matrix;
    }

    public MatrixND create(List<?>[] semantics) {
        return new MatrixNDImpl(this, semantics);
    }

    public MatrixND create(String name, int[] dim) {
        return new MatrixNDImpl(this, name, dim);
    }

    public MatrixND create(String name, int[] dim, String[] dimNames) {
        return new MatrixNDImpl(this, name, dim, dimNames);
    }

    /**
     * Create new matrix and force backend to be data.
     * 
     * @param name matrix's name (can be null)
     * @param dim matrix's dimension (must be provided)
     * @param dimNames dimensions' name (can be null)
     * @param data backend used to new matrix
     * @return  new Matrix with specified backend
     */
    public MatrixND create(String name, int[] dim, String[] dimNames, Vector data) {
        return new MatrixNDImpl(this, name, dim, dimNames);
    }

    public MatrixND create(String name, List<?>[] semantics) {
        return new MatrixNDImpl(this, name, semantics);
    }

    public MatrixND create(String name, List<?>[] semantics, String[] dimNames) {
        return new MatrixNDImpl(this, name, semantics, dimNames);
    }

    /**
     * Create new matrix and force backend to be data.
     *
     * @param name matrix's name (can be null)
     * @param semantics semantiques (must be provided)
     * @param dimNames dimensions' name (can be null)
     * @param data backend used to new matrix
     * @return new Matrix with specified backend
     */
    public MatrixND create(String name, List<?>[] semantics, String[] dimNames, Vector data) {
        return new MatrixNDImpl(this, name, semantics, dimNames, data);
    }

    /**
     * Create new matrix by copying matrix in argument
     * @param matrix
     * @return
     */
    public MatrixND create(MatrixND matrix) {
        return new MatrixNDImpl(this, matrix);
    }

    /**
     * Create new matrix by copying matrix in argument, and force backend
     * to be data.
     * @param matrix matrix to copied
     * @param data backend used to new matrix
     */
    public MatrixND create(MatrixND matrix, Vector data) {
        return new MatrixNDImpl(this, matrix, data);
    }

    /**
     * Crée une nouvelle matrice identité. Une matrice identité est une matrice
     * à 2 dimensions dont tous les éléments de la diagonal vaut 1
     * 
     * @param size la taille de la matrice
     * @return une nouvelle matrice identité
     */
    public MatrixND matrixId(int size) {
        MatrixND result = create(new int[] { size, size });
        for (int i = 0; i < size; i++) {
            result.setValue(i, i, 1);
        }
        return result;
    }

    protected Vector createVector(int length) {
        try {
            Vector result = (Vector)vectorClass.newInstance();
            if (useLazyVector) {
                result = new LazyVector(result, length);
            }
            result.init(length);
            return result;
        } catch (Exception eee) {
            throw new RuntimeException("Can't create vector", eee);
        }
    }
    
    public MatrixProxy createProxy(List<?>[] semantics, MatrixProvider matrixProvider) {
        MatrixProxy matrixProxy = new MatrixProxy(this, semantics);
        matrixProxy.setMatrixProvider(matrixProvider);
        return matrixProxy;
    }

    public MatrixProxy createProxy(String name, int[] dim, MatrixProvider matrixProvider) {
        MatrixProxy matrixProxy = new MatrixProxy(this, name, dim);
        matrixProxy.setMatrixProvider(matrixProvider);
        return matrixProxy;
    }

    public MatrixProxy createProxy(String name, int[] dim, String[] dimNames, MatrixProvider matrixProvider) {
        MatrixProxy matrixProxy = new MatrixProxy(this, name, dim, dimNames);
        matrixProxy.setMatrixProvider(matrixProvider);
        return matrixProxy;
    }

    public MatrixProxy createProxy(String name, List<?>[] semantics, MatrixProvider matrixProvider) {
        MatrixProxy matrixProxy = new MatrixProxy(this, name, semantics);
        matrixProxy.setMatrixProvider(matrixProvider);
        return matrixProxy;
    }

    public MatrixProxy createProxy(String name, List<?>[] semantics, String[] dimNames, MatrixProvider matrixProvider) {
        MatrixProxy matrixProxy = new MatrixProxy(this, name, semantics, dimNames);
        matrixProxy.setMatrixProvider(matrixProvider);
        return matrixProxy;
    }

} // MatrixFactory
