package jaxx.runtime.swing.navigation;

import jaxx.runtime.JAXXAction;
import jaxx.runtime.JAXXContextEntryDef;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.swing.navigation.NavigationTreeModel.NavigationTreeNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.JTree;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreePath;
import java.awt.Component;

/** A {@link javax.swing.event.TreeSelectionListener}  implementation@author chemit */
public abstract class NavigationTreeSelectionAdapter implements TreeSelectionListener {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(NavigationTreeSelectionAdapter.class);

    //static public final String NAVIGATION_CONTEXT_PATH = "navigation-context-path";

    //static public final String NAVIGATION_SELECTED_NODE = "navigation-selected-node";

    static public final String NAVIGATION_SELECTED_BEAN = "navigation-selected-bean";

    static public final JAXXContextEntryDef<String> NAVIGATION_SELECTED_PATH_ENTRY_DEF = JAXXContextEntryDef.newDef("navigation-selected-path", String.class);

    static public final JAXXContextEntryDef<NavigationTreeNode> NAVIGATION_SELECTED_NODE_ENTRY_DEF = JAXXContextEntryDef.newDef("navigation-selected-node", NavigationTreeNode.class);

    static public final JAXXContextEntryDef<Boolean> GO_BACK_DEF = JAXXContextEntryDef.newDef("goBack", Boolean.class);

    /** defined the stategy of instanciation of ui */
    public enum Strategy {
        /** instanciate a ui for a node */
        PER_NODE,
        /** instanciate only one a ui for a type,nodes will share the instanciation */
        PER_UI_TYPE
    }

    /** la classe d'ui par defaut, associé à un noeud de l'arbe */
    protected Class<? extends JAXXObject> defaultUIClass;

    protected Class<? extends JAXXAction> defaultUIHandlerClass;

    /** l'ui contenant l'arbre de navigation */
    protected JAXXObject context;

    protected Strategy strategy;

    protected NavigationTreeSelectionAdapter(Class<? extends JAXXObject> defaultUIClass, Class<? extends JAXXAction> defaultUIHandlerClass, JAXXObject context, Strategy strategy) {
        this.defaultUIClass = defaultUIClass;
        this.defaultUIHandlerClass = defaultUIHandlerClass;
        this.context = context;
        this.strategy = strategy;
    }


    protected abstract NavigationTreeModel getNavigationTreeModel();

    /**
     * @return le composent actuellement visible associé au noeud courant ou au noeud précédent
     *         lors d'un changement de noeud.
     */
    protected abstract Component getCurrentUI();

    /**
     * @param node le noeud associé à l'ui à retrouver
     * @return l'ui associé au novueau noeud sélectionné
     */
    protected abstract Component getUI(NavigationTreeNode node);

    /**
     * @param event     l'evenement de selection de noeud
     * @param component le composent actuellement visible
     * @return <code>true</code> si le composent a bien été fermé, <code>false</code> sinon
     * @throws Exception if any
     */
    protected abstract boolean closeUI(TreeSelectionEvent event, Component component) throws Exception;

    /**
     * Instancie une nouvelle ui associé à un noeud de l'arbre de navigation
     *
     * @param node le noeud associé à l'ui à créer
     * @return la nouvelle ui associée au noeud
     * @throws Exception if any
     */
    protected abstract Component createUI(NavigationTreeNode node) throws Exception;

    /**
     * Ouvre l'ui associée au noeud sélectionné dans l'arbre de navigation.
     *
     * @param newUI l'ui associé au noeud sélectionné à ouvrir
     * @param node  le node de l'ui a ouvrir
     * @throws Exception if any
     */
    protected abstract void openUI(Component newUI, NavigationTreeNode node) throws Exception;

    /**
     * Retourne au noeud précdemment sélectionné dans l'arbre de navigation, avec la possibilité de notifier
     * une erreure survenue.
     *
     * @param event l'évènement de changement de noeud sélectionné.
     * @param e     l'erreur recontrée (ou null si pas d"erreur)
     */
    protected abstract void goBackToPreviousNode(TreeSelectionEvent event, Exception e);

    /**
     * Prepare le nouveau noeud sélectionné.
     *
     * @param event l'évènement de selection du noeud
     * @return le noeud selectionné et preparé
     */
    protected NavigationTreeNode prepareNode(TreeSelectionEvent event) {
        NavigationTreeNode node = (NavigationTreeNode) event.getPath().getLastPathComponent();

        if (node.getJaxxClass() == null) {
            // no ui is associated with this node, display a empty content
            node.setJaxxClass(defaultUIClass);
        }

        if (node.getJaxxActionClass() == null) {
            node.setJaxxActionClass(defaultUIHandlerClass);
        }
        return node;
    }

    @Override
    public void valueChanged(TreeSelectionEvent event) {
        if (event.getOldLeadSelectionPath() != null && event.getOldLeadSelectionPath().equals(event.getPath())) {
            // do not treate this if no path changed
            return;
        }

        Boolean goBack = GO_BACK_DEF.getContextValue(context);
        if (goBack != null && goBack) {
            // do not treate this, apsecial flag told us :)
            GO_BACK_DEF.removeContextValue(context);
            return;
        }

        try {

            NavigationTreeNode node = prepareNode(event);

            String path = node.getContextPath();

            if (log.isTraceEnabled()) {
                log.trace(path);
            }

            Component newUI = getUI(node);
            Component component = getCurrentUI();

            if (newUI != null && strategy == Strategy.PER_NODE && newUI.equals(component)) {
                // call back from goto back to previous node, do nothing
                return;
            }

            if (!closeUI(event, component)) {
                GO_BACK_DEF.setContextValue(context, Boolean.TRUE);
                // previous ui was not closed, so reselect the previous node in navigation
                goBackToPreviousNode(event, null);
                // and quit
                return;
            }

            // now, we are free to open the ui associated with the selected node in navigation

            // always clean cache on the node before all
            node.cachedBean = null;
            if (node.renderer != null) {
                node.renderer.setRendererCachedValue(null);
            }
            // before all, attach bean in context associated with the selected node in naivgation tree
            Object data = getNavigationTreeModel().getJAXXContextValue(context, path);

            addSelectedBeanInContext(node, data);

            if (newUI == null) {
                // instanciate a new ui associated with the selected node
                newUI = createUI(node);
            }

            // save in context current node context path
            NAVIGATION_SELECTED_PATH_ENTRY_DEF.setContextValue(context, node.getContextPath());

            // save in context current node
            NAVIGATION_SELECTED_NODE_ENTRY_DEF.setContextValue(context, node);

            // really open the ui associated with the selected node
            openUI(newUI, node);

        } catch (Exception e) {
            // remove data from context

            // if any error, go back to previvous node
            goBackToPreviousNode(event, e);
        }
    }

    protected void addSelectedBeanInContext(NavigationTreeNode node, Object data) {

        if (log.isDebugEnabled()) {
            log.debug("find data for contextPath <" + node.getContextPath() + "> : " + (data == null ? null : data.getClass()));
        }

        context.removeContextValue(Object.class, NAVIGATION_SELECTED_BEAN);

        if (data != null) {
            context.setContextValue(data, NAVIGATION_SELECTED_BEAN);
            //todo should we not use this to avoid conflict in context ?
            context.setContextValue(data);
        }
    }

    protected String getNodeConstraints(NavigationTreeNode node) {
        String constraints;
        switch (strategy) {
            case PER_NODE:
                constraints = node.getContextPath();
                break;
            case PER_UI_TYPE:
                constraints = node.getJaxxClass().getName();
                break;
            default:
                throw new IllegalArgumentException("could not find constraint for node : " + node);
        }
        return constraints;
    }

    protected void returnToPreviousNode(JTree tree, TreeSelectionEvent event) {
        // go back to previous node
        // put in context a tag to not come back again here
        TreePath oldPath = event.getOldLeadSelectionPath();
        //NavigationTreeNode oldNode = (NavigationTreeNode) oldPath.getLastPathComponent();
        if (oldPath != null) {
            tree.setSelectionPath(oldPath);
        }
    }

}