/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.reflect;

import jaxx.compiler.CompilerException;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.JAXXEngine;
import jaxx.compiler.SymbolTable;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.JAXXObjectDescriptor;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Mirrors the class <code>java.lang.ClassLoader</code>.  JAXX uses <code>ClassDescriptor</code> instead of <code>Class</code>
 * almost everywhere so that it can handle circular dependencies (there can't be a <code>Class</code> object for an uncompiled
 * JAXX or Java source file, and a compiler must be allow references to symbols in uncompiled source files in order to handle
 * circular dependencies).
 */
public class ClassDescriptorLoader {

    private static Map<String, ClassDescriptor> descriptors = new HashMap<String, ClassDescriptor>();

    private ClassDescriptorLoader() {
    }

    public static synchronized ClassDescriptor getClassDescriptor(String className) throws ClassNotFoundException {
        return getClassDescriptor(className, Thread.currentThread().getContextClassLoader());
        //return getClassDescriptor(className, ClassDescriptorLoader.class.getClassLoader());
    }

    public static synchronized ClassDescriptor getClassDescriptor(String className, ClassLoader classLoader) throws ClassNotFoundException {
        ClassDescriptor result = descriptors.get(className);
        if (result != null) {
            return result;
        }

//        if (result == null) {
        if (JAXXEngine.isRegistred() && JAXXEngine.get().getSymbolTable(className) != null) {
            result = createClassDescriptorFromSymbolTable(className, classLoader);
        } else {
            if (classLoader == null) {
                classLoader = ClassDescriptorLoader.class.getClassLoader();
            }

            String relativePath = className.replaceAll("\\.", "/");
            String relativePathPattern = ".*";// + className + ".*"; // used to ensure that the located resource has the right character cases

            // find the most recently updated source for the class -- Java source, JAXX source, or compiled class file
            long javaLastModified = -1;
            URL javaFile = classLoader.getResource(relativePath + ".java");
            if (javaFile != null && javaFile.toString().startsWith("file:") && javaFile.toString().matches(relativePathPattern)) {
                javaLastModified = JAXXCompiler.URLtoFile(javaFile).lastModified();
            }

            long classLastModified = -1;
            URL classFile = classLoader.getResource(relativePath + ".class");
            if (classFile != null && classFile.toString().startsWith("file:") && classFile.toString().matches(relativePathPattern)) {
                classLastModified = JAXXCompiler.URLtoFile(classFile).lastModified();
            }

            long jaxxLastModified = -1;
            URL jaxxFile = classLoader.getResource(relativePath + ".jaxx");
            if (jaxxFile != null && jaxxFile.toString().startsWith("file:") && jaxxFile.toString().matches(relativePathPattern)) {
                File jaxxFilePath = JAXXCompiler.URLtoFile(jaxxFile);
                jaxxLastModified = jaxxFilePath.lastModified();
                String simplePath = jaxxFilePath.getPath();
                simplePath = simplePath.substring(0, simplePath.length() - ".jaxx".length());
                File cssFilePath = new File(simplePath + ".css");
                if (cssFilePath.exists()) {
                    jaxxLastModified = Math.max(jaxxLastModified, cssFilePath.lastModified());
                }
                File scriptFilePath = new File(simplePath + ".script");
                if (scriptFilePath.exists()) {
                    jaxxLastModified = Math.max(jaxxLastModified, scriptFilePath.lastModified());
                }
            }

            if (jaxxLastModified != -1 && JAXXEngine.isRegistred() && JAXXEngine.get().getSymbolTable(className) == null) {
                jaxxLastModified = -1; // file has been modified, but wasn't included in this
                }
            // compilation set so we don't have a symbol table

            if (javaLastModified != -1 || classLastModified != -1 || jaxxLastModified != -1) {
                if (jaxxLastModified > classLastModified && jaxxLastModified > javaLastModified) {
                    result = createClassDescriptorFromSymbolTable(className, classLoader);
                } else if (javaLastModified > classLastModified && javaLastModified > jaxxLastModified) {
                    result = createClassDescriptorFromJavaSource(javaFile, classLoader);
                }
            }
            // else work off of the class file.  This also handles the case where the class is available, but wasn't in a location where
            // we could check its last modified date (in a JAR, over the network, etc.)
            if (result == null) {
                Class<?> javaClass = getClass(className, classLoader);
                result = createClassDescriptorFromClass(javaClass);
            }
        }
        descriptors.put(className, result);
//        }
        return result;
    }

    public static ClassDescriptor getClassDescriptor(Class<?> javaClass) {
        try {
            return getClassDescriptor(javaClass.getName(), javaClass.getClassLoader());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    public static Class<?> getPrimitiveBoxedClass(String className) throws ClassNotFoundException {
        if (className.equals("boolean")) {
            return Boolean.class;
        }
        if (className.equals("byte")) {
            return Byte.class;
        }
        if (className.equals("short")) {
            return Short.class;
        }
        if (className.equals("int")) {
            return Integer.class;
        }
        if (className.equals("long")) {
            return Long.class;
        }
        if (className.equals("float")) {
            return Float.class;
        }
        if (className.equals("double")) {
            return Double.class;
        }
        if (className.equals("char")) {
            return Character.class;
        }
        if (className.equals("void")) {
            return Void.class;
        }
        return null;
    }

    public static Class<?> getPrimitiveClass(String className) throws ClassNotFoundException {
        if (className.equals("boolean")) {
            return boolean.class;
        }
        if (className.equals("byte")) {
            return byte.class;
        }
        if (className.equals("short")) {
            return short.class;
        }
        if (className.equals("int")) {
            return int.class;
        }
        if (className.equals("long")) {
            return long.class;
        }
        if (className.equals("float")) {
            return float.class;
        }
        if (className.equals("double")) {
            return double.class;
        }
        if (className.equals("char")) {
            return char.class;
        }
        if (className.equals("void")) {
            return void.class;
        }
        // detect arrays
        int arrayCount = 0;
        while (className.endsWith("[]")) {
            arrayCount++;
            className = className.substring(0, className.length() - 2);
        }
        Class<?> klass = null;
        if (arrayCount > 0) {
            klass = getPrimitiveClass(className);
            if (klass == null) {
                // none primitive array
                return null;
            }
            // must take the boxed class, other it does not works
            // to make a Class.forName("[Lchar;"); but works
            // with Class.forName("[LCharacter;"); ...
            klass = getPrimitiveBoxedClass(className);
            className = klass.getName();

            className = "L" + className + ";";
            while (arrayCount > 0) {
                className = "[" + className;
                arrayCount--;
            }
            //System.out.println("primitive array class "+className);
            return Class.forName(className);
        }
        return null;
    }

    public static Class<?> getClass(String className, ClassLoader classLoader) throws ClassNotFoundException {
        Class<?> klass = getPrimitiveClass(className);
        if (klass != null) {
            return klass;
        }
        // try an array of none primitive classes
        int arrayCount = 0;
        while (className.endsWith("[]")) {
            arrayCount++;
            className = className.substring(0, className.length() - 2);
        }
        if (arrayCount > 0) {
            className = "L" + className + ";";
            while (arrayCount > 0) {
                className = "[" + className;
                arrayCount--;
            }
        }
        try {
            return classLoader != null ? Class.forName(className, true, classLoader) : Class.forName(className);
        } catch (ClassNotFoundException e) {
            // perharps we are in a inner class ?
            int dotIndex = className.lastIndexOf(".");
            if (dotIndex > -1) {
                String parentFQN = className.substring(0, dotIndex);
                String simpleName = className.substring(dotIndex + 1);
                try {
                    Class<?> parentClass = classLoader != null ? Class.forName(parentFQN, true, classLoader) : Class.forName(parentFQN);
                    for (Class<?> innerClass : parentClass.getClasses()) {
                        if (simpleName.equals(innerClass.getSimpleName())) {
                            return innerClass;
                        }
                    }
                } catch (ClassNotFoundException e1) {
                    // no super class,so let the first exception throw...
                }
            }
            throw e;
        } catch (NoClassDefFoundError e) {

            throw new ClassNotFoundException(e.toString());
        }
    }

    private static MethodDescriptor createMethodDescriptor(Method javaMethod, ClassLoader classLoader) {
        String methodName = javaMethod.getName();
        int modifiers = javaMethod.getModifiers();
        String returnType = javaMethod.getReturnType().getName();
        Class<?>[] javaParameters = javaMethod.getParameterTypes();
        String[] parameters = new String[javaParameters.length];
        for (int i = 0; i < parameters.length; i++) {
            parameters[i] = javaParameters[i].getName();
        }
        return new MethodDescriptor(methodName, modifiers, returnType, parameters, classLoader);
    }

    private static FieldDescriptor createFieldDescriptor(Field javaField, ClassLoader classLoader) {
        String fieldName = javaField.getName();
        int modifiers = javaField.getModifiers();
        String type = javaField.getType().getName();
        return new FieldDescriptor(fieldName, modifiers, type, classLoader);
    }

    private static JAXXObjectDescriptor getJAXXObjectDescriptor(Class<?> jaxxClass) {
        if (!JAXXObject.class.isAssignableFrom(jaxxClass) || jaxxClass == JAXXObject.class) {
            return null;
        }
        try {
            Method getJAXXObjectDescriptor = jaxxClass.getMethod("$getJAXXObjectDescriptor", new Class<?>[0]);
            return (JAXXObjectDescriptor) getJAXXObjectDescriptor.invoke(null);
        } catch (NoSuchMethodException e) {
            throw new CompilerException("Expected JAXXObject " + jaxxClass.getName() + " to have a static method named $getJAXXObjectDescriptor");
        } catch (IllegalAccessException e) {
            throw new CompilerException("Expected JAXXObject " + jaxxClass.getName() + "'s $getJAXXObjectDescriptor method to be public");
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e);
        }
    }

    private static ClassDescriptor createClassDescriptorFromJavaSource(URL javaSource, ClassLoader classLoader) throws ClassNotFoundException {
        try {
            InputStream in = javaSource.openStream();
            Reader reader = new InputStreamReader(in, "utf-8");
            ClassDescriptor result = JavaFileParser.parseJavaFile(javaSource.toString(), reader, classLoader);
            reader.close();
            return result;
        } catch (IOException e) {
            throw new ClassNotFoundException(e.toString());
        }
    }

    private static ClassDescriptor createClassDescriptorFromSymbolTable(String className, ClassLoader classLoader) throws ClassNotFoundException {
        final JAXXCompiler compiler = JAXXEngine.get().getJAXXCompiler(className);
        final SymbolTable symbolTable = JAXXEngine.get().getSymbolTable(className);
        if (symbolTable == null) {
            throw new CompilerException("Internal error: no symbol table was generated for class '" + className + "'");
        }
        ClassDescriptor superclass = getClassDescriptor(symbolTable.getSuperclassName(), classLoader);
        List<MethodDescriptor> publicMethods = symbolTable.getScriptMethods();
        List<FieldDescriptor> publicFields = symbolTable.getScriptFields();
        //List<MethodDescriptor> declaredMethods = new ArrayList<MethodDescriptor>(publicMethods);
        //List<FieldDescriptor> declaredFields = new ArrayList<FieldDescriptor>(publicFields);
        Iterator<MethodDescriptor> methods = publicMethods.iterator();
        while (methods.hasNext()) {
            MethodDescriptor method = methods.next();
            if (!Modifier.isPublic(method.getModifiers())) {
                methods.remove();
            }
        }
        Iterator<FieldDescriptor> fields = publicFields.iterator();
        while (fields.hasNext()) {
            FieldDescriptor field = fields.next();
            if (!Modifier.isPublic(field.getModifiers())) {
                fields.remove();
            }
        }
        publicMethods.addAll(Arrays.asList(superclass.getMethodDescriptors()));
        publicFields.addAll(Arrays.asList(superclass.getFieldDescriptors()));
        int dotPos = className.lastIndexOf(".");
        String packageName = dotPos != -1 ? className.substring(0, dotPos) : null;
        Set<String> interfaces = new HashSet<String>();
        ClassDescriptor[] superclassInterfaces = superclass.getInterfaces();
        for (ClassDescriptor superclassInterface : superclassInterfaces) {
            interfaces.add(superclassInterface.getName());
        }
        interfaces.add(JAXXObject.class.getName());
        return new ClassDescriptor(className, packageName, symbolTable.getSuperclassName(),
                interfaces.toArray(new String[interfaces.size()]), false, false, null, null, classLoader,
                publicMethods.toArray(new MethodDescriptor[publicMethods.size()]),
                publicFields.toArray(new FieldDescriptor[publicFields.size()])) {

            @Override
            public FieldDescriptor getDeclaredFieldDescriptor(String name) throws NoSuchFieldException {
                String type = symbolTable.getClassTagIds().get(name);
                if (type != null) {
                    return new FieldDescriptor(name, Modifier.PROTECTED, type, compiler.getClassLoader());
                }
                throw new NoSuchFieldException(name);
            }

            @Override
            public MethodDescriptor getDeclaredMethodDescriptor(String name, ClassDescriptor... parameterTypes) throws NoSuchMethodException {
                throw new NoSuchMethodException(name);
            }

            @Override
            public JAXXObjectDescriptor getJAXXObjectDescriptor() {
                return compiler.getJAXXObjectDescriptor();
            }
        };
    }

    private static ClassDescriptor createClassDescriptorFromClass(final Class<?> javaClass) {
        String name = javaClass.getName();
        Package p = javaClass.getPackage();
        String packageName = p != null ? p.getName() : null;
        Class<?> superclass = javaClass.getSuperclass();
        String superclassName = superclass != null ? superclass.getName() : null;
        Class<?>[] interfaces = javaClass.getInterfaces();
        String[] interfaceNames = new String[interfaces.length];
        for (int i = 0; i < interfaces.length; i++) {
            interfaceNames[i] = interfaces[i].getName();
        }
        boolean isInterface = javaClass.isInterface();
        boolean isArray = javaClass.isArray();
        String componentTypeName = isArray ? javaClass.getComponentType().getName() : null;
        JAXXObjectDescriptor jaxxObjectDescriptor = getJAXXObjectDescriptor(javaClass);
        ClassLoader classLoader = javaClass.getClassLoader();
        Method[] javaMethods = javaClass.getMethods();
        MethodDescriptor[] methods = new MethodDescriptor[javaMethods.length];
        for (int i = 0; i < methods.length; i++) {
            methods[i] = createMethodDescriptor(javaMethods[i], javaClass.getClassLoader());
        }
        Field[] javaFields = javaClass.getFields();
        FieldDescriptor[] fields = new FieldDescriptor[javaFields.length];
        for (int i = 0; i < fields.length; i++) {
            fields[i] = createFieldDescriptor(javaFields[i], javaClass.getClassLoader());
        }
        return new ClassDescriptor(name, packageName, superclassName, interfaceNames, isInterface, isArray, componentTypeName, jaxxObjectDescriptor, classLoader, methods, fields) {

            @Override
            public FieldDescriptor getDeclaredFieldDescriptor(String name) throws NoSuchFieldException {
                return createFieldDescriptor(javaClass.getDeclaredField(name), javaClass.getClassLoader());
            }

            @Override
            public MethodDescriptor getDeclaredMethodDescriptor(String name, ClassDescriptor... parameterTypes) throws NoSuchMethodException {
                try {
                    Class[] parameterTypeClasses = new Class[parameterTypes.length];
                    for (int i = 0; i < parameterTypes.length; i++) {
                        parameterTypeClasses[i] = Class.forName(parameterTypes[i].getName());
                    }
                    return createMethodDescriptor(javaClass.getDeclaredMethod(name, parameterTypeClasses), javaClass.getClassLoader());
                } catch (ClassNotFoundException e) {
                    throw new RuntimeException(e);
                }
            }
        };
    }

    public static void checkSupportClass(Class<?> handlerClass, ClassDescriptor beanClass, Class<?>... tagClasses) {
        for (Class<?> tagClass : tagClasses) {
            if (getClassDescriptor(tagClass).isAssignableFrom(beanClass)) {
                return;
            }
        }
        throw new IllegalArgumentException(handlerClass.getName() + " does not support the class " + beanClass.getName());
    }

    public static void reset() {
        descriptors.clear();
    }
}
