/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.java;

import java.io.PrintWriter;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.Map.Entry;
import jaxx.compiler.java.JavaMethod.MethodOrder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Java file generator.
 *
 * @author chemit
 * @since 2.0.0
 */
public class JavaFileGenerator {

    /**
     * Logger
     */
    static private final Log log = LogFactory.getLog(JavaFileGenerator.class);

    public static JavaField newField(int modifiers, String returnType, String name, boolean override) {
        return newField(modifiers, returnType, name, override, null);
    }

    public static JavaField newField(int modifiers, String returnType, String name, boolean override, String initializer) {
        return new JavaField(modifiers, returnType, name, override, initializer);
    }

    public static JavaMethod newMethod(int modifiers, String returnType, String name, String initializer, boolean override, String[] exceptions, JavaArgument... arguments) {
        return new JavaMethod(modifiers, returnType, name, arguments, exceptions, initializer, override);
    }

    public static JavaMethod newMethod(int modifiers, String returnType, String name, String initializer, boolean override, JavaArgument... arguments) {
        return newMethod(modifiers, returnType, name, initializer, override, new String[0], arguments);
    }
    /**
     * End of line
     */
    protected final String eol;
    /**
     * verbose flag when generates
     */
    protected final boolean verbose;
    /**
     * current prefix indent size
     */
    protected int indentationLevel = 0;

    public JavaFileGenerator(String eol, boolean verbose) {
        this.eol = eol;
        this.verbose = verbose && log.isDebugEnabled();
    }

    public String generateImport(String anImport) {
        return "import " + anImport + ';' + eol;
    }

    public void generateFile(JavaFile f, PrintWriter result) {
        if (verbose) {
            log.info(f.getName());
        }
        indentationLevel = 0;
        if (f.getName().indexOf(".") != -1) {
            result.append("package ").append(f.getName().substring(0, f.getName().lastIndexOf("."))).append(";");
            result.append(eol).append(eol);
        }
        String[] imports = f.getImports();

        if (imports.length > 0) {
            for (String anImport : imports) {
                result.append(generateImport(anImport));
            }
            result.append(eol);
        }
        result.append(generateClass(f));
    }

    public String generateClass(JavaFile f) {

        if (verbose) {
            log.info(f.getName());
        }
        StringBuffer result = new StringBuffer();
        String genericType = f.getGenericType();

        result.append(f.getModifiersText());
        if (f.isAbstractClass()) {
            result.append("abstract ");
        }
        result.append("class ");
        result.append(f.getName().substring(f.getName().lastIndexOf(".") + 1));
        if (genericType != null) {
            result.append('<').append(genericType).append('>');
        }
        result.append(" extends ");
        result.append(f.getSuperClass());
        if (f.getSuperGenericType() != null) {
            result.append('<').append(f.getSuperGenericType()).append('>');
        }
        List<String> interfaces = f.getInterfaces();

        if (interfaces != null && !interfaces.isEmpty()) {
            result.append(" implements ").append(interfaces.get(0));
            for (int i = 1; i < interfaces.size(); i++) {
                result.append(", ").append(interfaces.get(i));
            }
        }
        result.append(" {").append(eol);

        // generate fields

        List<JavaField> fields = f.getFields();

        if (!fields.isEmpty()) {
            java.util.Collections.sort(fields); // sort fields

            for (JavaField field : fields) {
                if (log.isDebugEnabled()) {
                    log.debug("generate field " + field);
                }
                String txt = generateField(field);
                result.append(addIndentation(txt, 4, eol)).append(eol);
            }
            result.append(eol);
        }

        // generate raw body

        StringBuffer rawBodyCode = f.getRawBodyCode();

        if (rawBodyCode.length() > 0) {
            result.append(addIndentation("/* begin raw body code */\n", 4, eol));
            String s = rawBodyCode.toString();
            if (!s.startsWith(eol)) {
                result.append(eol);
            }
            result.append(addIndentation(s, 4, eol)).append(eol);
            result.append(addIndentation("/* end raw body code */", 4, eol));
            result.append(eol);
        }

        // generate inner classes

        List<JavaFile> innerClasses = f.getInnerClasses();
        for (JavaFile innerClass : innerClasses) {
            indentationLevel += 4;
            try {
                String txt = generateClass(innerClass);
                result.append(addIndentation(txt, 4, eol));
                result.append(eol).append(eol);
            } finally {
                indentationLevel -= 4;
            }
        }

        // generate methods

        EnumMap<MethodOrder, List<JavaMethod>> map = JavaMethod.getSortedMethods(f.getMethods());
        for (Entry<MethodOrder, List<JavaMethod>> entry : map.entrySet()) {
            List<JavaMethod> list = entry.getValue();
            if (!list.isEmpty()) {
                result.append(addIndentation(entry.getKey().getHeader(), 4, eol));
                result.append(eol).append(eol);
                for (JavaMethod method : list) {
                    String txt = generateMethod(method);
                    result.append(addIndentation(txt, 4, eol));
                    result.append(eol).append(eol);
                }
            }
            list.clear();
        }
        map.clear();
        result.append("}");
        return result.toString();

    }

    public String generateField(JavaField f) {
        if (verbose) {
            log.info(f.getName());
        }
        StringBuffer result = new StringBuffer();
        result.append(f.getModifiersText());
        result.append(f.getType()).append(' ').append(f.getName());
        if (f.getInitializer() != null) {
            result.append(" = ").append(f.getInitializer());
        }
        result.append(';').append(eol);
        return result.toString();
    }

    public String generateMethod(JavaMethod m) {
        if (verbose) {
            log.info(m.getName());
        }

        StringBuffer result = new StringBuffer();
        if (m.isOverride()) {
            result.append("@Override").append(eol);
        }
        result.append(m.getModifiersText());
        if (m.getReturnType() != null) {
            result.append(m.getReturnType());
            result.append(' ');
        }
        result.append(m.getName());
        result.append('(');
        JavaArgument[] arguments = m.getArguments();

        if (arguments != null && arguments.length > 0) {
            result.append(generateArgument(arguments[0]));
            for (int i = 1; i < arguments.length; i++) {
                result.append(", ").append(generateArgument(arguments[i]));
            }
        }
        result.append(") {");
        result.append(eol);
        String body = m.getBody();

        if (body != null) {
            String formattedBodyCode = addIndentation(body.trim(), 4, eol);
            if (formattedBodyCode.length() > 0) {
                result.append(formattedBodyCode).append(eol);
            }
        }
        result.append("}");
        return result.toString();

    }

    public String generateArgument(JavaArgument argument) {
        String result = argument.getType() + ' ' + argument.getName();
        return argument.isFinal() ? "final " + result : result;
    }

    public String addIndentation(String source, int indentation, String lineSeparator) {
        return indent(source, indentationLevel + indentation, false, lineSeparator);
    }

    public String setIndentation(String source, int indentation, String lineSeparator) {
        return indent(source, indentationLevel + indentation, true, lineSeparator);
    }

    public static String indent(String source, int indentation, boolean trim, String lineSeparator) {
        if (trim) {
            source = source.trim();
        }
        char[] spaces = new char[indentation];
        Arrays.fill(spaces, ' ');
        StringBuffer result = new StringBuffer();
        String[] lines = source.split(lineSeparator + "|\n");
        for (int i = 0; i < lines.length; i++) {
            if (i > 0) {
                result.append(lineSeparator);
            }
            result.append(spaces);
            result.append(trim ? lines[i].trim() : lines[i]);
        }
        return result.toString();
    }
}
