/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.java;

import java.lang.reflect.Modifier;
import java.util.Comparator;

/**
 * Represents a field in a Java source file being generated for output.  <code>JavaFields</code> are created
 * and added to a {@link JavaFile}, which can then output Java source code.
 */
public class JavaField extends JavaElement implements Comparable<JavaField> {

    /**
     * Field comparator
     */
    static final JavaFieldComparator COMPARATOR = new JavaFieldComparator();
    /**
     * type of field (fqn)
     */
    private String type;
    /**
     * initializer of field (can be null)
     */
    private String initializer;
    /**
     * flag to known where a field overrides a super-field
     */
    private boolean override;

    /**
     * Constructs a new <code>JavaField</code>.  The <code>modifiers</code> parameter is a bit mask of the
     * constants from {@link java.lang.reflect.Modifier}, and the <code>type</code> of the field should be
     * represented as it would appear in Java source code.
     *
     * @param modifiers the modifier keywords that should appear as part of the field's declaration
     * @param type      the type of the field as it would appear in Java source code
     * @param name      the field's name
     * @param override 
     */
    public JavaField(int modifiers, String type, String name, boolean override) {
        this(modifiers, type, name, override, null);
    }

    /**
     * Constructs a new <code>JavaField</code>.  The <code>modifiers</code> parameter is a bit mask of the
     * constants from <code>java.lang.reflect.Modifier</code>, and the <code>type</code> of the field should be
     * represented as it would appear in Java source code.  The <code>initializer</code> is the initial
     * value of the field as it would appear in Java source code, or <code>null</code> to leave it at the
     * default value.
     *
     * @param modifiers   the modifier keywords that should appear as part of the field's declaration
     * @param type        the type of the field as it would appear in Java source code
     * @param name        the field's name
     * @param override 
     * @param initializer the initial value of the field, as it would appear in Java source code
     */
    public JavaField(int modifiers, String type, String name, boolean override, String initializer) {
        super(modifiers, name);
        this.type = type;
        this.initializer = initializer;
        this.override = override;
    }

    /**
     * Returns the field's type, as it would be represented in Java source code.
     *
     * @return the field's type
     */
    public String getType() {
        return type;
    }

    public boolean isOverride() {
        return override;
    }

    public void setOverride(boolean override) {
        this.override = override;
    }

    public String getInitializer() {
        return initializer;
    }

    @Override
    public int compareTo(JavaField o) {
        return COMPARATOR.compare(this, o);
    }

    static class JavaFieldComparator implements Comparator<JavaField> {

        @Override
        public int compare(JavaField o1, JavaField o2) {
            int result;
            if ((result = compareStatic(o1, o2)) != 0) {
                return result;
            }
            // data sources must be on the last after all other fields
            if ((result = compareDataSource(o1, o2)) != 0) {
                return result;
            }
            // same static
            if ((result = compareVisibility(o1, o2)) != 0) {
                return result;
            }
            // same visibility, test name
            return o1.getName().compareTo(o2.getName());
        }

        public int compareStatic(JavaField o1, JavaField o2) {
            // first comparator modifiers : static always before none static
            if (Modifier.isStatic(o1.getModifiers()) && !Modifier.isStatic(o2.getModifiers())) {
                return -1;
            }
            if (!Modifier.isStatic(o1.getModifiers()) && Modifier.isStatic(o2.getModifiers())) {
                return 1;
            }
            return 0;
        }

        public int compareDataSource(JavaField o1, JavaField o2) {
            // first comparator modifiers : static always before none static
            if (o1.getName().startsWith("$DataSource") && !o2.getName().startsWith("$DataSource")) {
                return 1;
            }
            if (!o1.getName().startsWith("$DataSource") && o2.getName().startsWith("$DataSource")) {
                return -1;
            }
            return 0;
        }

        public int compareVisibility(JavaField o1, JavaField o2) {
            // first comparator modifiers : static always before none static
            if (!Modifier.isPublic(o1.getModifiers()) && Modifier.isPublic(o2.getModifiers())) {
                return 1;
            }
            if (Modifier.isPublic(o1.getModifiers()) && !Modifier.isPublic(o2.getModifiers())) {
                return -1;
            }
            if (Modifier.isProtected(o1.getModifiers()) && !Modifier.isProtected(o2.getModifiers())) {
                return -1;
            }
            if (!Modifier.isProtected(o1.getModifiers()) && Modifier.isProtected(o2.getModifiers())) {
                return 1;
            }
            if (Modifier.isPrivate(o1.getModifiers()) && !Modifier.isPrivate(o2.getModifiers())) {
                return -1;
            }
            if (!Modifier.isPrivate(o1.getModifiers()) && Modifier.isPrivate(o2.getModifiers())) {
                return 1;
            }
            return 0;
        }
    }
}
