/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.binding;

import jaxx.compiler.CompilerException;
import jaxx.compiler.JAXXCompiler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Represents a data binding in a JAXX file.  <code>DataBinding</code> uses 
 * {@link DataSource} to track changes to a source expression and update
 * the destination.
 */
public class DataBinding {

    /**
     * Logger
     */
    protected static final Log log = LogFactory.getLog(DataBinding.class);
    /**
     * Id of the data binding
     */
    private String id;
    /**
     * source of the data binding
     */
    private String source;
    /**
     * A Java snippet which will cause the destination property to be updated with the current value of
     * the binding.
     */
    private String assignment;

    /**
     * Creates a new data binding.
     *
     * @param source the Java source code for the data binding expression
     * @param id the data binding destination in the form <code>id.propertyName</code>
     * @param assignment Java snippet which will cause the destination property to be updated with the current value of the binding
     */
    public DataBinding(String source, String id, String assignment) {
        this.id = id;
        this.source = source;
        this.assignment = assignment;
        if (log.isDebugEnabled()) {
            log.debug("id=" + id + " assignement=" + assignment + " source=" + source);
        }
    }

    public String getId() {
        return id;
    }

    /**
     * Compiles the data binding expression.  This method calls methods in 
     * <code>JAXXCompiler</code> to add the Java code that performs the data
     * binding setup.
     *
     * @param compiler compiler which includes the data binding
     * @param quickNoDependencies true to optimize bindings with no dependencies by simply running them at startup time
     * @return {@code true} if the expression has dependencies, {@code false} otherwise
     * @throws CompilerException if a compilation error occurs
     */
    public boolean compile(JAXXCompiler compiler, boolean quickNoDependencies) throws CompilerException {

        DataSource dataSource = new DataSource(id, source, compiler);

        // handles all of the listener additions
        //TC-20091026 use 'this' instead of root object javaCode
//        boolean result = dataSource.compile("new jaxx.runtime.DataBindingListener(" + compiler.getRootObject().getJavaCode() + ", " + compiler.getJavaCode(id) + ")");
        boolean result = dataSource.compile("new DataBindingListener(this, " + compiler.getJavaCode(id) + ")");
        if (log.isDebugEnabled()) {
            log.debug(id + " isBinding ? " + result);
        }
        String eol = JAXXCompiler.getLineSeparator();
        if (!result && quickNoDependencies) {
            // layout is specially handled early in the chain
            if (!id.endsWith(".layout")) {
                compiler.appendInitDataBindings(assignment + eol);
            }
            return false; // no dependencies, just a static expression
        }

        StringBuilder buffer = new StringBuilder();
        if (compiler.hasProcessDataBinding()) {
            buffer.append(" else ");
        } else {
            buffer.append("    ");
        }
        buffer.append("if (").append(compiler.getJavaCode(id)).append(".equals($dest)) {").append(eol);
        String objectCode = dataSource.getObjectCode();
        if (log.isDebugEnabled()) {
            log.debug(id + " objectCode= " + objectCode);
        }
        //TC-20091026 no need to test objectCode not null if on root object
//        boolean needTest = objectCode != null;
        boolean needTest = objectCode != null && !compiler.getRootObject().getId().equals(objectCode);
        if (needTest) {
            buffer.append("        if (").append(objectCode).append(" != null) {").append(eol);
            buffer.append("    ");
        }
        buffer.append("        ").append(assignment.trim());
        if (needTest) {
            buffer.append(eol).append("        }");
        }
        buffer.append(eol).append("    }");
        compiler.appendProcessDataBinding(buffer.toString());
        return true;
    }
}
