package org.nuiton.jaxx.application.swing.action;

/*
 * #%L
 * JAXX :: Application Swing
 * %%
 * Copyright (C) 2008 - 2014 Code Lutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Worker to execute logic action.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 2.8
 */
public class ApplicationActionSwingWorker<A extends AbstractApplicationAction> extends SwingWorker<Void, String> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(ApplicationActionSwingWorker.class);

    /**
     * Timer used to launch timerTask (to open load dialog).
     *
     * @since 2.8.2
     */
    private static final Timer t = new Timer();

    protected final ApplicationActionUI actionUI;

    protected final A action;

    protected Throwable error;

    protected TimerTask timer;

    protected ApplicationActionSwingWorker(A action) {

        this.action = action;
        this.actionUI = action.getContext().getActionUI();
    }

    public Throwable getError() {
        return error;
    }

    public boolean isFailed() {
        return error != null;
    }

    @Override
    protected Void doInBackground() throws Exception {

        if (!isCancelled()) {

            timer = new TuttiActionTimerTask();

            t.schedule(timer, 1000);

            try {

                action.doAction();

            } catch (Throwable e) {
                if (log.isErrorEnabled()) {
                    log.error("Task [" + this + "] Error while doAction: ", e);
                }
                error = e;
            } finally {
                if (log.isDebugEnabled()) {
                    log.debug("Task [" + this + "] done");
                }
            }
        }
        return null;
    }

    @Override
    protected void done() {
        super.done();

        if (log.isDebugEnabled()) {
            log.debug("Task [" + this + "] execute done method after all");
        }

        if (error == null) {

            // success hook
            action.postSuccessAction();
        } else {

            // fail hook
            action.postFailedAction(error);
        }

        action.releaseAction();

        if (timer != null) {

            timer.cancel();
        }

        actionUI.close();

        updateBusyState(false);
    }

    protected void updateBusyState(boolean busy) {

        boolean hideBody = action.isHideBody();

        action.getContext().setBusy(busy);

        if (hideBody) {
            action.getContext().setHideBody(!busy);
        }

    }

    protected class TuttiActionTimerTask extends TimerTask {

        public TuttiActionTimerTask() {
            action.addPropertyChangeListener(AbstractApplicationAction.PROPERTY_DONE, new PropertyChangeListener() {

                @Override
                public void propertyChange(PropertyChangeEvent evt) {

                    // prevent memory leak, TuttiActionTimerTask is created each time see https://forge.nuiton.org/issues/3990
                    action.removePropertyChangeListener(AbstractApplicationAction.PROPERTY_DONE, this);

                    // we do NOT want the timer to wake up then the actionUI dialog
                    cancel();

                    //if (actionUI.isVisible()) {
//}
                    SwingUtilities.invokeLater(actionUI::close);
                }
            });
        }

        @Override
        public void run() {
            if (isCancelled() || isDone()) {

                if (log.isDebugEnabled()) {
                    log.debug("Task [" + action + "] was already canceled or done, do nothing");
                }
            } else {

                if (log.isDebugEnabled()) {
                    log.debug("Task [" + action + "] is started, show waiting dialog");
                }

                SwingUtilities.invokeLater(() -> {

                    synchronized (actionUI) {

                        try {
                            actionUI.open(action);
                        } catch (Exception e) {

                            // swallow any exception due to ui bad initialisation
                            // this should not happen, but it does :(
                            // See https://forge.codelutin.com/issues/7489
                            // See http://stackoverflow.com/questions/15545055/nimbus-java-lang-classcastexception-javax-swing-plaf-nimbus-derivedcoloruires
                            // See http://bugs.java.com/bugdatabase/view_bug.do?bug_id=6785663
                            if (log.isErrorEnabled()) {
                                log.error("Something was wrong while opening the action ui, but we do not care", e);
                            }
                        }
                    }
                });
            }
        }
    }
}
