package org.nuiton.eugene.java;

/*
 * #%L
 * EUGene :: Java templates
 * %%
 * Copyright (C) 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableSet;
import org.nuiton.eugene.models.extension.tagvalue.TagValueMetadata;
import org.nuiton.eugene.models.extension.tagvalue.TagValueUtil;
import org.nuiton.eugene.models.extension.tagvalue.matcher.EqualsTagValueNameMatcher;
import org.nuiton.eugene.models.extension.tagvalue.provider.DefaultTagValueMetadatasProvider;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelPackage;

import java.util.Set;

import static org.nuiton.i18n.I18n.n;
import static org.nuiton.i18n.I18n.t;

/**
 * Defines all tag values managed by Java templates.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @plexus.component role="org.nuiton.eugene.models.extension.tagvalue.provider.TagValueMetadatasProvider" role-hint="eugene-java-templates"
 * @since 2.5.6
 */
public class EugeneJavaTagValues extends DefaultTagValueMetadatasProvider {

    @Override
    public String getDescription() {
        return t("eugene.java.tagvalues");
    }

    public enum Store implements TagValueMetadata {

        /**
         * Boolean tag value for JavaBean objects to place on a classifier or package.
         *
         * @see #isBean(ObjectModelClassifier, ObjectModelPackage)
         * @see #isBean(ObjectModelPackage)
         * @since 2.5.6
         */
        bean(n("eugene.java.stereotype.bean"), boolean.class, null, ObjectModelClassifier.class, ObjectModelPackage.class),

        /**
         * Tag value to authorize user to generate some bean with methods, some generators won't generate them : Lots code).
         *
         * By default, user should never add methods in bean classes, simply write them in your java code!.
         *
         * You can globaly use it on the complete model, on packages, or to a specific classifier.
         *
         * @see #isAcceptBeanWithMethods(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
         * @since 3.0
         */
        acceptBeanWithMethods(n("eugene.java.tagvalue.acceptBeanWithMethods"), boolean.class, null, ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class),


        /**
         * Tag value to authorize user to override abstract classes.
         *
         * By default, user should never override abstract classes but works on implementation ones.
         *
         * You can globaly use it on the complete model, on packages, or to a specific classifier.
         *
         * @see #isOverrideAbstractClasses(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
         * @since 3.0
         */
        overrideAbstractClasses(n("eugene.java.tagvalue.overrideAbstractClasses"), boolean.class, null, ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class),

        /**
         * Tag value to generate property change support on generated beans.
         *
         * You can globaly use it on the complete model, on packages, or to a specific classifier.
         *
         * @see #isSkipGeneratePropertyChangeSupport(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
         * @since 3.0
         */
        generatePropertyChangeSupport(n("eugene.java.tagvalue.generatePropertyChangeSupport"), boolean.class, null, ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class),

        /**
         * Tag value to generate lazy instanciation of any collection to avoid NPEs.
         *
         * You can globaly use it on the complete model or a package, or to a specific classifier.
         *
         * @see #isSkipGenerateNotEmptyCollections(ObjectModelClassifier, ObjectModelPackage, ObjectModel)
         * @since 3.0
         */
        generateNotEmptyCollections(n("eugene.java.tagvalue.generateNotEmptyCollections"), boolean.class, null, ObjectModel.class, ObjectModelPackage.class, ObjectModelClassifier.class),

        /**
         * To use java 8 new syntax and api in generation.
         *
         * You can globaly use it on the complete model.
         *
         * @see #isUseJava8(ObjectModel)
         * @since 2.15
         */
        java8(n("eugene.java.tagvalue.java8"), boolean.class, "false", ObjectModel.class);

        private final Set<Class<?>> targets;
        private final Class<?> type;
        private final String i18nDescriptionKey;
        private final String defaultValue;

        Store(String i18nDescriptionKey, Class<?> type, String defaultValue, Class<?>... targets) {
            this.targets = ImmutableSet.copyOf(targets);
            this.type = type;
            this.i18nDescriptionKey = i18nDescriptionKey;
            this.defaultValue = defaultValue;
        }

        @Override
        public String getName() {
            return name();
        }

        @Override
        public Set<Class<?>> getTargets() {
            return targets;
        }

        @Override
        public Class<?> getType() {
            return type;
        }

        @Override
        public Class<EqualsTagValueNameMatcher> getMatcherClass() {
            return EqualsTagValueNameMatcher.class;
        }

        @Override
        public String getDescription() {
            return t(i18nDescriptionKey);
        }

        @Override
        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public boolean isDeprecated() {
            return false;
        }

    }

    public EugeneJavaTagValues() {
        super((TagValueMetadata[]) Store.values());
    }

    /**
     * Obtain the value of the {@link Store#acceptBeanWithMethods} tag value on the given model, package or classifier.
     *
     * It will first look on the model, then and package and then in the given classifier.
     *
     * If no value found, then will use the default value of the tag value.
     *
     * @param classifier classifier to seek
     * @param aPackage   package to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see Store#acceptBeanWithMethods
     * @since 3.0
     */
    public boolean isAcceptBeanWithMethods(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        return TagValueUtil.findBooleanTagValue(Store.acceptBeanWithMethods, classifier, aPackage, model);
    }

    /**
     * Obtain the value of the {@link Store#overrideAbstractClasses} tag value on the given model, package or classifier.
     *
     * It will first look on the model, then and package and then in the given classifier.
     *
     * If no value found, then will use the default value of the tag value.
     *
     * @param classifier classifier to seek
     * @param aPackage   package to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see Store#overrideAbstractClasses
     * @since 3.0
     */
    public boolean isOverrideAbstractClasses(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        return TagValueUtil.findBooleanTagValue(Store.overrideAbstractClasses, classifier, aPackage, model);
    }

    /**
     * Obtain the value of the {@link Store#generatePropertyChangeSupport} tag value on the given model, package or classifier.
     *
     * It will first look on the model, then and package and then in the given classifier.
     *
     * If no value found, then will use the default value of the tag value.
     *
     * @param classifier classifier to seek
     * @param aPackage   package to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see Store#generatePropertyChangeSupport
     * @since 2.12
     */
    public boolean isSkipGeneratePropertyChangeSupport(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        return TagValueUtil.findBooleanTagValue(Store.generatePropertyChangeSupport, classifier, aPackage, model);
    }

    /**
     * Obtain the value of the {@link Store#generateNotEmptyCollections} tag value on the given model, package or classifier.
     *
     * It will first look on the model, then and package and then in the given classifier.
     *
     * If no value found, then will use the default value of the tag value.
     *
     * @param classifier classifier to seek
     * @param aPackage   package to seek
     * @param model      model to seek
     * @return the none empty value of the found tag value or {@code null} if not found nor empty.
     * @see Store#generateNotEmptyCollections
     * @since 2.12
     */
    public boolean isSkipGenerateNotEmptyCollections(ObjectModelClassifier classifier, ObjectModelPackage aPackage, ObjectModel model) {
        return TagValueUtil.findBooleanTagValue(Store.generateNotEmptyCollections, classifier, aPackage, model);
    }

    /**
     * Obtain the value of the {@link Store#java8} tag value on the given model.
     *
     * @param model model to seek
     * @return {@code true} the none empty value of the found tag value or {@code false} if not found nor empty.
     * @see Store#java8
     * @since 2.15
     */
    public boolean isUseJava8(ObjectModel model) {
        return TagValueUtil.findBooleanTagValue(Store.java8, model);
    }

    /**
     * Check if the given classifier has the {@link Store#bean} stereotype.
     *
     * @param classifier classifier to test
     * @return {@code true} if stereotype was found, {@code false otherwise}
     * @see Store#bean
     */
    public boolean isBean(ObjectModelClassifier classifier, ObjectModelPackage aPackage) {
        return TagValueUtil.findBooleanTagValue(Store.bean, classifier, aPackage);
    }

    /**
     * Check if the given aPackage has the {@link Store#bean} stereotype.
     *
     * @param aPackage classifier to test
     * @return {@code true} if stereotype was found, {@code false otherwise}
     * @see Store#bean
     */
    public boolean isBean(ObjectModelPackage aPackage) {
        return TagValueUtil.findBooleanTagValue(Store.bean, aPackage);
    }
}
