package org.nuiton.eugene.java;

/*
 * #%L
 * EUGene :: Java templates
 * $Id: SimpleJavaBeanTransformer.java 1401 2014-08-05 06:32:30Z tchemit $
 * $HeadURL: https://svn.nuiton.org/eugene/tags/eugene-2.13/eugene-java-templates/src/main/java/org/nuiton/eugene/java/SimpleJavaBeanTransformer.java $
 * %%
 * Copyright (C) 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */




import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorException;
import org.nuiton.eugene.models.object.ObjectModel;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelPackage;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * SimpleJavaBeanTransformer generates simple bean with pcs support
 * (and nothing else) according to the JavaBeans 1.1 norm with no Impl generation mecanism.
 * <p/>
 * So if there is so operation described on model, you should use the
 * {@link JavaBeanTransformer} instead.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.eugene.java.SimpleJavaBeanTransformer"
 * @since 2.6
 */
public class SimpleJavaBeanTransformer extends AbstractJavaBeanTransformer {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SimpleJavaBeanTransformer.class);

    @Override
    public void transformFromModel(ObjectModel model) {

        String className = model.getName() + "BeanFactory";

        if (canGenerateFactory(model, className)) {

            generateBeanFactory(model, className);
        }
    }

    @Override
    public void transformFromClass(ObjectModelClass input) {

        ObjectModelPackage aPackage = getPackage(input);

        if (!JavaTemplatesStereoTypes.hasBeanStereotype(input, aPackage)) {

            //  not a bean
            return;
        }

        String interfaceName = getBeanInterfaceName(aPackage, input);
        String abstractClassName = getAbstractBeanClassName(aPackage, input);
        String className = getBeanClassName(aPackage, input);
        String defaultClassName = getBeanDefaultsClassName(aPackage, input);

        boolean generateDefaults = canGenerateBeanDefaults(aPackage, input);
        boolean generateInterface = canGenerateInterface(aPackage, input, interfaceName);
        boolean generateAbstractClass = canGenerateAbstractBean(input, abstractClassName);
        boolean generateClass = canGenerateBean(input, className);

        String prefix = getConstantPrefix(input);
        setConstantPrefix(prefix);

        ObjectModelClass outputAbstractClass = null;

        if (generateAbstractClass) {

            outputAbstractClass = generateAbstractBeanClass(aPackage,
                                                            input,
                                                            abstractClassName,
                                                            interfaceName);
        }

        if (generateInterface) {

            ObjectModelClass outputAbstractClass1 = outputAbstractClass;

            if (outputAbstractClass == null) {

                outputAbstractClass1 = input;

            }

            generateBeanInterface(aPackage, input, interfaceName, outputAbstractClass1);

            if (generateAbstractClass) {

                // add override annotation on all public operations
                for (ObjectModelOperation operation : getPublicOperations(outputAbstractClass)) {
                    addAnnotation(outputAbstractClass,
                                  operation,
                                  Override.class);
                }
            }

        }

        if (generateClass) {

            generateBeanClass(input, className, abstractClassName);

        }

        if (generateDefaults) {

            generateBeanDefaults(aPackage, input, defaultClassName);
        }
    }

    protected boolean canGenerateFactory(ObjectModel model, String className) {

        boolean generateFactory =
                getJavaTemplatesTagValues().isSimpleBeanGenerateFactory(model);

        String defaultPackage = getDefaultPackageName();

        String fqn = defaultPackage + "." + className;
        boolean canGenerate = generateFactory &&
                              !isInClassPath(fqn);

        return canGenerate;
    }

    protected boolean canGenerateInterface(ObjectModelPackage aPackage,
                                           ObjectModelClass input,
                                           String className) {

        boolean generateInterface =
                getJavaTemplatesTagValues().isSimpleBeanGenerateInterface(input, aPackage, model);

        String fqn = input.getPackageName() + "." + className;

        boolean canGenerate = generateInterface && !isInClassPath(fqn);

        return canGenerate;
    }

    protected boolean canGenerateAbstractBean(ObjectModelClass input, String className) {
        String fqn = input.getPackageName() + "." + className;
        boolean canGenerate = !isInClassPath(fqn);
        return canGenerate;
    }

    protected boolean canGenerateBean(ObjectModelClass input, String className) {
        String fqn = input.getPackageName() + "." + className;
        boolean canGenerate = !isInClassPath(fqn);

        if (canGenerate) {

            // check there is no operation on input
            if (!input.getOperations().isEmpty()) {

                canGenerate = false;
//                throw new GeneratorException(
//                        "Can't generate a simple bean as class " +
//                        fqn +
//                        " contains so operations." +
//                        "\nUse instead the JavaBeanTransformer.");
            }
        }
        return canGenerate;
    }

    protected boolean canGenerateBeanDefaults(ObjectModelPackage aPackage, ObjectModelClass input) {

        boolean withInput = input != null;

        boolean canGenerate =
                getJavaTemplatesTagValues().isSimpleBeanGenerateDefaults(
                        input, aPackage, model);

        if (canGenerate) {

            if (withInput) {

                // class in not abstract
                // class is a bean

                canGenerate = !input.isAbstract() &&
                              JavaTemplatesStereoTypes.hasBeanStereotype(input, aPackage);

            }
        }

        return canGenerate;
    }

    protected boolean canGenerateBeanAbstractDefaults(ObjectModelClass input,
                                                      String className) {

        String classPackage = input.getPackageName();

        String fqn = classPackage + "." + className;

        boolean canGenerate = !isInClassPath(fqn);

        return canGenerate;
    }

    protected void generateBeanFactory(ObjectModel model, String className) {
        String defaultPackage = getDefaultPackageName();

        ObjectModelClass output = createClass(className, defaultPackage);

        for (ObjectModelClass aClass : model.getClasses()) {

            String packageName = aClass.getPackageName();

            ObjectModelPackage aPackage = getPackage(packageName);

            if (!aClass.isAbstract() &&
                JavaTemplatesStereoTypes.hasBeanStereotype(aClass, aPackage)) {

                String typeName = getBeanInterfaceName(aPackage, aClass);
                String typeBeanName = getBeanClassName(aPackage, aClass);
                addImport(output, packageName + "." + typeName);
                addImport(output, packageName + "." + typeBeanName);
                ObjectModelOperation operation = addOperation(
                        output,
                        "typeOf" + typeName,
                        "<BeanType extends " + typeName + "> Class<BeanType>",
                        ObjectModelJavaModifier.STATIC,
                        ObjectModelJavaModifier.PUBLIC
                );
                setOperationBody(operation, ""
    +"\n"
+"        return (Class<BeanType>) "+typeBeanName+".class;\n"
+"    "
                );
                operation = addOperation(
                        output,
                        "new" + typeName,
                        typeName,
                        ObjectModelJavaModifier.STATIC,
                        ObjectModelJavaModifier.PUBLIC
                );
                setOperationBody(operation, ""
    +"\n"
+"        return new "+typeBeanName+"();\n"
+"    "
                );
            }
        }
    }

    protected ObjectModelInterface generateBeanInterface(ObjectModelPackage aPackage,
                                                         ObjectModelClass input,
                                                         String className,
                                                         ObjectModelClass outputClass) {

        ObjectModelInterface output =
                createInterface(className, input.getPackageName());

        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        boolean superClassIsBean = false;

        String superClass = null;

        // test if a super class has bean stereotype

        Collection<ObjectModelClass> superclasses = input.getSuperclasses();
        if (CollectionUtils.isNotEmpty(superclasses)) {
            for (ObjectModelClass superclass : superclasses) {
                if (JavaTemplatesStereoTypes.hasBeanStereotype(superclass, aPackage)) {
                    superClassIsBean = true;
                    superClass = superclass.getPackageName() + "." +
                                 getBeanInterfaceName(aPackage, superclass);
                    break;
                }
                superClass = superclass.getQualifiedName();
            }
        }

        if (superClass == null) {

            // try to find a super class by tag-value
            superClass =
                    getJavaTemplatesTagValues().getSimpleBeanInterfaceSuperClassTagValue(
                            input, aPackage, model);
        }

        boolean serializableFound = addInterfaces(input, output, superClass);

        if (serializableFound || superClassIsBean) {
            addInterface(output, Serializable.class);
        }

        generateI18nBlockAndConstants(aPackage, input, output);

        for (ObjectModelOperation operation : getPublicOperations(outputClass)) {
            cloneOperation(operation, output, true);
        }

        for (ObjectModelOperation operation : getPublicOperations(input)) {
            cloneOperation(operation, output, true);
        }

        return output;
    }

    protected ObjectModelClass generateBeanClass(ObjectModelClass input,
                                                 String className,
                                                 String abstractClassName) {

        ObjectModelClass output;

        if (input.isAbstract()) {
            output = createAbstractClass(className, input.getPackageName());
        } else {
            output = createClass(className, input.getPackageName());
        }

        setSuperClass(output, abstractClassName);

        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        addSerializable(input, output, true);

        return output;
    }

    protected ObjectModelClass generateAbstractBeanClass(ObjectModelPackage aPackage,
                                                         ObjectModelClass input,
                                                         String className,
                                                         String interfaceName) {

        boolean generateInterface = interfaceName != null;

        String superClass = null;

        // test if a super class has bean stereotype
        boolean superClassIsBean = false;
        Collection<ObjectModelClass> superclasses = input.getSuperclasses();
        if (CollectionUtils.isNotEmpty(superclasses)) {
            for (ObjectModelClass superclass : superclasses) {
                if (JavaTemplatesStereoTypes.hasBeanStereotype(superclass, aPackage)) {
                    superClassIsBean = true;
                    superClass = superclass.getPackageName() + "." + getBeanClassName(aPackage, superclass);
                    break;
                } else {
                    superClass = superclass.getQualifiedName();
                }
            }
        }

        if (!superClassIsBean) {

            // try to find a super class by tag-value
            superClass =
                    getJavaTemplatesTagValues().getSimpleBeanSuperClassTagValue(
                            input, aPackage, model);
            if (superClass != null) {

                // will act as if super class is a bean
                superClassIsBean = true;
            }
        }

        ObjectModelClass output;

        output = createAbstractClass(className, input.getPackageName());

        if (superClass != null) {
            setSuperClass(output, superClass);
        }
        if (log.isDebugEnabled()) {
            log.debug("will generate " + output.getQualifiedName());
        }

        boolean serializableFound;

        if (generateInterface) {

            addInterface(output, interfaceName);

            serializableFound = true;
        } else {

            serializableFound = addInterfaces(input, output, null);

            generateI18nBlockAndConstants(aPackage, input, output);
        }

        addSerializable(input, output, serializableFound || superClassIsBean);

        // Get available properties
        List<ObjectModelAttribute> properties = getProperties(input);

        boolean usePCS = getJavaTemplatesTagValues().isGeneratePropertyChangeSupport(input, aPackage, model);

        boolean generateBooleanGetMethods = getEugeneTagValues().isGenerateBooleanGetMethods(input, aPackage, model);
        boolean generateNotEmptyCollections = getJavaTemplatesTagValues().isGenerateNotEmptyCollections(input, aPackage, model) ;

        // Add properties field + javabean methods
        for (ObjectModelAttribute attr : properties) {

            createProperty(output,
                           attr,
                           usePCS,
                           generateBooleanGetMethods,
                           generateNotEmptyCollections);
        }

        if (!superClassIsBean) {
            addDefaultMethodForNoneBeanSuperClass(output, usePCS, properties);
        }
        return output;
    }

    protected void generateBeanDefaults(ObjectModelPackage aPackage,
                                        ObjectModelClass aClass,
                                        String defaultClassName) {

        boolean generateDefault = canGenerateBeanDefaults(aPackage, aClass);

        if (!generateDefault) {
            return;
        }

        String packageName = aClass.getPackageName();
        String typeName = getBeanInterfaceName(aPackage, aClass);
        String typeBeanName = getBeanClassName(aPackage, aClass);

        String abstractoutclassName = "Abstract" + defaultClassName;

        if (!isInClassPath(packageName, defaultClassName)) {

            // generate defaults class

            ObjectModelClass output = createClass(defaultClassName, packageName);
            setSuperClass(output, packageName + "." + abstractoutclassName);
            if (log.isDebugEnabled()) {
                log.debug("will generate " + output.getQualifiedName());
            }
        }


        if (!isInClassPath(packageName, abstractoutclassName)) {

            // generate abstract defaults class

            // try to find a super class by tag-value
            String superClassName =
                    getJavaTemplatesTagValues().getSimpleBeanDefaultsSuperClassTagValue(
                            aClass, aPackage, model);

            ObjectModelClass output = createAbstractClass(abstractoutclassName, packageName);
            if (StringUtils.isNotBlank(superClassName)) {
                setSuperClass(output, superClassName);
            }

            if (log.isDebugEnabled()) {
                log.debug("will generate " + output.getQualifiedName());
            }

            ObjectModelOperation operation = addOperation(
                    output,
                    "typeOf" + typeName,
                    "<BeanType extends " + typeName + "> Class<BeanType>",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            setOperationBody(operation, ""
    +"\n"
+"        return (Class<BeanType>) "+typeBeanName+".class;\n"
+"    "
            );
            operation = addOperation(
                    output,
                    "new" + typeName,
                    typeName,
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            setOperationBody(operation, ""
    +"\n"
+"        return new "+typeBeanName+"();\n"
+"    "
            );

            addImport(output, Binder.class);
            addImport(output, BinderFactory.class);
            operation = addOperation(
                    output,
                    "new" + typeName,
                    "<BeanType extends " + typeName + "> BeanType",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "BeanType", "source");
            setOperationBody(operation, ""
    +"\n"
+"        Class<BeanType> sourceType = typeOf"+typeName+"();\n"
+"        Binder<BeanType,BeanType> binder = BinderFactory.newBinder(sourceType);\n"
+"        BeanType result = new"+typeName+"(source, binder);\n"
+"        return result;\n"
+"    "
            );

            operation = addOperation(
                    output,
                    "new" + typeName,
                    "<BeanType extends " + typeName + "> BeanType",
                    ObjectModelJavaModifier.STATIC,
                    ObjectModelJavaModifier.PUBLIC
            );
            addParameter(operation, "BeanType", "source");
            addParameter(operation, "Binder<BeanType, BeanType>", "binder");
            setOperationBody(operation, ""
    +"\n"
+"        BeanType result = (BeanType) new"+typeName+"();\n"
+"        binder.copy(source, result);\n"
+"        return result;\n"
+"    "
            );
        }
    }

    protected Collection<ObjectModelOperation> getPublicOperations(ObjectModelClass clazz) {

        Collection<ObjectModelOperation> result = new ArrayList<ObjectModelOperation>();
        for (ObjectModelOperation operation : clazz.getOperations()) {

            ObjectModelJavaModifier visibility =
                    ObjectModelJavaModifier.fromVisibility(
                            operation.getVisibility());
            if (ObjectModelJavaModifier.PUBLIC == visibility) {
                result.add(operation);
            }
        }
        return result;
    }

    protected String getBeanInterfaceName(ObjectModelPackage aPackage, ObjectModelClass input) {
        String interfaceNamePrefix = getJavaTemplatesTagValues().getSimpleBeanInterfaceNamePrefixTagValue(input, aPackage, model);
        String interfaceNameSuffix = getJavaTemplatesTagValues().getSimpleBeanInterfaceNameSuffixTagValue(input, aPackage, model);

        return generateName(
                interfaceNamePrefix,
                input.getName(),
                interfaceNameSuffix
        );
    }

    protected String getBeanClassName(ObjectModelPackage aPackage, ObjectModelClass input) {
        String classNamePrefix = getJavaTemplatesTagValues().getSimpleBeanClassNamePrefixTagValue(input, aPackage, model);
        String classNameSuffix = getJavaTemplatesTagValues().getSimpleBeanClassNameSuffixTagValue(input, aPackage, model);

        return generateName(
                classNamePrefix,
                input.getName(),
                classNameSuffix
        );
    }

    protected String getAbstractBeanClassName(ObjectModelPackage aPackage, ObjectModelClass input) {
        String classNamePrefix = getJavaTemplatesTagValues().getSimpleBeanClassNamePrefixTagValue(input, aPackage, model);
        String classNameSuffix = getJavaTemplatesTagValues().getSimpleBeanClassNameSuffixTagValue(input, aPackage, model);

        return generateName(
                classNamePrefix,
                "Abstract" + input.getName(),
                classNameSuffix
        );
    }

    protected String getBeanDefaultsClassName(ObjectModelPackage aPackage, ObjectModelClass input) {
        String classNamePrefix = getJavaTemplatesTagValues().getSimpleBeanDefaultsClassNamePrefixTagValue(input, aPackage, model);
        String classNameSuffix = getJavaTemplatesTagValues().getSimpleBeanDefaultsClassNameSuffixTagValue(input, aPackage, model);
        return generateName(
                classNamePrefix,
                input.getName(),
                classNameSuffix
        );
    }

    protected String generateName(String prefix,
                                  String name,
                                  String suffix) {
        StringBuilder sb = new StringBuilder();
        if (StringUtils.isNotEmpty(prefix)) {
            sb.append(prefix);
        }
        sb.append(name);
        if (StringUtils.isNotEmpty(suffix)) {
            sb.append(suffix);
        }
        return sb.toString();
    }
}
