/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.javassist;

import java.lang.reflect.Modifier;

import javassist.CtClass;
import javassist.CtField;
import javassist.NotFoundException;
import javassist.bytecode.SignatureAttribute.ClassSignature;
import javassist.bytecode.SignatureAttribute.ObjectType;

import org.jboss.reflect.plugins.AnnotationHelper;
import org.jboss.reflect.spi.AnnotationValue;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.FieldInfo;
import org.jboss.reflect.spi.ModifierInfo;
import org.jboss.reflect.spi.MutableFieldInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.util.JBossStringBuilder;

/**
 * FieldInfo that relies on Javassist to answer reflective queries and to access
 * the represented field.
 * 
 *  This class also relies on Javassist to perform the instrumentation operations defined in
 *  {@code MutableFieldInfo}.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 104251 $
 * @see MutableFieldInfo
 */
// TODO shouldn  JavassistFieldInfo extend FieldInfoImpl just like
// ReflectFieldInfoImpl?
public class JavassistFieldInfo extends JavassistAnnotatedInfo implements MutableFieldInfo
{
   /** The serialVersionUID */
   private static final long serialVersionUID = -104555531831318930L;

   /** The field */
   private final CtField ctField;
   
   /** The field implementation */
   private transient volatile JavassistField field;

   /** The type */
   private transient volatile TypeInfo fieldType;
   
   /** The type info */
   protected final JavassistTypeInfo typeInfo;

   /**
    * Create a new JavassistFieldInfo.
    * 
    * @param annotationHelper the annotation helper
    * @param typeInfo the type info
    * @param ctField the field
    */
   public JavassistFieldInfo(AnnotationHelper annotationHelper, JavassistTypeInfo typeInfo, CtField ctField)
   {
      super(annotationHelper);
      this.typeInfo = typeInfo;
      this.ctField = ctField;
   }

   public String getName()
   {
      return ctField.getName();
   }

   public int getModifiers()
   {
      return ctField.getModifiers();
   }

   public boolean isPublic()
   {
      return Modifier.isPublic(getModifiers());
   }

   public boolean isStatic()
   {
      return Modifier.isStatic(getModifiers());
   }

   public boolean isVolatile()
   {
      return Modifier.isVolatile(getModifiers());
   }

   public ClassInfo getDeclaringClass()
   {
      return typeInfo;
   }

   public TypeInfo getType()
   {
      if (fieldType != null)
         return fieldType;
      try
      {
         ObjectType type = JavassistHelper.getFieldSignature(ctField);
         if (type != null)
         {
            ClassSignature sig = JavassistHelper.getClassSignature(ctField.getDeclaringClass());
            fieldType = typeInfo.getFactory().getTypeInfo(typeInfo.getClassLoaderInternal(), type, JavassistTypeVariableSpy.createForField(sig));
         }
         else
         {
            CtClass clazz = ctField.getType();
            fieldType = typeInfo.getFactory().getTypeInfo(clazz);
         }
         return fieldType;
      }
      catch (NotFoundException e)
      {
         throw JavassistTypeInfoFactoryImpl.raiseFieldNotFound(getName(), e);
      }
   }

   public Object get(Object target) throws Throwable
   {
      if (field == null)
         field = JavassistReflectionFactory.INSTANCE.createField(this);
      
      JavassistAccessController.checkAccess(this);
      
      return field.get(target);
   }

   public Object set(Object target, Object value) throws Throwable
   {
      if (field == null)
         field = JavassistReflectionFactory.INSTANCE.createField(this);

      JavassistAccessController.checkAccess(this);

      field.set(target, value);
      return null;
   }

   @Override
   protected int getHashCode()
   {
      return getName().hashCode();
   }

   @Override
   public boolean equals(Object obj)
   {
      if (obj == this)
         return true;
      if (obj == null || obj instanceof FieldInfo == false)
         return false;
      
      FieldInfo other = (FieldInfo) obj;
      if (getName().equals(other.getName()) == false)
         return false;
      return getDeclaringClass().equals(other.getDeclaringClass());
   }

   @Override
   public void toShortString(JBossStringBuilder buffer)
   {
      buffer.append(getName());
   }

   @Override
   protected void toString(JBossStringBuilder buffer)
   {
      buffer.append("name=").append(getName());
      super.toString(buffer);
   }
   
   public AnnotationValue[] getAnnotations()
   {
      return getAnnotations(ctField);
   }
   
   public CtField getCtField()
   {
      return ctField;
   }

   public void setModifier(ModifierInfo modifier)
   {
      ctField.setModifiers(modifier.getModifiers());
      typeInfo.clearFieldCache();
   }

   public void setName(String name)
   {
      ctField.setName(name);
      typeInfo.clearFieldCache();
   }

   public void setType(ClassInfo type)
   {
      ctField.setType(JavassistUtil.toCtClass(type));
      typeInfo.clearFieldCache();
   }
   
   public void setType(String type)
   {
      ctField.setType(JavassistUtil.toCtClass(typeInfo.getCtClass().getClassPool(), type));
      typeInfo.clearFieldCache();
   }   
}
