/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.javassist;

import java.io.Serializable;

import javassist.CtClass;
import javassist.Modifier;

import org.jboss.reflect.spi.AnnotationValue;
import org.jboss.reflect.spi.ArrayInfo;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.InterfaceInfo;
import org.jboss.reflect.spi.PrimitiveInfo;
import org.jboss.reflect.spi.TypeInfo;

/**
 * Javassist array info
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 103926 $
 */
public class JavassistArrayInfoImpl extends JavassistTypeInfo implements ArrayInfo
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 9195834689976459024L;
   
   /** Unknown annotations */
   static final AnnotationValue[] UNKNOWN_ANNOTATIONS = new AnnotationValue[0];
   
   private volatile int modifiers;

   /** The component type */
   protected final TypeInfo componentType;
   
   /** The hash code */
   protected final int hash;

   private static String getName(TypeInfo componentType)
   {
      StringBuilder builder = new StringBuilder();
      builder.append("[");
      TypeInfo temp = componentType;
      while (temp.isArray())
      {
         builder.append("[");
         temp = ((JavassistArrayInfoImpl) temp).componentType;
      }
      if (temp.isPrimitive())
      {
         // Use the signature encoded name for the primitive element type
         String encodedName = PrimitiveInfo.getPrimativeArrayType(temp.getName());
         builder.append(encodedName);
      }
      else
      {
         builder.append("L").append(temp.getName()).append(";");
      }
      return builder.toString();
   }
   
   /**
    * Create a new JavassistArrayInfoImpl.
    * 
    * @param factory the factory
    * @param ctClass the ctClass
    * @param clazz the class
    * @param componentType the component type
    */
   JavassistArrayInfoImpl(JavassistTypeInfoFactoryImpl factory, CtClass ctClass, Class<? extends Object> clazz, TypeInfo componentType)
   {
      super(factory, getName(componentType), ctClass, clazz);
      this.componentType = componentType;
      hash = calculateHash();
   }

   @Override
   public TypeInfo getComponentType()
   {
      return componentType;
   }
   
   @Override
   public AnnotationValue getAnnotation(String name)
   {
      return null;
   }

   @Override
   public AnnotationValue[] getAnnotations()
   {
      return UNKNOWN_ANNOTATIONS;
   }

   @Override
   public boolean isAnnotationPresent(String name)
   {
      return false;
   }

   @Override
   public boolean equals(Object o)
   {
      if (this == o) return true;
      if (!(o instanceof ArrayInfo)) return false;
      if (!super.equals(o)) return false;

      final ArrayInfo arrayInfo = (ArrayInfo) o;

      if (!componentType.equals(arrayInfo.getComponentType())) return false;

      return true;
   }

   @Override
   public int hashCode() { return hash; }

   /**
    * Calculate the hash code
    */
   protected int calculateHash()
   {
      int result = super.hashCode();
      result = 29 * result + componentType.hashCode();
      return result;
   }

   @Override
   public InterfaceInfo[] getGenericInterfaces()
   {
      return getInterfaces();
   }

   @Override
   public InterfaceInfo[] getInterfaces()
   {
      //Arrays will always implement Cloneable and Serializable (10.7 in Java Language specification)
      InterfaceInfo cloneable = (InterfaceInfo)getFactory().get(Cloneable.class);
      InterfaceInfo serializable = (InterfaceInfo)getFactory().get(Serializable.class);
      return new InterfaceInfo[] {cloneable, serializable};
   }

   @Override
   public int getModifiers()
   {
      //Array modifiers will be public/private/protected as the component type and are always final and abstract
      if (modifiers == 0)
      {
         TypeInfo info = componentType;
         if (info.isArray())
            modifiers = ((ClassInfo)info).getModifiers();
         else
         {
            int mod = 0;
            if (componentType instanceof ClassInfo)
            {
               int typeModifiers = ((ClassInfo)componentType).getModifiers();
   
               if (Modifier.isPublic(typeModifiers))
                  mod = Modifier.PUBLIC;
               else if (Modifier.isProtected(typeModifiers))
                  mod = Modifier.PROTECTED;
               else if (Modifier.isPrivate(typeModifiers))
                  mod = Modifier.PRIVATE;
            }
            else if (componentType instanceof PrimitiveInfo)
            {
               mod = Modifier.PUBLIC;
            }
            mod |= Modifier.FINAL;
            mod |= Modifier.ABSTRACT;
            modifiers = mod;
         }
      }
      return modifiers;
   }
}
